"""Helper functions for processing and manipulating response content.

This module contains utility functions designed to assist in the handling and
modification of response content generated by the language model. These functions
are crucial for maintaining the integrity and usability of the responses, ensuring
that they conform to the expected format and are free from unnecessary tags or
elements.
"""

import json
import re


def remove_think_tags(response_content: str) -> str:
    """
    Remove the content enclosed within <think> and </think> tags from the response content.

    Args:
        response_content (str): The original response content potentially containing <think> tags.

    Returns:
        str: The response content with <think> and </think> tags and their content removed.
    """
    if "<think>" in response_content:
        start_think_index = response_content.find("<think>")
        end_think_index = response_content.rfind("</think>")
        if start_think_index != -1 and end_think_index != -1:
            response_content = (
                response_content[:start_think_index]
                + response_content[end_think_index + len("</think>") :].strip()
            )
    return response_content


def clean_plain_text_response(response_content: str) -> str:
    """
    Clean plain text responses like reflection and planning by removing code blocks and
    standalone JSON.

    Args:
        response_content (str): The original plain text response potentially containing
                               code blocks or JSON objects.

    Returns:
        str: The cleaned response with code blocks and standalone JSON removed.
    """
    # Check if the entire content is a JSON object
    if response_content.strip().startswith("{") and response_content.strip().endswith("}"):
        try:
            json.loads(response_content.strip())
            # If it parses as valid JSON, remove it completely
            return ""
        except json.JSONDecodeError:
            # Not valid JSON, keep the content
            pass

    # Remove code blocks
    lines = response_content.split("\n")
    cleaned_lines = []
    in_code_block = False
    code_block_start_index = -1

    for i, line in enumerate(lines):
        if line.strip().startswith("```"):
            if not in_code_block:
                in_code_block = True
                code_block_start_index = i
            else:
                in_code_block = False
                # Add an empty line if there's content before and after the code block
                if code_block_start_index > 0 and i < len(lines) - 1:
                    cleaned_lines.append("")
            continue
        if not in_code_block:
            cleaned_lines.append(line.rstrip())

    cleaned_content = "\n".join(cleaned_lines)

    # Remove JSON objects embedded in the text
    pattern = r'\{(?:[^{}]|"[^"]*")*\}'
    cleaned_content = re.sub(pattern, "", cleaned_content)

    # Clean up any double spaces and preserve line breaks
    cleaned_content = re.sub(r" +", " ", cleaned_content)

    # Remove trailing spaces at the end of each line and leading spaces at the beginning
    # of each line
    cleaned_content = "\n".join(line.strip() for line in cleaned_content.split("\n"))

    return cleaned_content.strip()


def clean_json_response(response_content: str) -> str:
    """
    Clean JSON responses by extracting the JSON content from various formats.

    Args:
        response_content (str): The original JSON response potentially containing
                               code blocks, markdown formatting, or other text.

    Returns:
        str: The extracted JSON content as a string.
    """
    response_content = remove_think_tags(response_content)

    # Check for JSON content between the text "JSON response content: ```json" and "```"
    json_response_marker = "```json"
    if json_response_marker in response_content:
        start_index = response_content.find(json_response_marker) + len(json_response_marker)
        response_content = response_content[start_index:]

        end_index = response_content.find("```")
        if end_index != -1:
            json_content = response_content[:end_index].strip()
            # Validate if this is valid JSON before returning
            try:
                json.loads(json_content)
                return json_content
            except json.JSONDecodeError:
                # If not valid JSON, continue with other extraction methods
                pass

    # Check if the entire content is already valid JSON
    try:
        if response_content.strip().startswith("{") and response_content.strip().endswith("}"):
            json.loads(response_content.strip())
            return response_content.strip()
    except json.JSONDecodeError:
        pass

    # Check for JSON code block format with triple backticks
    json_block_patterns = ["```json\n", "```\n"]

    for pattern in json_block_patterns:
        if pattern in response_content:
            start_index = response_content.find(pattern)
            content_after_marker = response_content[start_index + len(pattern) :]
            end_index = content_after_marker.find("```")

            if end_index != -1:
                extracted_content = content_after_marker[:end_index].strip()
                try:
                    json.loads(extracted_content)
                    return extracted_content
                except json.JSONDecodeError:
                    # Continue to next pattern if this isn't valid JSON
                    pass

    # If no specific markers found, try to extract JSON object directly
    # Look for the first { and the last }
    first_brace = response_content.find("{")
    last_brace = response_content.rfind("}")

    if first_brace != -1 and last_brace != -1 and last_brace > first_brace:
        extracted_json = response_content[first_brace : last_brace + 1].strip()
        try:
            json.loads(extracted_json)
            return extracted_json
        except json.JSONDecodeError:
            # If not valid JSON, return the best attempt
            pass

    # If we couldn't extract valid JSON, return the original content
    return response_content.strip()
