"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
test('create HTTPS redirect', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'test', { env: { region: 'us-east-1' } });
    // WHEN
    new lib_1.HttpsRedirect(stack, 'Redirect', {
        recordNames: ['foo.example.com', 'baz.example.com'],
        targetDomain: 'bar.example.com',
        zone: aws_route53_1.HostedZone.fromHostedZoneAttributes(stack, 'HostedZone', {
            hostedZoneId: 'ID',
            zoneName: 'example.com',
        }),
    });
    // THEN
    expect(stack).toHaveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            RedirectAllRequestsTo: {
                HostName: 'bar.example.com',
                Protocol: 'https',
            },
        },
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: ['foo.example.com', 'baz.example.com'],
            DefaultRootObject: '',
        },
    });
    expect(stack).toHaveResource('AWS::Route53::RecordSet', {
        Type: 'A',
        Name: 'foo.example.com.',
        HostedZoneId: 'ID',
    });
    expect(stack).toHaveResource('AWS::Route53::RecordSet', {
        Type: 'AAAA',
        Name: 'foo.example.com.',
        HostedZoneId: 'ID',
    });
    expect(stack).toHaveResource('AWS::Route53::RecordSet', {
        Type: 'A',
        Name: 'baz.example.com.',
        HostedZoneId: 'ID',
    });
    expect(stack).toHaveResource('AWS::Route53::RecordSet', {
        Type: 'AAAA',
        Name: 'baz.example.com.',
        HostedZoneId: 'ID',
    });
});
test('create HTTPS redirect for apex', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'test', { env: { region: 'us-east-1' } });
    // WHEN
    new lib_1.HttpsRedirect(stack, 'Redirect', {
        targetDomain: 'bar.example.com',
        zone: aws_route53_1.HostedZone.fromHostedZoneAttributes(stack, 'HostedZone', {
            hostedZoneId: 'ID',
            zoneName: 'example.com',
        }),
    });
    // THEN
    expect(stack).toHaveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            RedirectAllRequestsTo: {
                HostName: 'bar.example.com',
                Protocol: 'https',
            },
        },
    });
    expect(stack).toHaveResource('AWS::Route53::RecordSet', {
        Type: 'A',
        Name: 'example.com.',
    });
    expect(stack).toHaveResource('AWS::Route53::RecordSet', {
        Type: 'AAAA',
        Name: 'example.com.',
    });
});
test('create HTTPS redirect with existing cert', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'test', { env: { region: 'us-east-1' } });
    // WHEN
    new lib_1.HttpsRedirect(stack, 'Redirect', {
        recordNames: ['foo.example.com'],
        certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Certificate', 'arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d'),
        targetDomain: 'bar.example.com',
        zone: aws_route53_1.HostedZone.fromHostedZoneAttributes(stack, 'HostedZone', {
            hostedZoneId: 'ID',
            zoneName: 'example.com',
        }),
    });
    // THEN
    expect(stack).toHaveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            RedirectAllRequestsTo: {
                HostName: 'bar.example.com',
                Protocol: 'https',
            },
        },
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            ViewerCertificate: {
                AcmCertificateArn: 'arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d',
            },
        },
    });
});
test('throws when certificate in region other than us-east-1 is supplied', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'test', { env: { region: 'us-east-1' } });
    const certificate = aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Certificate', 'arn:aws:acm:us-east-2:123456789012:certificate/11-3336f1-44483d-adc7-9cd375c5169d');
    // WHEN / THEN
    expect(() => {
        new lib_1.HttpsRedirect(stack, 'Redirect', {
            recordNames: ['foo.example.com'],
            certificate,
            targetDomain: 'bar.example.com',
            zone: aws_route53_1.HostedZone.fromHostedZoneAttributes(stack, 'HostedZone', {
                hostedZoneId: 'ID',
                zoneName: 'example.com',
            }),
        });
    }).toThrow(/The certificate must be in the us-east-1 region and the certificate you provided is in us-east-2./);
});
//# sourceMappingURL=data:application/json;base64,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