"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpsRedirect = void 0;
const crypto = require("crypto");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
/**
 * Allows creating a domainA -> domainB redirect using CloudFront and S3.
 * You can specify multiple domains to be redirected.
 */
class HttpsRedirect extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const domainNames = (_a = props.recordNames) !== null && _a !== void 0 ? _a : [props.zone.zoneName];
        if (props.certificate) {
            const certificateRegion = core_1.Stack.of(this).parseArn(props.certificate.certificateArn).region;
            if (!core_1.Token.isUnresolved(certificateRegion) && certificateRegion !== 'us-east-1') {
                throw new Error(`The certificate must be in the us-east-1 region and the certificate you provided is in ${certificateRegion}.`);
            }
        }
        const redirectCertArn = props.certificate ? props.certificate.certificateArn : new aws_certificatemanager_1.DnsValidatedCertificate(this, 'RedirectCertificate', {
            domainName: domainNames[0],
            subjectAlternativeNames: domainNames,
            hostedZone: props.zone,
            region: 'us-east-1',
        }).certificateArn;
        const redirectBucket = new aws_s3_1.Bucket(this, 'RedirectBucket', {
            websiteRedirect: {
                hostName: props.targetDomain,
                protocol: aws_s3_1.RedirectProtocol.HTTPS,
            },
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        const redirectDist = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'RedirectDistribution', {
            defaultRootObject: '',
            originConfigs: [{
                    behaviors: [{ isDefaultBehavior: true }],
                    customOriginSource: {
                        domainName: redirectBucket.bucketWebsiteDomainName,
                        originProtocolPolicy: aws_cloudfront_1.OriginProtocolPolicy.HTTP_ONLY,
                    },
                }],
            aliasConfiguration: {
                acmCertRef: redirectCertArn,
                names: domainNames,
            },
            comment: `Redirect to ${props.targetDomain} from ${domainNames.join(', ')}`,
            priceClass: aws_cloudfront_1.PriceClass.PRICE_CLASS_ALL,
            viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        });
        domainNames.forEach((domainName) => {
            const hash = crypto.createHash('md5').update(domainName).digest('hex').substr(0, 6);
            const aliasProps = {
                recordName: domainName,
                zone: props.zone,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.CloudFrontTarget(redirectDist)),
            };
            new aws_route53_1.ARecord(this, `RedirectAliasRecord${hash}`, aliasProps);
            new aws_route53_1.AaaaRecord(this, `RedirectAliasRecordSix${hash}`, aliasProps);
        });
    }
}
exports.HttpsRedirect = HttpsRedirect;
//# sourceMappingURL=data:application/json;base64,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