"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RollbackEvent = exports.EcsDeploymentGroup = void 0;
const path = require("path");
const aws_codedeploy_1 = require("@aws-cdk/aws-codedeploy");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
class EcsDeploymentGroup extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const { applicationName, deploymentGroupName, deploymentConfig, ecsServices, targetGroupNames, prodTrafficListener, testTrafficListener, terminationWaitTimeInMinutes = 60, autoRollbackOnEvents, } = props;
        if (terminationWaitTimeInMinutes > 2880) {
            throw new Error('Invalid TerminationWaitTimeInMinutes: The maximum setting is 2880 minutes (2 days).');
        }
        const codeDeployEcsRole = new aws_iam_1.Role(this, 'EcsCodeDeployRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('codedeploy.amazonaws.com'),
            managedPolicies: [aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSCodeDeployRoleForECS')],
        });
        this.application = new aws_codedeploy_1.EcsApplication(this, 'EcsApplication', {
            applicationName,
        });
        const serviceToken = core_1.CustomResourceProvider.getOrCreate(this, 'Custom::EcsDeploymentGroup', {
            codeDirectory: path.join(__dirname, 'lambdas', 'ecs-deployment-group'),
            runtime: core_1.CustomResourceProviderRuntime.NODEJS_12,
            policyStatements: [
                {
                    Effect: aws_iam_1.Effect.ALLOW,
                    Action: ['codeDeploy:CreateDeploymentGroup', 'codeDeploy:UpdateDeploymentGroup', 'codeDeploy:DeleteDeploymentGroup'],
                    Resource: '*',
                },
                {
                    Effect: aws_iam_1.Effect.ALLOW,
                    Action: ['iam:PassRole'],
                    Resource: codeDeployEcsRole.roleArn,
                },
            ],
        });
        const ecsDeploymentGroup = new core_1.CustomResource(this, 'CustomResource', {
            serviceToken,
            resourceType: 'Custom::EcsDeploymentGroup',
            properties: {
                ApplicationName: this.application.applicationName,
                DeploymentGroupName: deploymentGroupName,
                ServiceRoleArn: codeDeployEcsRole.roleArn,
                TargetGroupNames: targetGroupNames,
                EcsServices: ecsServices.map((service) => ({
                    ClusterName: service.clusterName,
                    ServiceName: service.serviceName,
                })),
                ProdTrafficListenerArn: prodTrafficListener.listenerArn,
                TestTrafficListenerArn: testTrafficListener.listenerArn,
                TerminationWaitTimeInMinutes: terminationWaitTimeInMinutes,
                AutoRollbackOnEvents: autoRollbackOnEvents,
            },
        });
        this.deploymentGroupName = ecsDeploymentGroup.ref;
        this.deploymentGroupArn = this.arnForDeploymentGroup(this.application.applicationName, this.deploymentGroupName);
        this.deploymentConfig = deploymentConfig || aws_codedeploy_1.EcsDeploymentConfig.ALL_AT_ONCE;
    }
    arnForDeploymentGroup(applicationName, deploymentGroupName) {
        return `arn:${core_1.Aws.PARTITION}:codedeploy:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:deploymentgroup:${applicationName}/${deploymentGroupName}`;
    }
}
exports.EcsDeploymentGroup = EcsDeploymentGroup;
var RollbackEvent;
(function (RollbackEvent) {
    RollbackEvent["DEPLOYMENT_FAILURE"] = "DEPLOYMENT_FAILURE";
    RollbackEvent["DEPLOYMENT_STOP_ON_ALARM"] = "DEPLOYMENT_STOP_ON_ALARM";
    RollbackEvent["DEPLOYMENT_STOP_ON_REQUEST"] = "DEPLOYMENT_STOP_ON_REQUEST";
})(RollbackEvent = exports.RollbackEvent || (exports.RollbackEvent = {}));
//# sourceMappingURL=data:application/json;base64,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