"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualNodeListener = void 0;
const cdk = require("@aws-cdk/core");
const utils_1 = require("./private/utils");
const shared_interfaces_1 = require("./shared-interfaces");
/**
 * (experimental) Defines listener for a VirtualNode.
 *
 * @experimental
 */
class VirtualNodeListener {
    /**
     * (experimental) Returns an HTTP Listener for a VirtualNode.
     *
     * @experimental
     */
    static http(props = {}) {
        return new VirtualNodeListenerImpl(shared_interfaces_1.Protocol.HTTP, props.healthCheck, props.timeout, props.port);
    }
    /**
     * (experimental) Returns an HTTP2 Listener for a VirtualNode.
     *
     * @experimental
     */
    static http2(props = {}) {
        return new VirtualNodeListenerImpl(shared_interfaces_1.Protocol.HTTP2, props.healthCheck, props.timeout, props.port);
    }
    /**
     * (experimental) Returns an GRPC Listener for a VirtualNode.
     *
     * @experimental
     */
    static grpc(props = {}) {
        return new VirtualNodeListenerImpl(shared_interfaces_1.Protocol.GRPC, props.healthCheck, props.timeout, props.port);
    }
    /**
     * (experimental) Returns an TCP Listener for a VirtualNode.
     *
     * @experimental
     */
    static tcp(props = {}) {
        return new VirtualNodeListenerImpl(shared_interfaces_1.Protocol.TCP, props.healthCheck, props.timeout, props.port);
    }
}
exports.VirtualNodeListener = VirtualNodeListener;
class VirtualNodeListenerImpl extends VirtualNodeListener {
    constructor(protocol, healthCheck, timeout, port = 8080) {
        super();
        this.protocol = protocol;
        this.healthCheck = healthCheck;
        this.timeout = timeout;
        this.port = port;
    }
    bind(_scope) {
        return {
            listener: {
                portMapping: {
                    port: this.port,
                    protocol: this.protocol,
                },
                healthCheck: this.healthCheck ? this.renderHealthCheck(this.healthCheck) : undefined,
                timeout: this.timeout ? this.renderTimeout(this.timeout) : undefined,
            },
        };
    }
    renderHealthCheck(hc) {
        if (hc === undefined) {
            return undefined;
        }
        if (hc.protocol === shared_interfaces_1.Protocol.TCP && hc.path) {
            throw new Error('The path property cannot be set with Protocol.TCP');
        }
        if (hc.protocol === shared_interfaces_1.Protocol.GRPC && hc.path) {
            throw new Error('The path property cannot be set with Protocol.GRPC');
        }
        const healthCheck = {
            healthyThreshold: hc.healthyThreshold || 2,
            intervalMillis: (hc.interval || cdk.Duration.seconds(5)).toMilliseconds(),
            path: hc.path || (hc.protocol === shared_interfaces_1.Protocol.HTTP ? '/' : undefined),
            port: hc.port || this.port,
            protocol: hc.protocol || this.protocol,
            timeoutMillis: (hc.timeout || cdk.Duration.seconds(2)).toMilliseconds(),
            unhealthyThreshold: hc.unhealthyThreshold || 2,
        };
        utils_1.validateHealthChecks(healthCheck);
        return healthCheck;
    }
    renderTimeout(timeout) {
        return ({
            [this.protocol]: {
                idle: (timeout === null || timeout === void 0 ? void 0 : timeout.idle) !== undefined ? {
                    unit: 'ms',
                    value: timeout === null || timeout === void 0 ? void 0 : timeout.idle.toMilliseconds(),
                } : undefined,
                perRequest: (timeout === null || timeout === void 0 ? void 0 : timeout.perRequest) !== undefined ? {
                    unit: 'ms',
                    value: timeout === null || timeout === void 0 ? void 0 : timeout.perRequest.toMilliseconds(),
                } : undefined,
            },
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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