"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Pipe = exports.DesiredState = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_pipes_1 = require("aws-cdk-lib/aws-pipes");
const logs_1 = require("./logs");
const source_1 = require("./source");
/**
 * The state the pipe should be in.
 */
var DesiredState;
(function (DesiredState) {
    /**
     * The pipe should be running.
     */
    DesiredState["RUNNING"] = "RUNNING";
    /**
     * The pipe should be stopped.
     * */
    DesiredState["STOPPED"] = "STOPPED";
})(DesiredState || (exports.DesiredState = DesiredState = {}));
class PipeBase extends aws_cdk_lib_1.Resource {
}
/**
 * An imported pipe.
 */
class ImportedPipe extends PipeBase {
    constructor(scope, id, pipeName) {
        super(scope, id);
        this.pipeName = pipeName;
        this.pipeArn = aws_cdk_lib_1.Stack.of(this).formatArn({
            service: 'pipes',
            partition: 'aws',
            resource: 'pipe',
            resourceName: this.pipeName,
        });
        this.pipeRole = aws_iam_1.Role.fromRoleName(this, 'Role', this.pipeName);
    }
}
/**
 * Amazon EventBridge Pipes connects sources to targets.
 *
 * Pipes are intended for point-to-point integrations between supported sources and targets,
 * with support for advanced transformations and enrichment.
 *
 * @see https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html
 */
class Pipe extends PipeBase {
    /**
     * Creates a pipe from the name of a pipe.
     */
    static fromPipeName(scope, id, pipeName) {
        return new ImportedPipe(scope, id, pipeName);
    }
    constructor(scope, id, props) {
        super(scope, id, { physicalName: props.pipeName });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_pipes_alpha_PipeProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Pipe);
            }
            throw error;
        }
        /**
         * Role setup
         */
        this.pipeRole =
            props.role ||
                new aws_iam_1.Role(this, 'Role', {
                    assumedBy: new aws_iam_1.ServicePrincipal('pipes.amazonaws.com'),
                });
        /**
         * Source / Filter setup
         */
        const source = props.source.bind(this);
        props.source.grantRead(this.pipeRole);
        /**
         * An optional dead-letter queue stores any events that are not successfully delivered to
         * a target after all retry attempts are exhausted. The IAM role needs permission to write
         * events to the dead-letter queue, either an SQS queue or SNS topic.
         */
        if (source_1.SourceWithDeadLetterTarget.isSourceWithDeadLetterTarget(props.source)) {
            props.source.grantPush(this.pipeRole, props.source.deadLetterTarget);
        }
        // Add the filter criteria to the source parameters
        const sourceParameters = {
            ...source.sourceParameters,
            filterCriteria: props.filter,
        };
        /**
         * Enrichment setup
         */
        const enrichment = props.enrichment?.bind(this);
        props.enrichment?.grantInvoke(this.pipeRole);
        /**
         * Target setup
         */
        const target = props.target.bind(this);
        props.target.grantPush(this.pipeRole);
        /**
         * Logs setup
         */
        const initialLogConfiguration = {
            level: props.logLevel || logs_1.LogLevel.ERROR,
            includeExecutionData: props.logIncludeExecutionData || undefined,
        };
        // Iterate over all the log destinations and add them to the log configuration
        const logConfiguration = props.logDestinations?.reduce((currentLogConfiguration, destination) => {
            const logDestinationConfig = destination.bind(this);
            destination.grantPush(this.pipeRole);
            const additionalLogConfiguration = logDestinationConfig.parameters;
            return { ...currentLogConfiguration, ...additionalLogConfiguration };
        }, initialLogConfiguration);
        /**
         * Pipe resource
         */
        const resource = new aws_pipes_1.CfnPipe(this, 'Resource', {
            name: props.pipeName,
            description: props.description,
            roleArn: this.pipeRole.roleArn,
            source: props.source.sourceArn,
            sourceParameters: sourceParameters,
            enrichment: props.enrichment?.enrichmentArn,
            enrichmentParameters: enrichment?.enrichmentParameters,
            target: props.target.targetArn,
            targetParameters: target.targetParameters,
            desiredState: props.desiredState,
            logConfiguration: logConfiguration,
            tags: props.tags,
        });
        this.pipeName = resource.ref;
        this.pipeArn = resource.attrArn;
    }
}
exports.Pipe = Pipe;
_a = JSII_RTTI_SYMBOL_1;
Pipe[_a] = { fqn: "@aws-cdk/aws-pipes-alpha.Pipe", version: "2.174.1-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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