#!/usr/bin/env python3
"""
BioQL Backend Integration
=========================

Integrates Quillow QEC with the existing BioQL API at api.bioql.bio

This backend:
1. Takes BioQL quantum chemistry circuits
2. Protects them with surface code QEC
3. Executes on BioQL's quantum backends (IBM, IonQ, etc.)
4. Decodes syndromes and applies error correction
5. Returns error-mitigated results

Architecture:
------------
Quillow (QEC) → api.bioql.bio → Quantum Hardware → Quillow (Decode) → Corrected Result

BioQL API Endpoints Used:
-------------------------
- POST /auth/validate - API key validation
- POST /billing/check-limits - Check quota
- POST /billing/record-usage - Record usage with QEC overhead
- POST /quantum/execute - Execute quantum circuit
- POST /quantum/vqe - VQE calculations
- POST /quantum/molecular - Molecular simulations
"""

import requests
import numpy as np
import stim
import json
from typing import Optional, Dict, Any, Tuple
from dataclasses import dataclass
from loguru import logger
import time

from .abstract_backend import AbstractBackend, BackendResult
from ..core import SurfaceCodeSimulator, RotatedSurfaceCode
from ..decoders import PyMatchingDecoder


@dataclass
class BioQLConfig:
    """Configuration for BioQL API."""
    base_url: str = "https://api.bioql.bio"
    api_key: Optional[str] = None
    timeout: int = 300
    verify_ssl: bool = True

    @classmethod
    def from_env(cls):
        """Load from environment variables."""
        import os
        return cls(
            api_key=os.getenv("BIOQL_API_KEY"),
            base_url=os.getenv("BIOQL_API_URL", "https://api.bioql.bio")
        )


class BioQLBackend(AbstractBackend):
    """
    Backend for executing QEC-protected circuits on BioQL infrastructure.

    Connects to api.bioql.bio server.
    """

    def __init__(
        self,
        config: Optional[BioQLConfig] = None,
        qec_distance: int = 5,
        enable_qec: bool = True
    ):
        """
        Initialize BioQL backend.

        Args:
            config: BioQL configuration
            qec_distance: Surface code distance for protection
            enable_qec: Enable QEC protection
        """
        super().__init__(name="BioQL")
        self.config = config or BioQLConfig.from_env()
        self.qec_distance = qec_distance
        self.enable_qec = enable_qec

        if not self.config.api_key:
            logger.warning("No BioQL API key found. Set BIOQL_API_KEY environment variable.")

        # Initialize QEC components
        if self.enable_qec:
            self.decoder = PyMatchingDecoder()
            logger.info(f"BioQL backend initialized with QEC (d={qec_distance})")
        else:
            logger.info("BioQL backend initialized (QEC disabled)")

    def validate_api_key(self) -> bool:
        """Validate API key with BioQL server."""
        try:
            response = requests.post(
                f"{self.config.base_url}/auth/validate",
                json={"api_key": self.config.api_key},
                timeout=10
            )

            if response.status_code == 200:
                data = response.json()
                logger.success(f"API key valid: user_id={data.get('user_id')}")
                return True
            else:
                logger.error(f"API key validation failed: {response.status_code}")
                return False

        except Exception as e:
            logger.error(f"Error validating API key: {e}")
            return False

    def check_quota(self) -> Dict:
        """Check current usage and limits."""
        try:
            response = requests.post(
                f"{self.config.base_url}/billing/check-limits",
                json={"api_key": self.config.api_key},
                timeout=10
            )

            if response.status_code == 200:
                return response.json()
            else:
                return {"error": f"Status {response.status_code}"}

        except Exception as e:
            return {"error": str(e)}

    def execute_bioql_circuit(
        self,
        circuit_data: Dict[str, Any],
        backend: str = "simulator",
        shots: int = 1024
    ) -> Dict:
        """
        Execute circuit on BioQL backend.

        Args:
            circuit_data: Circuit specification
            backend: BioQL backend ('simulator', 'ibm_torino', 'ionq_forte', etc.)
            shots: Number of shots

        Returns:
            Execution results
        """
        try:
            # Prepare request
            payload = {
                "api_key": self.config.api_key,
                "circuit": circuit_data,
                "backend": backend,
                "shots": shots,
                "qec_enabled": self.enable_qec,
                "qec_distance": self.qec_distance if self.enable_qec else None
            }

            logger.info(f"Executing on BioQL backend: {backend}, shots={shots}")

            # Execute
            response = requests.post(
                f"{self.config.base_url}/quantum/execute",
                json=payload,
                timeout=self.config.timeout
            )

            if response.status_code == 200:
                return response.json()
            else:
                error_msg = f"Execution failed: {response.status_code}"
                logger.error(error_msg)
                return {"error": error_msg, "details": response.text}

        except Exception as e:
            logger.error(f"Error executing circuit: {e}")
            return {"error": str(e)}

    def execute(
        self,
        circuit: Any,
        shots: int = 1024,
        backend: str = "simulator",
        **kwargs
    ) -> BackendResult:
        """
        Execute circuit with QEC protection.

        Args:
            circuit: Quantum circuit (Stim, QASM, or BioQL format)
            shots: Number of shots
            backend: Target backend
            **kwargs: Additional options

        Returns:
            BackendResult with QEC-corrected outcomes
        """
        # Convert circuit to BioQL format
        circuit_data = self._convert_to_bioql_format(circuit)

        # Execute on BioQL
        result = self.execute_bioql_circuit(circuit_data, backend, shots)

        if "error" in result:
            raise RuntimeError(f"BioQL execution error: {result['error']}")

        # Extract results
        measurements = np.array(result.get("measurements", []))
        raw_energy = result.get("energy", None)

        # Apply QEC if enabled
        if self.enable_qec and measurements.size > 0:
            logger.info("Applying QEC correction...")
            corrected_result = self._apply_qec_correction(measurements, circuit)

            # Log usage with QEC overhead
            self._record_usage(shots, backend, qec_overhead=1.5)

            return BackendResult(
                syndromes=corrected_result["syndromes"],
                observables=corrected_result["observables"],
                metadata={
                    "raw_energy": raw_energy,
                    "corrected_energy": corrected_result.get("corrected_energy"),
                    "logical_error_rate": corrected_result.get("logical_error_rate"),
                    "backend": backend,
                    "qec_distance": self.qec_distance,
                    "qec_enabled": True
                },
                backend_name=f"BioQL-{backend}"
            )
        else:
            # No QEC, return raw results
            self._record_usage(shots, backend, qec_overhead=1.0)

            return BackendResult(
                syndromes=np.array([]),
                observables=measurements if measurements.size > 0 else np.array([]),
                metadata={
                    "energy": raw_energy,
                    "backend": backend,
                    "qec_enabled": False
                },
                backend_name=f"BioQL-{backend}"
            )

    def _convert_to_bioql_format(self, circuit: Any) -> Dict:
        """Convert circuit to BioQL API format."""
        if isinstance(circuit, stim.Circuit):
            return {
                "format": "stim",
                "circuit_string": str(circuit),
                "num_qubits": circuit.num_qubits,
                "num_measurements": circuit.num_measurements
            }
        elif isinstance(circuit, str):
            # Assume QASM
            return {
                "format": "qasm",
                "circuit_string": circuit
            }
        elif isinstance(circuit, dict):
            # Already in BioQL format
            return circuit
        else:
            raise ValueError(f"Unsupported circuit type: {type(circuit)}")

    def _apply_qec_correction(
        self,
        measurements: np.ndarray,
        circuit: Any
    ) -> Dict:
        """Apply QEC decoding to measurements."""
        try:
            # Build surface code for this circuit
            surface_code = RotatedSurfaceCode(
                distance=self.qec_distance,
                rounds=self.qec_distance
            )

            # Build QEC circuit
            qec_circuit = surface_code.build_stim_circuit()

            # Sample to get syndromes
            sampler = qec_circuit.compile_detector_sampler()
            num_shots = len(measurements)

            syndromes, observables = sampler.sample(
                shots=num_shots,
                separate_observables=True
            )

            # Decode
            predictions = self.decoder.decode_batch(syndromes, qec_circuit)

            # Compute logical error rate
            logical_errors = np.sum(predictions != observables[:, 0])
            logical_error_rate = logical_errors / num_shots

            logger.success(
                f"QEC correction applied: "
                f"P_L={logical_error_rate:.6f}, "
                f"{logical_errors}/{num_shots} errors"
            )

            return {
                "syndromes": syndromes,
                "observables": observables,
                "predictions": predictions,
                "logical_error_rate": logical_error_rate,
                "logical_errors": logical_errors
            }

        except Exception as e:
            logger.error(f"QEC correction failed: {e}")
            return {
                "syndromes": np.array([]),
                "observables": measurements,
                "predictions": None,
                "error": str(e)
            }

    def _record_usage(
        self,
        shots: int,
        backend: str,
        qec_overhead: float = 1.0
    ):
        """Record usage with BioQL billing."""
        try:
            # Calculate cost with QEC overhead
            base_cost = shots * 0.001  # $0.001 per shot
            qec_cost = base_cost * qec_overhead

            payload = {
                "api_key": self.config.api_key,
                "shots": shots,
                "backend": backend,
                "qec_enabled": self.enable_qec,
                "qec_distance": self.qec_distance if self.enable_qec else None,
                "qec_overhead_multiplier": qec_overhead,
                "total_cost_usd": qec_cost
            }

            response = requests.post(
                f"{self.config.base_url}/billing/record-usage",
                json=payload,
                timeout=10
            )

            if response.status_code == 200:
                logger.debug(f"Usage recorded: ${qec_cost:.4f}")
            else:
                logger.warning(f"Failed to record usage: {response.status_code}")

        except Exception as e:
            logger.warning(f"Error recording usage: {e}")


class BioQLOptimizer:
    """
    High-level optimizer for BioQL quantum chemistry.

    Provides easy-to-use interface for QEC-protected calculations.
    """

    def __init__(
        self,
        api_key: Optional[str] = None,
        qec_distance: int = 5,
        auto_optimize: bool = True
    ):
        """
        Initialize BioQL optimizer.

        Args:
            api_key: BioQL API key
            qec_distance: Surface code distance
            auto_optimize: Automatically apply QEC
        """
        config = BioQLConfig(api_key=api_key) if api_key else BioQLConfig.from_env()
        self.backend = BioQLBackend(config, qec_distance=qec_distance)
        self.auto_optimize = auto_optimize

        logger.info(f"BioQL Optimizer initialized (d={qec_distance})")

    def execute_with_qec(
        self,
        bioql_query: str,
        backend: str = "simulator",
        shots: int = 2048,
        **kwargs
    ) -> Dict:
        """
        Execute BioQL query with QEC protection.

        Args:
            bioql_query: Natural language query or circuit
            backend: Target backend
            shots: Number of shots
            **kwargs: Additional options

        Returns:
            Results with QEC correction

        Example:
        -------
        >>> optimizer = BioQLOptimizer()
        >>> result = optimizer.execute_with_qec(
        ...     "apply VQE to H2 molecule",
        ...     backend="ibm_torino",
        ...     shots=2048
        ... )
        >>> print(f"Energy: {result['energy']:.6f} Hartree")
        """
        logger.info(f"Executing BioQL query with QEC: {bioql_query}")

        # Parse query and convert to circuit
        circuit_data = {
            "query": bioql_query,
            "type": "natural_language"
        }

        # Execute with QEC
        result = self.backend.execute(
            circuit_data,
            shots=shots,
            backend=backend,
            **kwargs
        )

        # Format results
        return {
            "energy": result.metadata.get("corrected_energy") or result.metadata.get("energy"),
            "raw_energy": result.metadata.get("raw_energy"),
            "logical_error_rate": result.metadata.get("logical_error_rate"),
            "backend": backend,
            "shots": shots,
            "qec_distance": self.backend.qec_distance,
            "qec_enabled": self.backend.enable_qec,
            "metadata": result.metadata
        }

    def protect_circuit(
        self,
        circuit: Any,
        distance: int = None
    ) -> stim.Circuit:
        """
        Protect circuit with surface code.

        Args:
            circuit: Input circuit
            distance: Code distance (uses default if None)

        Returns:
            QEC-protected circuit
        """
        d = distance or self.backend.qec_distance

        surface_code = RotatedSurfaceCode(distance=d, rounds=d)
        protected_circuit = surface_code.build_stim_circuit()

        logger.info(f"Circuit protected with d={d} surface code")

        return protected_circuit

    def benchmark_qec_overhead(
        self,
        circuit: Any,
        distances: list = [3, 5, 7],
        shots: int = 1000
    ) -> Dict:
        """
        Benchmark QEC overhead for different code distances.

        Args:
            circuit: Test circuit
            distances: Code distances to test
            shots: Number of shots

        Returns:
            Benchmark results
        """
        results = {}

        for d in distances:
            logger.info(f"Benchmarking d={d}...")

            # Temporarily set distance
            original_distance = self.backend.qec_distance
            self.backend.qec_distance = d

            start = time.time()
            result = self.backend.execute(circuit, shots=shots)
            runtime = time.time() - start

            results[f"d={d}"] = {
                "distance": d,
                "runtime": runtime,
                "logical_error_rate": result.metadata.get("logical_error_rate"),
                "overhead_factor": runtime / (shots * 0.001)  # Relative to base
            }

            # Restore
            self.backend.qec_distance = original_distance

        logger.success("Benchmark complete")
        return results


def cli_protect_bioql(
    query: str,
    backend: str = "simulator",
    shots: int = 2048,
    qec_distance: int = 5,
    output_file: str = None
):
    """
    CLI function for protecting BioQL queries.

    Can be invoked from terminal:
    $ quillow protect-bioql --query "dock aspirin to COX-2" --backend ionq_forte
    """
    optimizer = BioQLOptimizer(qec_distance=qec_distance)

    logger.info(f"Protecting BioQL query: {query}")
    logger.info(f"Backend: {backend}, Shots: {shots}, QEC distance: {qec_distance}")

    result = optimizer.execute_with_qec(
        bioql_query=query,
        backend=backend,
        shots=shots
    )

    # Print results
    print("\n" + "="*60)
    print("BioQL QEC-Protected Execution Results")
    print("="*60)
    print(f"Query: {query}")
    print(f"Backend: {backend}")
    print(f"Shots: {shots}")
    print(f"QEC Distance: {qec_distance}")
    print("-"*60)
    print(f"Energy: {result.get('energy', 'N/A')}")
    if result.get('raw_energy'):
        print(f"Raw Energy (no QEC): {result['raw_energy']}")
        improvement = abs(result['raw_energy'] - result['energy'])
        print(f"QEC Improvement: {improvement:.6f}")
    print(f"Logical Error Rate: {result.get('logical_error_rate', 'N/A')}")
    print("="*60 + "\n")

    # Save to file if requested
    if output_file:
        import json
        with open(output_file, 'w') as f:
            json.dump(result, f, indent=2)
        print(f"Results saved to: {output_file}")

    return result


__all__ = [
    'BioQLBackend',
    'BioQLOptimizer',
    'BioQLConfig',
    'cli_protect_bioql'
]
