#!/usr/bin/env python3
"""
Abstract Backend Base Class
===========================

Defines interface for all quantum backends.
"""

import numpy as np
from typing import Optional, Dict, Any
from dataclasses import dataclass
from abc import ABC, abstractmethod


@dataclass
class BackendResult:
    """Results from backend execution."""
    syndromes: np.ndarray
    observables: np.ndarray
    metadata: Dict[str, Any]
    backend_name: str

    @property
    def num_shots(self) -> int:
        return self.syndromes.shape[0]


class AbstractBackend(ABC):
    """
    Abstract base class for quantum backends.

    All backends must implement execute() method.
    """

    def __init__(self, name: str = "AbstractBackend"):
        """Initialize backend."""
        self.name = name

    @abstractmethod
    def execute(
        self,
        circuit: Any,
        shots: int = 1024,
        **kwargs
    ) -> BackendResult:
        """
        Execute circuit on backend.

        Args:
            circuit: Quantum circuit (format depends on backend)
            shots: Number of shots
            **kwargs: Backend-specific options

        Returns:
            BackendResult with syndromes and observables
        """
        pass
