# AUTOGENERATED! DO NOT EDIT! File to edit: 01_stats.ipynb (unless otherwise specified).

__all__ = ['get_page_count', 'get_file_metadata', 'add_comma_separation', 'has_ocr_layer', 'get_stats',
           'convert_timestamp', 'get_json_stats', 'get_dataframe_stats', 'export_stats_as_csv', 'display_stats']

# Cell
import json
import os
import datetime
from pathlib import Path
from typing import Any

import pandas as pd
import numpy as np
import fitz

from rich import print
from rich.console import Console
from rich.table import Table

# Cell
def get_page_count(filepath: "os.PathLike[Any]") -> int:
    """Gets the page count of a PDF file."""
    pdf_obj = fitz.open(filepath)
    return pdf_obj.page_count

# Cell
def get_file_metadata(filepath: "os.PathLike[Any]") -> dict:
    """Gets the metadata associated with a PDF file."""
    pdf_obj = fitz.open(filepath)
    return pdf_obj.metadata

# Cell
def add_comma_separation(input: int) -> str:
    """Adds comma-separation for thousands to an integer."""
    return f'{input:,.0f}'

# Cell
def has_ocr_layer(filepath: "os.PathLike[Any]") -> bool:
    """Checks whether a particular file has an OCR layer."""
    # TODO: fix this function
#     pdf_obj = fitz.open(filepath)
#     return len(pdf_obj[0].get_text("text")) == 0
    return False

# Cell
def get_stats(source_path: "os.PathLike[Any]") -> list:
    """Gathers statistics on the PDF data contained in a particular directory."""
    stats_data = []

    files = list(source_path.glob("**/*.pdf")) # searches source_path and all subfolders

    for file in files:
        file_data = {
            "filename": file.name,
            "pagecount": get_page_count(file),
            "has_ocr_layer": has_ocr_layer(file),
            "pdf_file_size_bytes": os.path.getsize(file),
            "date_created": datetime.datetime.fromtimestamp(os.path.getctime(file)),
            "date_last_modified": datetime.datetime.fromtimestamp(os.path.getmtime(file)),
            "author": get_file_metadata(file)['author'],
        }
        stats_data.append(file_data)

    return stats_data

# Cell
def convert_timestamp(item_date_object):
    """Helper function to convert a datetime object to timestamp when
    needed for a JSON object."""
    if isinstance(item_date_object, (datetime.date, datetime.datetime)):
        return item_date_object.timestamp()

# Cell
def get_json_stats(source_path: "os.PathLike[Any]") -> str:
    """Gathers statistics on the PDF data in a directory in JSON format."""
    return json.dumps(get_stats(source_path), default=convert_timestamp)

# Cell
def get_dataframe_stats(source_path: "os.PathLike[Any]") -> pd.core.frame.DataFrame:
    """Gathers statistics on the PDF data in a directory as a dataframe."""
    return pd.DataFrame.from_dict(get_stats(source_path))

# Cell
def export_stats_as_csv(source_path: "os.PathLike[Any]", destination_path: "os.PathLike[Any]" = Path("./stats.csv")) -> None:
    """Exports statistics on the PDF data as a CSV file."""
    get_dataframe_stats(source_path).to_csv(destination_path)

# Cell
def display_stats(stats_list: list) -> None:
    """Displays statistics on the PDF data contained in a particular directory."""
    table = Table(title="Stats for your PDF Files")

    table.add_column("PageCount", justify="right", style="green")
    table.add_column("Filename", justify="left", style="cyan", no_wrap=True, max_width=50)
    table.add_column("ocr_layer", justify="left", style="blue")
    table.add_column("pdf_file_size_bytes", justify="left", style="purple")
    table.add_column("author", justify="left", style="black")
    page_count = 0
    file_count = 0

    for item in stats_list:
        page_count += item['pagecount']
        file_count += 1

        if file_count <= 45:
            table.add_row(
                str(item['pagecount']),
                item['filename'],
                str(item['has_ocr_layer']),
                str(item['pdf_file_size_bytes']),
                item['author'],
            )
    if file_count > 45:
        table.add_row("...", "...", "...", "...", "...")

    console = Console()
    console.print(table)

    bold_str_count = f"[bold red]{add_comma_separation(page_count)}"
    console.print("[bold red]TOTAL PAGECOUNT:", bold_str_count)