# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['asynctinydb']

package_data = \
{'': ['*']}

install_requires = \
['nest-asyncio>=1.5.5,<2.0.0',
 'types-ujson>=5.5.0,<6.0.0',
 'ujson>=5.4.0,<6.0.0']

extras_require = \
{':python_full_version <= "3.10.0"': ['typing-extensions>=3.10.0,<5.0.0'],
 'all': ['pycryptodome>=3.15.0,<4.0.0',
         'Brotli>=1.0.9,<2.0.0',
         'blosc2>=0.4.1,<0.5.0'],
 'compression': ['Brotli>=1.0.9,<2.0.0', 'blosc2>=0.4.1,<0.5.0'],
 'encryption': ['pycryptodome>=3.15.0,<4.0.0']}

setup_kwargs = {
    'name': 'async-tinydb',
    'version': '1.4.1',
    'description': 'Yet Another Async TinyDB',
    'long_description': '<img src="./artwork/logo.png" alt="logo" style="zoom:50%;" />\n\n# What\'s This?\n\nAn asynchronous version of `TinyDB`.\n\nAlmost every method is asynchronous. And it\'s based on `TinyDB 4.7.0+`.  \n\nI will try to keep up with the latest version of `TinyDB`.\n\n\n\n# Incompatible Changes\n\n* **Asynchronous**: Say goodbye to blocking IO. **Don\'t forget to `await` async methods**!\n\n* **Drop support**: Only supports Python 3.10+.  (for 3.8+ go to this [branch](https://github.com/VermiIIi0n/async-tinydb/tree/legacy))\n\n* **`ujson`:** Using `ujson` instead of `json`. Some arguments aren\'t compatible with `json`[^1]\n\n* **Storage `closed` property**: Original `TinyDB` won\'t raise exceptions when operating on a closed file. Now the property `closed` of `Storage` classes is required to be implemented. An `IOError` should be raised.\n\n* **[Miscellaneous](#misc)**: Differences only matter in edge cases.\n\n# New Features\n\n* **Event hooks**: You can now use event hooks to do something before or after an operation. See [Event Hooks](#event-hooks) for more details.\n\n* **Redesigned ID & Doc class**: You can [replace](#replacing-id-&-document-class) and [customise them](#customise-id-class) more pleasingly.\n  \n* **DB-level caching**: This significantly improves the performance of all operations. However, the responsibility of converting the data to the correct type is transferred to the Storage[^2][^disable-db-level]. \n\n* **Built-in `Modifier`**: Use `Modifier` to easily [encrypt](#encryption), [compress](./docs/Modifier.md#Compression) and [extend types](./docs/Modifier.md#Conversion) of your database. Sure you can do much more than these. _(See [Modifier](./docs/Modifier.md))_\n\n* **Isolation Level**: Performance or ACID? It\'s up to you[^isolevel].\n\n* **Atomic Write**: **A**CID!\n\n# How to use it?\n\n## Installation\n\n* Minimum: `pip install async-tinydb`\n* Encryption: `pip install async-tinydb[encryption]`\n* Compression: `pip install async-tinydb[compression]`\n* Full: `pip install async-tinydb[all]`\n\n## Importing\n\n```Python\nfrom asynctinydb import TinyDB, where\n```\n\n## Using\n\nSee the [original `TinyDB` documents](https://tinydb.readthedocs.org). Insert an `await` in front of async methods. \n\nNotice that some codes are still blocking, for example, when calling `len()` on `TinyDB` or `Table` Objects.\n\nThat\'s it.\n\n******\n\n## Documents For Advances Usage\n\n* [Modifier](./docs/Modifier.md)\n* [Event Hooks](./docs/EventHooks.md)\n\n## Replacing ID & Document Class\n\n**NOTICE: Mixing classes in one table may cause errors!**\n\nWhen a table exists in a file, `Async-TinyDB` won\'t determine classes by itself, it is your duty to make sure classes are matching.\n\n### ID Classes\n\n* `IncreID`: Default ID class, mimics the behaviours of the original `int` ID but requires much fewer IO operations.\n* `UUID`: Uses `uuid.UUID`[^uuid-version].\n\n### Document Class\n\n* `Document`: Default document class, uses `dict`under the bonet.\n\n```Python\nfrom asynctinydb import TinyDB, UUID, IncreID, Document\n\ndb = TinyDB("database.db")\n\n# Setting ID class to UUID, document class to Document\ntab = db.table("table1", document_id_class=UUID, document_class=Document)\n```\n\n_See [Customisation](#customise-id-class) for more details_\n\n## Encryption\n\nCurrently only supports AES-GCM encryption.\n\nThere are two ways to use encryption:\n\n### 1. Use `EncryptedJSONStorage` directly\n\n```Python\nfrom asynctinydb import EncryptedJSONStorage, TinyDB\n\nasync def main():\n    db = TinyDB("db.json", key="your key goes here", storage=EncryptedJSONStorage)\n\n```\n\n### 2. Use  `Modifier` class\n\n_See [Encryption](./docs/Modifier.md#Encryption)_\n\n## Isolation Level\n\nTo avoid blocking codes, Async-TinyDB puts CPU-bound tasks into another thread (Useful with interpreters without GIL)\n\nUnfortunately, this introduces chances of data being modified when:\n\n* Manipulating mutable objects within `Document` instances in another coroutine\n* Performing updating/saving/searching operations (These operations are run in a different thread/process)\n* The conditions above are satisfied in the same `Table`\n* The conditions above are satisfied simultaneously.\n\nAvoid these operations or set a higher isolation level to mitigate this problem.\n\n```Python\ndb.isolevel = 2\n```\n\n`isolevel`:\n\n0. No isolation\n1. Serialised CRUD operations (Also ensures thread safety) (default)\n2. Deepcopy documents on CRUD (Ensures `Index` & `Query Cache` consistency)\n\n\n\n## DB-level caching\n\nDB-level caching improves performance dramatically.\n\nHowever, this may cause data inconsistency between `Storage` and `TinyDB` if the file that `Storage` referred to is been shared.\n\nTo disable it:\n\n```Python\ndb = TinyDB("./path", no_dbcache=True)\n```\n\n# Example Codes:\n\n## Simple One\n\n```Python\nimport asyncio\nfrom asynctinydb import TinyDB, Query\n\nasync def main():\n    db = TinyDB(\'test.json\')\n    await db.insert({"answer": 42})\n    print(await db.search(Query().answer == 42))  # >>> [{\'answer\': 42}] \n\nasyncio.run(main())\n```\n## Event Hooks Example\n\n```Python\nasync def main():\n    db = TinyDB(\'test.json\')\n\n    @db.storage.on.write.pre\n    async def mul(ev: str, s: Storage, data: dict):\n        data["_default"]["1"][\'answer\'] *= 2  # directly manipulate on data\n\n    @db.storage.on.write.post\n    async def _print(ev, s, anystr):\n      \tprint(anystr)  # print json dumped string\n \n    await db.insert({"answer": 21})  # insert() will trigger both write events\n    await db.close()\n    # Reload\n    db = TinyDB(\'test.json\')\n    print(await db.search(Query().answer == 42))  # >>> [{\'answer\': 42}] \n```\n\n## Customise ID Class\n\nInherit from `BaseID` and implement the following methods, and then you are good to go.\n\n```Python\nfrom asynctinydb import BaseID\n\nclass MyID(BaseID):\n  def __init__(self, value: Any):\n        """\n        You should be able to convert str into MyID instance if you want to use JSONStorage.\n        """\n\n    def __str__(self) -> str:\n        """\n        Optional.\n        It should be implemented if you want to use JSONStorage.\n        """\n\n    def __hash__(self) -> int:\n        ...\n\n    def __eq__(self, other: object) -> bool:\n        ...\n\n    @classmethod\n    def next_id(cls, table: Table, keys) -> MyID:\n        """\n        It should return a unique ID.\n        """\n\n    @classmethod\n    def mark_existed(cls, table: Table, new_id):\n        """\n        Marks an ID as existing; the same ID shouldn\'t be generated by next_id.\n        """\n\n    @classmethod\n    def clear_cache(cls, table: Table):\n        """\n        Clear cache of existing IDs, if such cache exists.\n        """\n```\n\n## Customise Document Class\n\n```Python\nfrom asynctinydb import BaseDocument\n\nclass MyDoc(BaseDocument):\n  """\n  I am too lazy to write those necessary methods.\n  """\n```\n\nAnyways, a BaseDocument class looks like this:\n\n```Python\nclass BaseDocument(Mapping[IDVar, Any]):\n    @property\n    @abstractmethod\n    def doc_id(self) -> IDVar:\n        raise NotImplementedError()\n\n    @doc_id.setter\n    def doc_id(self, value: IDVar):\n        raise NotImplementedError()\n```\n\nMake sure you have implemented all the methods required by  `BaseDocument` class.\n\n# Misc\n\n* **Lazy-load:** File loading & dirs creating are delayed to the first IO operation.\n* **`CachingMiddleWare`**: `WRITE_CACHE_SIZE` is now instance-specific.  \n  Example: `TinyDB("test.db", storage=CachingMiddleWare(JSONStorage, 1024))`\n\n\n\n[^1]: Why not `orjson`? Because `ujson` is fast enough and has more features.\n[^2]: e.g. `JSONStorage` needs to convert the keys to `str` by itself.\n[^UUID-version]:Currently using UUID4\n[^disable-db-level]: See [DB-level caching](#db-level-caching) to learn how to disable this feature if it causes dirty reads.\n[^isolevel]: See [isolevel](#isolation-level)\n',
    'author': 'VermiIIi0n',
    'author_email': 'dungeon.behind0t@icloud.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/VermiIIi0n/async-tinydb',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
