import copy
import re

from cerberus import Validator

from one_table.Utils import get_iso_8601_date


class Table(object):
    def __init__(self, params: dict):
        if 'name' not in params:
            raise Exception('Missing name property')

        self.table_name = params['name']
        self.schema = params['schema']
        self.models = params['schema']['models']
        self.client = params['client'].Table(params['name'])
        self.pk_key = self.schema['indexes']['primary']['hash']
        self.sk_key = self.schema['indexes']['primary']['sort']

    def prepare_insert_data(self, model, data):
        pass

    def create(self, model, data_):
        _schema = copy.deepcopy(self.models[model])
        sk = self.models[model][self.sk_key]['value']
        pk = self.models[model][self.pk_key]['value']
        del _schema[self.pk_key]
        del _schema[self.sk_key]
        v = Validator(_schema)
        v.validate(data_)
        if v.validate(data_):
            data_[self.pk_key] = pk.format(**data_)
            data_[self.sk_key] = sk.format(**data_)
            data_['created_at'] = get_iso_8601_date()
            data_['updated_at'] = get_iso_8601_date()
            data_['_type'] = model
            self.client.put_item(Item=data_)
            return {k: v for k, v in data_.items() if k in _schema.keys()}
        else:
            raise Exception(v.errors)

    def update(self, model, data_):
        _schema = copy.deepcopy(self.models[model])
        _data = copy.deepcopy(data_)
        sk = self.models[model][self.sk_key]['value']
        pk = self.models[model][self.pk_key]['value']
        del _schema[self.pk_key]
        del _schema[self.sk_key]
        pk_keys = re.findall(r'\{([A-Za-z0-9_]+)\}', pk)
        sk_keys = re.findall(r'\{([A-Za-z0-9_]+)\}', sk)
        index_keys = pk_keys + sk_keys
        for item in _schema:
            if item in index_keys:
                _schema[item]['required'] = True
            else:
                _schema[item]['required'] = False

        v = Validator(_schema)
        v.validate(data_)
        if v.validate(data_):
            data_key = data_.keys()
            expression_attribute_names = {}
            expression_attribute_values = {}
            update_expression = []
            data_['updated_at'] = get_iso_8601_date()
            for idx, key in enumerate(data_key):
                expression_attribute_names['#k{}'.format(idx)] = key
                expression_attribute_values[':v{}'.format(idx)] = data_[key]
                update_expression.append('#k{} = :v{}'.format(idx, idx))

            pk_key_parsed = pk.format(**data_)
            sk_key_parsed = sk.format(**data_)

            request = {
                'ConditionExpression': '(attribute_exists({})) and (attribute_exists({}))'.format(self.pk_key,
                                                                                                  self.sk_key),
                'ExpressionAttributeNames': expression_attribute_names,
                'ExpressionAttributeValues': expression_attribute_values,
                'UpdateExpression': 'set {}'.format(', '.join(update_expression)),
                'ReturnValues': 'ALL_NEW',
                'Key': {
                    'pk': pk_key_parsed,
                    'sk': sk_key_parsed
                }
            }
            response = self.client.update_item(**request)
            if 'Attributes' in response:
                return {k: v for k, v in response['Attributes'].items() if k in _schema.keys()}
            return response
        else:
            raise Exception(v.errors)

    def find(self, model, key, value):
        _schema = copy.deepcopy(self.models[model])
        del _schema[self.pk_key]
        del _schema[self.sk_key]
        request = {
            'ExpressionAttributeNames': {
                '#n0': self.schema['indexes']['primary']['hash'],
                '#n1': self.schema['indexes']['primary']['sort'],
                '#_2': key

            },
            'ExpressionAttributeValues': {
                ':v0': '{}#{}'.format(model, value),
                ':v1': '{}#'.format(model),
                ':_2': value

            },
            'FilterExpression': '#_2 = :_2',
            'KeyConditionExpression': '#n0 = :v0 and begins_with(#n1, :v1)',
        }
        response = self.client.query(**request)
        if len(response['Items']) > 0:
            return {k: v for k, v in response['Items'][0].items() if k in _schema.keys()}

        return response['Items']
