"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
/**
 * Type of route
 */
var RouteType;
(function (RouteType) {
    /**
     * HTTP route
     */
    RouteType["HTTP"] = "http";
    /**
     * TCP route
     */
    RouteType["TCP"] = "tcp";
})(RouteType = exports.RouteType || (exports.RouteType = {}));
/**
 * Route represents a new or existing route attached to a VirtualRouter and Mesh
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/routes.html
 */
class Route extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.routeName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        this.weightedTargets = new Array();
        this.virtualRouter = props.virtualRouter;
        const routeType = props.routeType !== undefined ? props.routeType :
            props.prefix !== undefined ? RouteType.HTTP :
                RouteType.TCP;
        if (routeType === RouteType.HTTP) {
            this.httpRoute = this.renderHttpRoute(props);
        }
        else {
            this.tcpRoute = this.renderTcpRoute(props);
        }
        const route = new appmesh_generated_1.CfnRoute(this, 'Resource', {
            routeName: this.physicalName,
            meshName: this.virtualRouter.mesh.meshName,
            virtualRouterName: this.virtualRouter.virtualRouterName,
            spec: {
                tcpRoute: this.tcpRoute,
                httpRoute: this.httpRoute,
            },
        });
        this.routeName = this.getResourceNameAttribute(route.attrRouteName);
        this.routeArn = this.getResourceArnAttribute(route.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualRouter/${props.virtualRouter.virtualRouterName}/route`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing route given an ARN
     */
    static fromRouteArn(scope, id, routeArn) {
        return new ImportedRoute(scope, id, { routeArn });
    }
    /**
     * Import an existing route given its name
     */
    static fromRouteName(scope, id, meshName, virtualRouterName, routeName) {
        return new ImportedRoute(scope, id, { meshName, virtualRouterName, routeName });
    }
    /**
     * Utility method to add weighted route targets to an existing route
     */
    renderWeightedTargets(props) {
        for (const t of props) {
            this.weightedTargets.push({
                virtualNode: t.virtualNode.virtualNodeName,
                weight: t.weight || 1,
            });
        }
        return this.weightedTargets;
    }
    renderHttpRoute(props) {
        return {
            match: {
                prefix: props.prefix || '/',
            },
            action: {
                weightedTargets: this.renderWeightedTargets(props.routeTargets),
            },
        };
    }
    renderTcpRoute(props) {
        return {
            action: {
                weightedTargets: this.renderWeightedTargets(props.routeTargets),
            },
        };
    }
}
exports.Route = Route;
/**
 * Represents and imported IRoute
 */
class ImportedRoute extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.routeArn) {
            this.routeArn = props.routeArn;
            this.routeName = cdk.Fn.select(4, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.routeArn).resourceName));
        }
        else if (props.routeName && props.meshName && props.virtualRouterName) {
            this.routeName = props.routeName;
            this.routeArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualRouter/${props.virtualRouterName}/route`,
                resourceName: this.routeName,
            });
        }
        else {
            throw new Error('Need either arn or three names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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