# This file is part of the Reproducible and Reusable Data Analysis Workflow
# Server (flowServ).
#
# Copyright (C) 2019-2020 NYU.
#
# flowServ is free software; you can redistribute it and/or modify it under the
# terms of the MIT License; see LICENSE file for more details.

"""Base class for workflow template parameters. Each parameter has a set of
properties that are used to (i) identify the parameter, (ii) define a nested
parameter structure, and (iii) render UI forms to collect parameter values.
"""

from abc import ABCMeta, abstractmethod

import flowserv.util as util


class ParameterBase(metaclass=ABCMeta):
    """Base class for template parameters. The base class maintains the unique
    parameter identifier, the type identifier, the parameter name, the optional
    description, an optional default value, the is_required flag, the parameter
    index position, and the optional parameter group.

    Implementing classes have to provide a static .from_dict() method that
    returns an instance of the class from a dictionary serialization. The
    dictionary serializations for each class are generated by the .to_dict()
    method.
    """
    def __init__(
        self, para_id, type_id, name, index, description=None,
        default_value=None, is_required=False, module_id=None
    ):
        """Initialize the base properties for a template parameter.

        Parameters
        ----------
        para_id: string
            Unique parameter identifier
        type_id: string
            Parameter type identifier.
        name: string
            Human-readable parameter name.
        index: int
            Index position of the parameter (for display purposes).
        description: string, default=None
            Descriptive text for the parameter.
        default_value: any, default=None
            Optional default value.
        is_required: bool, default=False
            Is required flag.
        module_id: string, default=None
            Optional identifier for parameter group that this parameter
            belongs to.
        """
        if para_id is None:
            raise ValueError('invalid identifier')
        self.para_id = para_id
        self.type_id = type_id
        self.name = name
        self.index = index
        self.description = description
        self.default_value = default_value
        self.is_required = is_required
        self.module_id = module_id

    def prompt(self):
        """Get default input prompt for the parameter declaration. The prompt
        contains an indication of the data type, the parameter name and the
        default value (if defined).
        Returns
        -------
        string
        """
        val = '{} ({})'.format(self.name, self.type_id)
        if self.default_value is not None:
            val += " [default '{}']".format(self.default_value)
        return val + ' $> '

    @abstractmethod
    def to_argument(self, value):
        """Validate the given argument value for the parameter type. Returns
        the argument representation for the value that is used to replace
        references to the parameter in workflow templates.

        Raises an InvalidArgumentError if the given value is not valid for the
        parameter type.

        Parameters
        ----------
        value: any
            User-provided value for a template parameter.

        Returns
        -------
        sting, float, or int

        Raises
        ------
        flowserv.error.InvalidArgumentError
        """
        raise NotImplementedError()  # pragma: no cover

    def to_dict(self):
        """Get dictionary serialization for the parameter declaration.
        Implementing classes can add elements to the base dictionary.

        Returns
        -------
        dict
        """
        return {
            'id': self.para_id,
            'type': self.type_id,
            'name': self.name,
            'index': self.index,
            'description': self.description,
            'defaultValue': self.default_value,
            'isRequired': self.is_required,
            'module': self.module_id
        }


class ParameterGroup(object):
    """Parameter groups are identifiable sets of parameters. These sets are
    primarily intended for display purposes in the front-end. Therefore, each
    group has a display name and an index position that defines the sort order
    for groups.
    """
    def __init__(self, module_id, name, index):
        """Initialize the object properties.

        Parameters
        ----------
        module_id: string
            Unique group identifier
        name: string
            Human-readable group name
        index: int
            Group sort order index
        """
        self.module_id = module_id
        self.name = name
        self.index = index

    @classmethod
    def from_dict(cls, doc, validate=True):
        """Create object instance from dictionary serialization.

        Parameters
        ----------
        doc: dict
            Dictionary serialization for parameter group handles
        validate: bool, default=True
            Validate the serialization if True.

        Returns
        -------
        flowserv.module.parameter.base.ParameterGroup

        Raises
        ------
        ValueError
        """
        if validate:
            util.validate_doc(
                doc,
                mandatory=['id', 'name', 'index']
            )
        return cls(
            module_id=doc['id'],
            name=doc['name'],
            index=doc['index']
        )

    def to_dict(self):
        """Get dictionary serialization for parameter group handle.

        Returns
        -------
        dict
        """
        return {
            'id': self.module_id,
            'name': self.name,
            'index': self.index
        }
