"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const have_resource_1 = require("@aws-cdk/assert/lib/assertions/have-resource");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('User Pool', () => {
    test('default setup', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool');
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPool', {
            AdminCreateUserConfig: {
                AllowAdminCreateUserOnly: true,
                InviteMessageTemplate: have_resource_1.ABSENT,
            },
            EmailVerificationMessage: 'The verification code to your new account is {####}',
            EmailVerificationSubject: 'Verify your new account',
            SmsVerificationMessage: 'The verification code to your new account is {####}',
            VerificationMessageTemplate: {
                DefaultEmailOption: 'CONFIRM_WITH_CODE',
                EmailMessage: 'The verification code to your new account is {####}',
                EmailSubject: 'Verify your new account',
                SmsMessage: 'The verification code to your new account is {####}',
            },
            SmsConfiguration: {
                SnsCallerArn: {
                    'Fn::GetAtt': ['PoolsmsRoleC3352CE6', 'Arn'],
                },
                ExternalId: 'Pool',
            },
            lambdaTriggers: have_resource_1.ABSENT,
        });
        expect(stack).toHaveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: 'sts:AssumeRole',
                        Condition: {
                            StringEquals: {
                                'sts:ExternalId': 'Pool',
                            },
                        },
                        Effect: 'Allow',
                        Principal: {
                            Service: 'cognito-idp.amazonaws.com',
                        },
                    },
                ],
                Version: '2012-10-17',
            },
            Policies: [
                {
                    PolicyDocument: {
                        Statement: [
                            {
                                Action: 'sns:Publish',
                                Effect: 'Allow',
                                Resource: '*',
                            },
                        ],
                        Version: '2012-10-17',
                    },
                    PolicyName: 'sns-publish',
                },
            ],
        });
    });
    test('self sign up option is correctly configured', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            selfSignUpEnabled: true,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPool', {
            AdminCreateUserConfig: {
                AllowAdminCreateUserOnly: false,
            },
        });
    });
    test('email verification via link is configured correctly', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            userVerification: {
                emailStyle: lib_1.VerificationEmailStyle.LINK,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            EmailVerificationMessage: have_resource_1.ABSENT,
            EmailVerificationSubject: have_resource_1.ABSENT,
            SmsVerificationMessage: 'The verification code to your new account is {####}',
            VerificationMessageTemplate: {
                DefaultEmailOption: 'CONFIRM_WITH_LINK',
                EmailMessageByLink: 'Verify your account by clicking on {##Verify Email##}',
                EmailSubjectByLink: 'Verify your new account',
                SmsMessage: 'The verification code to your new account is {####}',
            },
        });
    }),
        test('email and sms verification messages are validated', () => {
            const stack = new core_1.Stack();
            expect(() => new lib_1.UserPool(stack, 'Pool1', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.CODE,
                    emailBody: 'invalid email body',
                },
            })).toThrow(/Verification email body/);
            expect(() => new lib_1.UserPool(stack, 'Pool2', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.CODE,
                    emailBody: 'valid email body {####}',
                },
            })).not.toThrow();
            expect(() => new lib_1.UserPool(stack, 'Pool3', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.CODE,
                    smsMessage: 'invalid sms message',
                },
            })).toThrow(/SMS message/);
            expect(() => new lib_1.UserPool(stack, 'Pool4', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.CODE,
                    smsMessage: 'invalid sms message {####}',
                },
            })).not.toThrow();
            expect(() => new lib_1.UserPool(stack, 'Pool5', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.LINK,
                    emailBody: 'invalid email body {####}',
                },
            })).toThrow(/Verification email body/);
            expect(() => new lib_1.UserPool(stack, 'Pool6', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.LINK,
                    emailBody: 'invalid email body {##Verify Email##}',
                },
            })).not.toThrow();
        });
    test('validation is skipped for email and sms messages when tokens', () => {
        const stack = new core_1.Stack();
        const parameter = new core_1.CfnParameter(stack, 'Parameter');
        expect(() => new lib_1.UserPool(stack, 'Pool1', {
            userVerification: {
                emailStyle: lib_1.VerificationEmailStyle.CODE,
                emailBody: parameter.valueAsString,
            },
        })).not.toThrow();
        expect(() => new lib_1.UserPool(stack, 'Pool2', {
            userVerification: {
                emailStyle: lib_1.VerificationEmailStyle.CODE,
                smsMessage: parameter.valueAsString,
            },
        })).not.toThrow();
    });
    test('user invitation messages are configured correctly', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            userInvitation: {
                emailBody: 'invitation email body',
                emailSubject: 'invitation email subject',
                smsMessage: 'invitation sms',
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            AdminCreateUserConfig: {
                InviteMessageTemplate: {
                    EmailMessage: 'invitation email body',
                    EmailSubject: 'invitation email subject',
                    SMSMessage: 'invitation sms',
                },
            },
        });
    });
    test('smsRole property is recognized', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const role = aws_iam_1.Role.fromRoleArn(stack, 'smsRole', 'arn:aws:iam::664773442901:role/sms-role');
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            smsRole: role,
            smsRoleExternalId: 'test-external-id',
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            SmsConfiguration: {
                ExternalId: 'test-external-id',
                SnsCallerArn: role.roleArn,
            },
        });
    });
    test('import using id', () => {
        // GIVEN
        const stack = new core_1.Stack(undefined, undefined, {
            env: { region: 'some-region-1', account: '0123456789012' },
        });
        const userPoolId = 'test-user-pool';
        // WHEN
        const pool = lib_1.UserPool.fromUserPoolId(stack, 'userpool', userPoolId);
        expect(pool.userPoolId).toEqual(userPoolId);
        expect(pool.userPoolArn).toMatch(/cognito-idp:some-region-1:0123456789012:userpool\/test-user-pool/);
    });
    test('import using arn', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const userPoolArn = 'arn:aws:cognito-idp:us-east-1:0123456789012:userpool/test-user-pool';
        // WHEN
        const pool = lib_1.UserPool.fromUserPoolArn(stack, 'userpool', userPoolArn);
        expect(pool.userPoolId).toEqual('test-user-pool');
        expect(stack.resolve(pool.userPoolArn)).toEqual({
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':cognito-idp:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':userpool/test-user-pool',
                ]],
        });
    });
    test('support tags', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const pool = new lib_1.UserPool(stack, 'Pool', {
            userPoolName: 'myPool',
        });
        core_1.Tag.add(pool, 'PoolTag', 'PoolParty');
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'myPool',
            UserPoolTags: {
                PoolTag: 'PoolParty',
            },
        });
    });
    test('lambda triggers via properties are correctly configured', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const fn = fooFunction(stack, 'preSignUp');
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            lambdaTriggers: {
                preSignUp: fn,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            LambdaConfig: {
                PreSignUp: stack.resolve(fn.functionArn),
            },
        });
        expect(stack).toHaveResourceLike('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            FunctionName: stack.resolve(fn.functionArn),
            Principal: 'cognito-idp.amazonaws.com',
        });
    });
    test('add* API correctly appends triggers', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const createAuthChallenge = fooFunction(stack, 'createAuthChallenge');
        const customMessage = fooFunction(stack, 'customMessage');
        const defineAuthChallenge = fooFunction(stack, 'defineAuthChallenge');
        const postAuthentication = fooFunction(stack, 'postAuthentication');
        const postConfirmation = fooFunction(stack, 'postConfirmation');
        const preAuthentication = fooFunction(stack, 'preAuthentication');
        const preSignUp = fooFunction(stack, 'preSignUp');
        const preTokenGeneration = fooFunction(stack, 'preTokenGeneration');
        const userMigration = fooFunction(stack, 'userMigration');
        const verifyAuthChallengeResponse = fooFunction(stack, 'verifyAuthChallengeResponse');
        // WHEN
        const pool = new lib_1.UserPool(stack, 'Pool');
        pool.addTrigger(lib_1.UserPoolOperation.CREATE_AUTH_CHALLENGE, createAuthChallenge);
        pool.addTrigger(lib_1.UserPoolOperation.CUSTOM_MESSAGE, customMessage);
        pool.addTrigger(lib_1.UserPoolOperation.DEFINE_AUTH_CHALLENGE, defineAuthChallenge);
        pool.addTrigger(lib_1.UserPoolOperation.POST_AUTHENTICATION, postAuthentication);
        pool.addTrigger(lib_1.UserPoolOperation.POST_CONFIRMATION, postConfirmation);
        pool.addTrigger(lib_1.UserPoolOperation.PRE_AUTHENTICATION, preAuthentication);
        pool.addTrigger(lib_1.UserPoolOperation.PRE_SIGN_UP, preSignUp);
        pool.addTrigger(lib_1.UserPoolOperation.PRE_TOKEN_GENERATION, preTokenGeneration);
        pool.addTrigger(lib_1.UserPoolOperation.USER_MIGRATION, userMigration);
        pool.addTrigger(lib_1.UserPoolOperation.VERIFY_AUTH_CHALLENGE_RESPONSE, verifyAuthChallengeResponse);
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            LambdaConfig: {
                CreateAuthChallenge: stack.resolve(createAuthChallenge.functionArn),
                CustomMessage: stack.resolve(customMessage.functionArn),
                DefineAuthChallenge: stack.resolve(defineAuthChallenge.functionArn),
                PostAuthentication: stack.resolve(postAuthentication.functionArn),
                PostConfirmation: stack.resolve(postConfirmation.functionArn),
                PreAuthentication: stack.resolve(preAuthentication.functionArn),
                PreSignUp: stack.resolve(preSignUp.functionArn),
                PreTokenGeneration: stack.resolve(preTokenGeneration.functionArn),
                UserMigration: stack.resolve(userMigration.functionArn),
                VerifyAuthChallengeResponse: stack.resolve(verifyAuthChallengeResponse.functionArn),
            },
        });
        [createAuthChallenge, customMessage, defineAuthChallenge, postAuthentication,
            postConfirmation, preAuthentication, preSignUp, preTokenGeneration, userMigration,
            verifyAuthChallengeResponse].forEach((fn) => {
            expect(stack).toHaveResourceLike('AWS::Lambda::Permission', {
                Action: 'lambda:InvokeFunction',
                FunctionName: stack.resolve(fn.functionArn),
                Principal: 'cognito-idp.amazonaws.com',
            });
        });
    });
    test('fails when the same trigger is added twice', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const userpool = new lib_1.UserPool(stack, 'Pool');
        const fn1 = new lambda.Function(stack, 'fn1', {
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
        });
        const fn2 = new lambda.Function(stack, 'fn2', {
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
        });
        // WHEN
        userpool.addTrigger(lib_1.UserPoolOperation.CREATE_AUTH_CHALLENGE, fn1);
        expect(() => userpool.addTrigger(lib_1.UserPoolOperation.CREATE_AUTH_CHALLENGE, fn2)).toThrow(/already exists/);
    });
    test('no username aliases specified', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool');
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameAttributes: have_resource_1.ABSENT,
            AliasAttributes: have_resource_1.ABSENT,
        });
    });
    test('fails when preferredUsername is used without username', () => {
        const stack = new core_1.Stack();
        expect(() => new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { preferredUsername: true },
        })).toThrow(/username/);
    });
    test('username and email are specified as the username aliases', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { username: true, email: true },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameAttributes: have_resource_1.ABSENT,
            AliasAttributes: ['email'],
        });
    });
    test('email and phone number are specified as the username aliases', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { email: true, phone: true },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameAttributes: ['email', 'phone_number'],
            AliasAttributes: have_resource_1.ABSENT,
        });
    });
    test('email and phone number are autoverified, by default, if they are specified as signIn', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool1', {
            userPoolName: 'Pool1',
            signInAliases: { email: true },
        });
        new lib_1.UserPool(stack, 'Pool2', {
            userPoolName: 'Pool2',
            signInAliases: { email: true, phone: true },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool1',
            AutoVerifiedAttributes: ['email'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool2',
            AutoVerifiedAttributes: ['email', 'phone_number'],
        });
    });
    test('explicit autoverify are correctly picked up', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { username: true },
            autoVerify: { email: true, phone: true },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            AutoVerifiedAttributes: ['email', 'phone_number'],
        });
    });
    test('sign in case sensitive is correctly picked up', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInCaseSensitive: false,
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameConfiguration: {
                CaseSensitive: false,
            },
        });
    });
    test('sign in case sensitive is absent by default', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {});
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameConfiguration: have_resource_1.ABSENT,
        });
    });
    test('standard attributes default to mutable', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            standardAttributes: {
                fullname: {
                    required: true,
                },
                timezone: {
                    required: true,
                },
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Schema: [
                {
                    Name: 'name',
                    Required: true,
                    Mutable: true,
                },
                {
                    Name: 'zoneinfo',
                    Required: true,
                    Mutable: true,
                },
            ],
        });
    });
    test('mutable standard attributes', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            userPoolName: 'Pool',
            standardAttributes: {
                fullname: {
                    required: true,
                    mutable: true,
                },
                timezone: {
                    required: true,
                    mutable: true,
                },
            },
        });
        new lib_1.UserPool(stack, 'Pool1', {
            userPoolName: 'Pool1',
            standardAttributes: {
                fullname: {
                    mutable: false,
                },
                timezone: {
                    mutable: false,
                },
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool',
            Schema: [
                {
                    Mutable: true,
                    Name: 'name',
                    Required: true,
                },
                {
                    Mutable: true,
                    Name: 'zoneinfo',
                    Required: true,
                },
            ],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool1',
            Schema: [
                {
                    Name: 'name',
                    Required: false,
                    Mutable: false,
                },
                {
                    Name: 'zoneinfo',
                    Required: false,
                    Mutable: false,
                },
            ],
        });
    });
    test('schema is absent when attributes are not specified', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', { userPoolName: 'Pool' });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool',
            Schema: have_resource_1.ABSENT,
        });
    });
    test('optional mutable standardAttributes', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            userPoolName: 'Pool',
            standardAttributes: {
                timezone: {
                    mutable: true,
                },
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool',
            Schema: [
                {
                    Mutable: true,
                    Required: false,
                    Name: 'zoneinfo',
                },
            ],
        });
    });
    test('custom attributes with default constraints', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            customAttributes: {
                'custom-string-attr': new lib_1.StringAttribute(),
                'custom-number-attr': new lib_1.NumberAttribute(),
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Schema: [
                {
                    Name: 'custom-string-attr',
                    AttributeDataType: 'String',
                    StringAttributeConstraints: have_resource_1.ABSENT,
                    NumberAttributeConstraints: have_resource_1.ABSENT,
                },
                {
                    Name: 'custom-number-attr',
                    AttributeDataType: 'Number',
                    StringAttributeConstraints: have_resource_1.ABSENT,
                    NumberAttributeConstraints: have_resource_1.ABSENT,
                },
            ],
        });
    });
    test('custom attributes with constraints', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            customAttributes: {
                'custom-string-attr': new lib_1.StringAttribute({ minLen: 5, maxLen: 50 }),
                'custom-number-attr': new lib_1.NumberAttribute({ min: 500, max: 2000 }),
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Schema: [
                {
                    AttributeDataType: 'String',
                    Name: 'custom-string-attr',
                    StringAttributeConstraints: {
                        MaxLength: '50',
                        MinLength: '5',
                    },
                },
                {
                    AttributeDataType: 'Number',
                    Name: 'custom-number-attr',
                    NumberAttributeConstraints: {
                        MaxValue: '2000',
                        MinValue: '500',
                    },
                },
            ],
        });
    });
    test('mfaTypes is ignored when mfaEnforcement is undefined or set to OFF', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool1', {
            userPoolName: 'Pool1',
            mfaSecondFactor: {
                sms: true,
                otp: true,
            },
        });
        new lib_1.UserPool(stack, 'Pool2', {
            userPoolName: 'Pool2',
            mfa: lib_1.Mfa.OFF,
            mfaSecondFactor: {
                sms: true,
                otp: true,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool1',
            MfaConfiguration: have_resource_1.ABSENT,
            EnabledMfas: have_resource_1.ABSENT,
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool2',
            MfaConfiguration: 'OFF',
            EnabledMfas: have_resource_1.ABSENT,
        });
    });
    test('sms mfa type is the default when mfaEnforcement is set to REQUIRED or OPTIONAL', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool1', {
            userPoolName: 'Pool1',
            mfa: lib_1.Mfa.OPTIONAL,
        });
        new lib_1.UserPool(stack, 'Pool2', {
            userPoolName: 'Pool2',
            mfa: lib_1.Mfa.REQUIRED,
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool1',
            MfaConfiguration: 'OPTIONAL',
            EnabledMfas: ['SMS_MFA'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool2',
            MfaConfiguration: 'ON',
            EnabledMfas: ['SMS_MFA'],
        });
    });
    test('mfa type is correctly picked up when specified', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            mfa: lib_1.Mfa.REQUIRED,
            mfaSecondFactor: {
                sms: true,
                otp: true,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            EnabledMfas: ['SMS_MFA', 'SOFTWARE_TOKEN_MFA'],
        });
    });
    test('password policy is correctly set', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            passwordPolicy: {
                tempPasswordValidity: core_1.Duration.days(2),
                minLength: 15,
                requireDigits: true,
                requireLowercase: true,
                requireUppercase: true,
                requireSymbols: true,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Policies: {
                PasswordPolicy: {
                    TemporaryPasswordValidityDays: 2,
                    MinimumLength: 15,
                    RequireLowercase: true,
                    RequireUppercase: true,
                    RequireNumbers: true,
                    RequireSymbols: true,
                },
            },
        });
    });
    test('password minimum length is set to the default when other parts of the policy is configured', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            passwordPolicy: {
                tempPasswordValidity: core_1.Duration.days(2),
                requireDigits: true,
            },
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Policies: {
                PasswordPolicy: {
                    MinimumLength: 8,
                },
            },
        });
    });
    test('throws when tempPassword validity is not in round days', () => {
        const stack = new core_1.Stack();
        expect(() => new lib_1.UserPool(stack, 'Pool', {
            passwordPolicy: {
                tempPasswordValidity: core_1.Duration.hours(30),
            },
        })).toThrow();
    });
    test('temp password throws an error when above the max', () => {
        const stack = new core_1.Stack();
        expect(() => new lib_1.UserPool(stack, 'Pool', {
            passwordPolicy: {
                tempPasswordValidity: core_1.Duration.days(400),
            },
        })).toThrow(/tempPasswordValidity cannot be greater than/);
    });
    test('throws when minLength is out of range', () => {
        const stack = new core_1.Stack();
        expect(() => new lib_1.UserPool(stack, 'Pool1', {
            passwordPolicy: {
                minLength: 5,
            },
        })).toThrow(/minLength for password must be between/);
        expect(() => new lib_1.UserPool(stack, 'Pool2', {
            passwordPolicy: {
                minLength: 100,
            },
        })).toThrow(/minLength for password must be between/);
    });
    test('email transmission settings are recognized correctly', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            emailSettings: {
                from: 'from@myawesomeapp.com',
                replyTo: 'replyTo@myawesomeapp.com',
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            EmailConfiguration: {
                From: 'from@myawesomeapp.com',
                ReplyToEmailAddress: 'replyTo@myawesomeapp.com',
            },
        });
    });
});
test('addClient', () => {
    // GIVEN
    const stack = new core_1.Stack();
    // WHEN
    const userpool = new lib_1.UserPool(stack, 'Pool');
    userpool.addClient('UserPoolClient', {
        userPoolClientName: 'userpoolclient',
    });
    const imported = lib_1.UserPool.fromUserPoolId(stack, 'imported', 'imported-userpool-id');
    imported.addClient('UserPoolImportedClient', {
        userPoolClientName: 'userpoolimportedclient',
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
        ClientName: 'userpoolclient',
        UserPoolId: stack.resolve(userpool.userPoolId),
    });
    expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
        ClientName: 'userpoolimportedclient',
        UserPoolId: stack.resolve(imported.userPoolId),
    });
});
test('addDomain', () => {
    // GIVEN
    const stack = new core_1.Stack();
    // WHEN
    const userpool = new lib_1.UserPool(stack, 'Pool');
    userpool.addDomain('UserPoolDomain', {
        cognitoDomain: {
            domainPrefix: 'userpooldomain',
        },
    });
    const imported = lib_1.UserPool.fromUserPoolId(stack, 'imported', 'imported-userpool-id');
    imported.addDomain('UserPoolImportedDomain', {
        cognitoDomain: {
            domainPrefix: 'userpoolimporteddomain',
        },
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolDomain', {
        Domain: 'userpooldomain',
        UserPoolId: stack.resolve(userpool.userPoolId),
    });
    expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolDomain', {
        Domain: 'userpoolimporteddomain',
        UserPoolId: stack.resolve(imported.userPoolId),
    });
});
test('registered identity providers', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const userPool = new lib_1.UserPool(stack, 'pool');
    const provider1 = lib_1.UserPoolIdentityProvider.fromProviderName(stack, 'provider1', 'provider1');
    const provider2 = lib_1.UserPoolIdentityProvider.fromProviderName(stack, 'provider2', 'provider2');
    // WHEN
    userPool.registerIdentityProvider(provider1);
    userPool.registerIdentityProvider(provider2);
    // THEN
    expect(userPool.identityProviders).toEqual([provider1, provider2]);
});
function fooFunction(scope, name) {
    return new lambda.Function(scope, name, {
        functionName: name,
        code: lambda.Code.inline('foo'),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
    });
}
describe('AccountRecoverySetting should be configured correctly', () => {
    test('EMAIL_AND_PHONE_WITHOUT_MFA', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'pool', { accountRecovery: lib_1.AccountRecovery.EMAIL_AND_PHONE_WITHOUT_MFA });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPool', {
            AccountRecoverySetting: {
                RecoveryMechanisms: [
                    { Name: 'verified_email', Priority: 1 },
                    { Name: 'verified_phone_number', Priority: 2 },
                ],
            },
        });
    });
    test('PHONE_WITHOUT_MFA_AND_EMAIL', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'pool', { accountRecovery: lib_1.AccountRecovery.PHONE_WITHOUT_MFA_AND_EMAIL });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPool', {
            AccountRecoverySetting: {
                RecoveryMechanisms: [
                    { Name: 'verified_phone_number', Priority: 1 },
                    { Name: 'verified_email', Priority: 2 },
                ],
            },
        });
    });
    test('EMAIL_ONLY', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'pool', { accountRecovery: lib_1.AccountRecovery.EMAIL_ONLY });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPool', {
            AccountRecoverySetting: {
                RecoveryMechanisms: [
                    { Name: 'verified_email', Priority: 1 },
                ],
            },
        });
    });
    test('PHONE_ONLY_WITHOUT_MFA', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'pool', { accountRecovery: lib_1.AccountRecovery.PHONE_ONLY_WITHOUT_MFA });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPool', {
            AccountRecoverySetting: {
                RecoveryMechanisms: [
                    { Name: 'verified_phone_number', Priority: 1 },
                ],
            },
        });
    });
    test('NONE', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'pool', { accountRecovery: lib_1.AccountRecovery.NONE });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPool', {
            AccountRecoverySetting: {
                RecoveryMechanisms: [
                    { Name: 'admin_only', Priority: 1 },
                ],
            },
        });
    });
    test('PHONE_AND_EMAIL', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'pool', { accountRecovery: lib_1.AccountRecovery.PHONE_AND_EMAIL });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPool', {
            AccountRecoverySetting: have_resource_1.ABSENT,
        });
    });
    test('default', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'pool');
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPool', {
            AccountRecoverySetting: {
                RecoveryMechanisms: [
                    { Name: 'verified_phone_number', Priority: 1 },
                    { Name: 'verified_email', Priority: 2 },
                ],
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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