"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MicroAppsChildDeployer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const lambdaNodejs = require("aws-cdk-lib/aws-lambda-nodejs");
const logs = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
/**
 * Create a new MicroApps Child Deployer construct.
 */
class MicroAppsChildDeployer extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props === undefined) {
            throw new Error('props cannot be undefined');
        }
        const { appEnv, deployerTimeout = aws_cdk_lib_1.Duration.minutes(2), assetNameRoot, assetNameSuffix, removalPolicy, parentDeployerLambdaARN, edgeToOriginRoleARN, } = props;
        //
        // Deployer Lambda Function
        //
        const iamRoleDeployerName = assetNameRoot
            ? `${assetNameRoot}-deployer${assetNameSuffix}`
            : undefined;
        const iamRoleDeployer = new iam.Role(this, 'deployer-role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            roleName: iamRoleDeployerName,
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
            inlinePolicies: {
                deployPolicy: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            effect: iam.Effect.ALLOW,
                            actions: ['lambda:InvokeFunction'],
                            resources: [`${parentDeployerLambdaARN}:currentVersion`],
                        }),
                    ],
                }),
            },
        });
        // Create Deployer Lambda Function
        const deployerFuncName = assetNameRoot
            ? `${assetNameRoot}-deployer${assetNameSuffix}`
            : undefined;
        const deployerFuncProps = {
            functionName: deployerFuncName,
            role: iamRoleDeployer,
            memorySize: 1769,
            logRetention: logs.RetentionDays.ONE_MONTH,
            runtime: lambda.Runtime.NODEJS_16_X,
            timeout: deployerTimeout,
            environment: {
                NODE_ENV: appEnv,
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                PARENT_DEPLOYER_LAMBDA_ARN: parentDeployerLambdaARN,
                EDGE_TO_ORIGIN_ROLE_ARN: edgeToOriginRoleARN,
            },
        };
        if (process.env.NODE_ENV === 'test' &&
            fs_1.existsSync(path.join(__dirname, '..', '..', 'microapps-deployer', 'dist', 'index.js'))) {
            // This is for local dev
            this._deployerFunc = new lambda.Function(this, 'deployer-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', 'microapps-deployer', 'dist')),
                handler: 'index.handler',
                ...deployerFuncProps,
            });
        }
        else if (fs_1.existsSync(path.join(__dirname, 'microapps-deployer', 'index.js'))) {
            // This is for built apps packaged with the CDK construct
            this._deployerFunc = new lambda.Function(this, 'deployer-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, 'microapps-deployer')),
                handler: 'index.handler',
                ...deployerFuncProps,
            });
        }
        else {
            this._deployerFunc = new lambdaNodejs.NodejsFunction(this, 'deployer-func', {
                entry: path.join(__dirname, '..', '..', 'microapps-deployer', 'src', 'index.ts'),
                handler: 'handler',
                bundling: {
                    minify: true,
                    sourceMap: true,
                },
                ...deployerFuncProps,
            });
        }
        if (removalPolicy !== undefined) {
            this._deployerFunc.applyRemovalPolicy(removalPolicy);
        }
        // Grant full control over lambdas that indicate they are microapps
        const policyAPIManageLambdas = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['lambda:*'],
            resources: [
                `arn:aws:lambda:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:function:*`,
                `arn:aws:lambda:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:function:*:*`,
            ],
            conditions: {
                StringEquals: { 'aws:ResourceTag/microapp-managed': 'true' },
            },
        });
        this._deployerFunc.addToRolePolicy(policyAPIManageLambdas);
    }
    get deployerFunc() {
        return this._deployerFunc;
    }
}
exports.MicroAppsChildDeployer = MicroAppsChildDeployer;
_a = JSII_RTTI_SYMBOL_1;
MicroAppsChildDeployer[_a] = { fqn: "@pwrdrvr/microapps-cdk.MicroAppsChildDeployer", version: "0.4.0-alpha.3" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiTWljcm9BcHBzQ2hpbGREZXBsb3llci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9NaWNyb0FwcHNDaGlsZERlcGxveWVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsMkJBQWdDO0FBQ2hDLDZCQUE2QjtBQUM3Qiw2Q0FBMkQ7QUFDM0QsMkNBQTJDO0FBQzNDLGlEQUFpRDtBQUNqRCw4REFBOEQ7QUFDOUQsNkNBQTZDO0FBQzdDLDJDQUF1QztBQW1FdkM7O0dBRUc7QUFDSCxNQUFhLHNCQUF1QixTQUFRLHNCQUFTO0lBTW5ELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBbUM7UUFDM0UsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLEtBQUssS0FBSyxTQUFTLEVBQUU7WUFDdkIsTUFBTSxJQUFJLEtBQUssQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1NBQzlDO1FBRUQsTUFBTSxFQUNKLE1BQU0sRUFDTixlQUFlLEdBQUcsc0JBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQ3JDLGFBQWEsRUFDYixlQUFlLEVBQ2YsYUFBYSxFQUNiLHVCQUF1QixFQUN2QixtQkFBbUIsR0FDcEIsR0FBRyxLQUFLLENBQUM7UUFFVixFQUFFO1FBQ0YsMkJBQTJCO1FBQzNCLEVBQUU7UUFFRixNQUFNLG1CQUFtQixHQUFHLGFBQWE7WUFDdkMsQ0FBQyxDQUFDLEdBQUcsYUFBYSxZQUFZLGVBQWUsRUFBRTtZQUMvQyxDQUFDLENBQUMsU0FBUyxDQUFDO1FBQ2QsTUFBTSxlQUFlLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxlQUFlLEVBQUU7WUFDMUQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO1lBQzNELFFBQVEsRUFBRSxtQkFBbUI7WUFDN0IsZUFBZSxFQUFFO2dCQUNmLEdBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsMENBQTBDLENBQUM7YUFDdkY7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsWUFBWSxFQUFFLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQztvQkFDbkMsVUFBVSxFQUFFO3dCQUNWLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQzs0QkFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSzs0QkFDeEIsT0FBTyxFQUFFLENBQUMsdUJBQXVCLENBQUM7NEJBQ2xDLFNBQVMsRUFBRSxDQUFDLEdBQUcsdUJBQXVCLGlCQUFpQixDQUFDO3lCQUN6RCxDQUFDO3FCQUNIO2lCQUNGLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQztRQUVILGtDQUFrQztRQUNsQyxNQUFNLGdCQUFnQixHQUFHLGFBQWE7WUFDcEMsQ0FBQyxDQUFDLEdBQUcsYUFBYSxZQUFZLGVBQWUsRUFBRTtZQUMvQyxDQUFDLENBQUMsU0FBUyxDQUFDO1FBQ2QsTUFBTSxpQkFBaUIsR0FBbUQ7WUFDeEUsWUFBWSxFQUFFLGdCQUFnQjtZQUM5QixJQUFJLEVBQUUsZUFBZTtZQUNyQixVQUFVLEVBQUUsSUFBSTtZQUNoQixZQUFZLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxTQUFTO1lBQzFDLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7WUFDbkMsT0FBTyxFQUFFLGVBQWU7WUFDeEIsV0FBVyxFQUFFO2dCQUNYLFFBQVEsRUFBRSxNQUFNO2dCQUNoQixtQ0FBbUMsRUFBRSxHQUFHO2dCQUN4QywwQkFBMEIsRUFBRSx1QkFBdUI7Z0JBQ25ELHVCQUF1QixFQUFFLG1CQUFtQjthQUM3QztTQUNGLENBQUM7UUFDRixJQUNFLE9BQU8sQ0FBQyxHQUFHLENBQUMsUUFBUSxLQUFLLE1BQU07WUFDL0IsZUFBVSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsb0JBQW9CLEVBQUUsTUFBTSxFQUFFLFVBQVUsQ0FBQyxDQUFDLEVBQ3RGO1lBQ0Esd0JBQXdCO1lBQ3hCLElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxlQUFlLEVBQUU7Z0JBQzlELElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLG9CQUFvQixFQUFFLE1BQU0sQ0FBQyxDQUFDO2dCQUMzRixPQUFPLEVBQUUsZUFBZTtnQkFDeEIsR0FBRyxpQkFBaUI7YUFDckIsQ0FBQyxDQUFDO1NBQ0o7YUFBTSxJQUFJLGVBQVUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxvQkFBb0IsRUFBRSxVQUFVLENBQUMsQ0FBQyxFQUFFO1lBQzdFLHlEQUF5RDtZQUN6RCxJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFO2dCQUM5RCxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsb0JBQW9CLENBQUMsQ0FBQztnQkFDdkUsT0FBTyxFQUFFLGVBQWU7Z0JBQ3hCLEdBQUcsaUJBQWlCO2FBQ3JCLENBQUMsQ0FBQztTQUNKO2FBQU07WUFDTCxJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksWUFBWSxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFO2dCQUMxRSxLQUFLLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxvQkFBb0IsRUFBRSxLQUFLLEVBQUUsVUFBVSxDQUFDO2dCQUNoRixPQUFPLEVBQUUsU0FBUztnQkFDbEIsUUFBUSxFQUFFO29CQUNSLE1BQU0sRUFBRSxJQUFJO29CQUNaLFNBQVMsRUFBRSxJQUFJO2lCQUNoQjtnQkFDRCxHQUFHLGlCQUFpQjthQUNyQixDQUFDLENBQUM7U0FDSjtRQUNELElBQUksYUFBYSxLQUFLLFNBQVMsRUFBRTtZQUMvQixJQUFJLENBQUMsYUFBYSxDQUFDLGtCQUFrQixDQUFDLGFBQWEsQ0FBQyxDQUFDO1NBQ3REO1FBRUQsbUVBQW1FO1FBQ25FLE1BQU0sc0JBQXNCLEdBQUcsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3JELE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsT0FBTyxFQUFFLENBQUMsVUFBVSxDQUFDO1lBQ3JCLFNBQVMsRUFBRTtnQkFDVCxrQkFBa0IsaUJBQUcsQ0FBQyxNQUFNLElBQUksaUJBQUcsQ0FBQyxVQUFVLGFBQWE7Z0JBQzNELGtCQUFrQixpQkFBRyxDQUFDLE1BQU0sSUFBSSxpQkFBRyxDQUFDLFVBQVUsZUFBZTthQUM5RDtZQUNELFVBQVUsRUFBRTtnQkFDVixZQUFZLEVBQUUsRUFBRSxrQ0FBa0MsRUFBRSxNQUFNLEVBQUU7YUFDN0Q7U0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsYUFBYSxDQUFDLGVBQWUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQzdELENBQUM7SUE5R0QsSUFBVyxZQUFZO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLGFBQWEsQ0FBQztJQUM1QixDQUFDOztBQUpILHdEQWlIQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGV4aXN0c1N5bmMgfSBmcm9tICdmcyc7XG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0IHsgQXdzLCBEdXJhdGlvbiwgUmVtb3ZhbFBvbGljeSB9IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIGxhbWJkYU5vZGVqcyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbGFtYmRhLW5vZGVqcyc7XG5pbXBvcnQgKiBhcyBsb2dzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1sb2dzJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuXG4vKipcbiAqIFByb3BlcnRpZXMgdG8gaW5pdGlhbGl6ZSBhbiBpbnN0YW5jZSBvZiBgTWljcm9BcHBzQ2hpbGREZXBsb3llcmAuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTWljcm9BcHBzQ2hpbGREZXBsb3llclByb3BzIHtcbiAgLyoqXG4gICAqIEFSTiBvZiB0aGUgcGFyZW50IERlcGxveWVyIExhbWJkYSBGdW5jdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgcGFyZW50RGVwbG95ZXJMYW1iZGFBUk46IHN0cmluZztcblxuICAvKipcbiAgICogQVJOIG9mIHRoZSBJQU0gUm9sZSBmb3IgdGhlIEVkZ2UgdG8gT3JpZ2luIExhbWJkYSBGdW5jdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgZWRnZVRvT3JpZ2luUm9sZUFSTjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBSZW1vdmFsUG9saWN5IG92ZXJyaWRlIGZvciBjaGlsZCByZXNvdXJjZXNcbiAgICpcbiAgICogTm90ZTogaWYgc2V0IHRvIERFU1RST1kgdGhlIFMzIGJ1Y2tlcyB3aWxsIGhhdmUgYGF1dG9EZWxldGVPYmplY3RzYCBzZXQgdG8gYHRydWVgXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gcGVyIHJlc291cmNlIGRlZmF1bHRcbiAgICovXG4gIHJlYWRvbmx5IHJlbW92YWxQb2xpY3k/OiBSZW1vdmFsUG9saWN5O1xuXG4gIC8qKlxuICAgKiBBcHBsaWNhdGlvbiBlbnZpcm9ubWVudCwgcGFzc2VkIGFzIGBOT0RFX0VOVmBcbiAgICogdG8gdGhlIFJvdXRlciBhbmQgRGVwbG95ZXIgTGFtYmRhIGZ1bmN0aW9uc1xuICAgKi9cbiAgcmVhZG9ubHkgYXBwRW52OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIE9wdGlvbmFsIGFzc2V0IG5hbWUgcm9vdFxuICAgKlxuICAgKiBAZXhhbXBsZSBtaWNyb2FwcHNcbiAgICogQGRlZmF1bHQgLSByZXNvdXJjZSBuYW1lcyBhdXRvIGFzc2lnbmVkXG4gICAqL1xuICByZWFkb25seSBhc3NldE5hbWVSb290Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbCBhc3NldCBuYW1lIHN1ZmZpeFxuICAgKlxuICAgKiBAZXhhbXBsZSAtZGV2LXByLTEyXG4gICAqIEBkZWZhdWx0IG5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGFzc2V0TmFtZVN1ZmZpeD86IHN0cmluZztcblxuICAvKipcbiAgICogRGVwbG95ZXIgdGltZW91dFxuICAgKlxuICAgKiBGb3IgbGFyZ2VyIGFwcGxpY2F0aW9ucyB0aGlzIG5lZWRzIHRvIGJlIHNldCB1cCB0byAyLTUgbWludXRlcyBmb3IgdGhlIFMzIGNvcHlcbiAgICpcbiAgICogQGRlZmF1bHQgMiBtaW51dGVzXG4gICAqL1xuICByZWFkb25seSBkZXBsb3llclRpbWVvdXQ/OiBEdXJhdGlvbjtcbn1cblxuLyoqXG4gKiBSZXByZXNlbnRzIGEgTWljcm9BcHBzIENoaWxkIERlcGxveWVyXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSU1pY3JvQXBwc0NoaWxkRGVwbG95ZXIge1xuICAvKipcbiAgICogTGFtYmRhIGZ1bmN0aW9uIGZvciB0aGUgRGVwbG95ZXJcbiAgICovXG4gIHJlYWRvbmx5IGRlcGxveWVyRnVuYzogbGFtYmRhLklGdW5jdGlvbjtcbn1cblxuLyoqXG4gKiBDcmVhdGUgYSBuZXcgTWljcm9BcHBzIENoaWxkIERlcGxveWVyIGNvbnN0cnVjdC5cbiAqL1xuZXhwb3J0IGNsYXNzIE1pY3JvQXBwc0NoaWxkRGVwbG95ZXIgZXh0ZW5kcyBDb25zdHJ1Y3QgaW1wbGVtZW50cyBJTWljcm9BcHBzQ2hpbGREZXBsb3llciB7XG4gIHByaXZhdGUgX2RlcGxveWVyRnVuYzogbGFtYmRhLkZ1bmN0aW9uO1xuICBwdWJsaWMgZ2V0IGRlcGxveWVyRnVuYygpOiBsYW1iZGEuSUZ1bmN0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5fZGVwbG95ZXJGdW5jO1xuICB9XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM/OiBNaWNyb0FwcHNDaGlsZERlcGxveWVyUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgaWYgKHByb3BzID09PSB1bmRlZmluZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcigncHJvcHMgY2Fubm90IGJlIHVuZGVmaW5lZCcpO1xuICAgIH1cblxuICAgIGNvbnN0IHtcbiAgICAgIGFwcEVudixcbiAgICAgIGRlcGxveWVyVGltZW91dCA9IER1cmF0aW9uLm1pbnV0ZXMoMiksXG4gICAgICBhc3NldE5hbWVSb290LFxuICAgICAgYXNzZXROYW1lU3VmZml4LFxuICAgICAgcmVtb3ZhbFBvbGljeSxcbiAgICAgIHBhcmVudERlcGxveWVyTGFtYmRhQVJOLFxuICAgICAgZWRnZVRvT3JpZ2luUm9sZUFSTixcbiAgICB9ID0gcHJvcHM7XG5cbiAgICAvL1xuICAgIC8vIERlcGxveWVyIExhbWJkYSBGdW5jdGlvblxuICAgIC8vXG5cbiAgICBjb25zdCBpYW1Sb2xlRGVwbG95ZXJOYW1lID0gYXNzZXROYW1lUm9vdFxuICAgICAgPyBgJHthc3NldE5hbWVSb290fS1kZXBsb3llciR7YXNzZXROYW1lU3VmZml4fWBcbiAgICAgIDogdW5kZWZpbmVkO1xuICAgIGNvbnN0IGlhbVJvbGVEZXBsb3llciA9IG5ldyBpYW0uUm9sZSh0aGlzLCAnZGVwbG95ZXItcm9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdsYW1iZGEuYW1hem9uYXdzLmNvbScpLFxuICAgICAgcm9sZU5hbWU6IGlhbVJvbGVEZXBsb3llck5hbWUsXG4gICAgICBtYW5hZ2VkUG9saWNpZXM6IFtcbiAgICAgICAgaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdzZXJ2aWNlLXJvbGUvQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlJyksXG4gICAgICBdLFxuICAgICAgaW5saW5lUG9saWNpZXM6IHtcbiAgICAgICAgZGVwbG95UG9saWN5OiBuZXcgaWFtLlBvbGljeURvY3VtZW50KHtcbiAgICAgICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgICAgICAgYWN0aW9uczogWydsYW1iZGE6SW52b2tlRnVuY3Rpb24nXSxcbiAgICAgICAgICAgICAgcmVzb3VyY2VzOiBbYCR7cGFyZW50RGVwbG95ZXJMYW1iZGFBUk59OmN1cnJlbnRWZXJzaW9uYF0sXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBDcmVhdGUgRGVwbG95ZXIgTGFtYmRhIEZ1bmN0aW9uXG4gICAgY29uc3QgZGVwbG95ZXJGdW5jTmFtZSA9IGFzc2V0TmFtZVJvb3RcbiAgICAgID8gYCR7YXNzZXROYW1lUm9vdH0tZGVwbG95ZXIke2Fzc2V0TmFtZVN1ZmZpeH1gXG4gICAgICA6IHVuZGVmaW5lZDtcbiAgICBjb25zdCBkZXBsb3llckZ1bmNQcm9wczogT21pdDxsYW1iZGEuRnVuY3Rpb25Qcm9wcywgJ2hhbmRsZXInIHwgJ2NvZGUnPiA9IHtcbiAgICAgIGZ1bmN0aW9uTmFtZTogZGVwbG95ZXJGdW5jTmFtZSxcbiAgICAgIHJvbGU6IGlhbVJvbGVEZXBsb3llcixcbiAgICAgIG1lbW9yeVNpemU6IDE3NjksXG4gICAgICBsb2dSZXRlbnRpb246IGxvZ3MuUmV0ZW50aW9uRGF5cy5PTkVfTU9OVEgsXG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTZfWCxcbiAgICAgIHRpbWVvdXQ6IGRlcGxveWVyVGltZW91dCxcbiAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgIE5PREVfRU5WOiBhcHBFbnYsXG4gICAgICAgIEFXU19OT0RFSlNfQ09OTkVDVElPTl9SRVVTRV9FTkFCTEVEOiAnMScsXG4gICAgICAgIFBBUkVOVF9ERVBMT1lFUl9MQU1CREFfQVJOOiBwYXJlbnREZXBsb3llckxhbWJkYUFSTixcbiAgICAgICAgRURHRV9UT19PUklHSU5fUk9MRV9BUk46IGVkZ2VUb09yaWdpblJvbGVBUk4sXG4gICAgICB9LFxuICAgIH07XG4gICAgaWYgKFxuICAgICAgcHJvY2Vzcy5lbnYuTk9ERV9FTlYgPT09ICd0ZXN0JyAmJlxuICAgICAgZXhpc3RzU3luYyhwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4nLCAnLi4nLCAnbWljcm9hcHBzLWRlcGxveWVyJywgJ2Rpc3QnLCAnaW5kZXguanMnKSlcbiAgICApIHtcbiAgICAgIC8vIFRoaXMgaXMgZm9yIGxvY2FsIGRldlxuICAgICAgdGhpcy5fZGVwbG95ZXJGdW5jID0gbmV3IGxhbWJkYS5GdW5jdGlvbih0aGlzLCAnZGVwbG95ZXItZnVuYycsIHtcbiAgICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICcuLicsICcuLicsICdtaWNyb2FwcHMtZGVwbG95ZXInLCAnZGlzdCcpKSxcbiAgICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgICAuLi5kZXBsb3llckZ1bmNQcm9wcyxcbiAgICAgIH0pO1xuICAgIH0gZWxzZSBpZiAoZXhpc3RzU3luYyhwYXRoLmpvaW4oX19kaXJuYW1lLCAnbWljcm9hcHBzLWRlcGxveWVyJywgJ2luZGV4LmpzJykpKSB7XG4gICAgICAvLyBUaGlzIGlzIGZvciBidWlsdCBhcHBzIHBhY2thZ2VkIHdpdGggdGhlIENESyBjb25zdHJ1Y3RcbiAgICAgIHRoaXMuX2RlcGxveWVyRnVuYyA9IG5ldyBsYW1iZGEuRnVuY3Rpb24odGhpcywgJ2RlcGxveWVyLWZ1bmMnLCB7XG4gICAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnbWljcm9hcHBzLWRlcGxveWVyJykpLFxuICAgICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICAgIC4uLmRlcGxveWVyRnVuY1Byb3BzLFxuICAgICAgfSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMuX2RlcGxveWVyRnVuYyA9IG5ldyBsYW1iZGFOb2RlanMuTm9kZWpzRnVuY3Rpb24odGhpcywgJ2RlcGxveWVyLWZ1bmMnLCB7XG4gICAgICAgIGVudHJ5OiBwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4nLCAnLi4nLCAnbWljcm9hcHBzLWRlcGxveWVyJywgJ3NyYycsICdpbmRleC50cycpLFxuICAgICAgICBoYW5kbGVyOiAnaGFuZGxlcicsXG4gICAgICAgIGJ1bmRsaW5nOiB7XG4gICAgICAgICAgbWluaWZ5OiB0cnVlLFxuICAgICAgICAgIHNvdXJjZU1hcDogdHJ1ZSxcbiAgICAgICAgfSxcbiAgICAgICAgLi4uZGVwbG95ZXJGdW5jUHJvcHMsXG4gICAgICB9KTtcbiAgICB9XG4gICAgaWYgKHJlbW92YWxQb2xpY3kgIT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhpcy5fZGVwbG95ZXJGdW5jLmFwcGx5UmVtb3ZhbFBvbGljeShyZW1vdmFsUG9saWN5KTtcbiAgICB9XG5cbiAgICAvLyBHcmFudCBmdWxsIGNvbnRyb2wgb3ZlciBsYW1iZGFzIHRoYXQgaW5kaWNhdGUgdGhleSBhcmUgbWljcm9hcHBzXG4gICAgY29uc3QgcG9saWN5QVBJTWFuYWdlTGFtYmRhcyA9IG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgIGFjdGlvbnM6IFsnbGFtYmRhOionXSxcbiAgICAgIHJlc291cmNlczogW1xuICAgICAgICBgYXJuOmF3czpsYW1iZGE6JHtBd3MuUkVHSU9OfToke0F3cy5BQ0NPVU5UX0lEfTpmdW5jdGlvbjoqYCxcbiAgICAgICAgYGFybjphd3M6bGFtYmRhOiR7QXdzLlJFR0lPTn06JHtBd3MuQUNDT1VOVF9JRH06ZnVuY3Rpb246KjoqYCxcbiAgICAgIF0sXG4gICAgICBjb25kaXRpb25zOiB7XG4gICAgICAgIFN0cmluZ0VxdWFsczogeyAnYXdzOlJlc291cmNlVGFnL21pY3JvYXBwLW1hbmFnZWQnOiAndHJ1ZScgfSxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgdGhpcy5fZGVwbG95ZXJGdW5jLmFkZFRvUm9sZVBvbGljeShwb2xpY3lBUElNYW5hZ2VMYW1iZGFzKTtcbiAgfVxufVxuIl19