# **************************************************************************
# *
# * Authors:    David Herreros Calero (dherreros@cnb.csic.es)
# *             Scipion Team (scipion@cnb.csic.es)
# *
# *  BCU, Centro Nacional de Biotecnologia, CSIC
# *
# * This program is free software you can redistribute it and/or modify
# * it under the terms of the GNU General Public License as published by
# * the Free Software Foundation either version 2 of the License, or
# * (at your option) any later version.
# *
# * This program is distributed in the hope that it will be useful,
# * but WITHOUT ANY WARRANTY without even the implied warranty of
# * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# * GNU General Public License for more details.
# *
# * You should have received a copy of the GNU General Public License
# * along with this program if not, write to the Free Software
# * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
# * 02111-1307  USA
# *
# *  All comments concerning this program package may be sent to the
# *  e-mail address 'scipion@cnb.csic.es'
# *
# **************************************************************************
import datetime
from enum import Enum
from os.path import abspath
from dynamo.utils import getCatalogFile, createBoxingOutputObjects, getDynamoModels, getNewestModelModDate
from dynamo.viewers.DynamoTomoProvider import DynamoTomogramProvider
import pyworkflow.utils as pwutils
from pyworkflow import BETA
from pyworkflow.protocol import LEVEL_ADVANCED
from pyworkflow.protocol.params import PointerParam, IntParam, BooleanParam
from pyworkflow.utils.properties import Message
from pyworkflow.gui.dialog import askYesNo
from tomo.objects import SetOfMeshes, SetOfCoordinates3D
from tomo.protocols import ProtTomoPicking
from dynamo import Plugin, VLL_FILE, CATALOG_BASENAME
from dynamo.viewers.views_tkinter_tree import DynamoTomoDialog


class OutputsBoxing(Enum):
    coordinates = SetOfCoordinates3D
    meshes = SetOfMeshes


class DynamoBoxing(ProtTomoPicking):
    """Manual vectorial picker from Dynamo. After choosing the Tomogram to be picked, the tomo slicer from Dynamo will
     be direclty loaded with all the models previously saved in the disk (if any).
     This picking will save the "user points" defined in a set of models and generate a set of meshes with them. In case
     the user carries out the workflow model for each of the models from the Dynamo GUI, a set of coordinates will be
     also created, containing all the interpolated coordinates, and the calculated orientation. It is possible to
     create several models at once in a given tomogram. Once the coordinates are defined, the models are automatically
     saved in the catalogue and registered."""

    _label = 'vectorial picking'
    _devStatus = BETA
    _possibleOutputs = OutputsBoxing

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        self.dlg = None
        self.dynModelsPathDict = {}  # Used to store the path where the corresponding models to a tomo are stored

    # --------------------------- DEFINE param functions ----------------------
    def _defineParams(self, form):
        ProtTomoPicking._defineParams(self, form)

        form.addParam('boxSize', IntParam, label="Box Size")
        form.addParam('deleteGenMFiles', BooleanParam,
                      default=True,
                      label='Remove the .m files generated after the execution?',
                      expertLevel=LEVEL_ADVANCED,
                      help='It can be useful for developers to check exactly what was .m files were generated by '
                           'Scipion and executed by Dynamo.')

    # --------------------------- INSERT steps functions ----------------------
    def _insertAllSteps(self):
        self._insertFunctionStep(self.convertInputStep)
        self._insertFunctionStep(self.launchDynamoBoxingStep, interactive=True)

    # --------------------------- STEPS functions -----------------------------
    def convertInputStep(self):
        """Initialize the catalogue"""
        # Create the vll (list of tomos) file
        vllFile = self._getExtraPath(VLL_FILE)
        tomoCounter = 1  # Matlab begins counting in 1
        with open(vllFile, 'w') as tomoFid:
            for tomo in self.inputTomograms.get().iterItems():
                tomoPath = abspath(tomo.getFileName())
                tomoFid.write(tomoPath + '\n')
                self.dynModelsPathDict[tomo.getTsId()] = self._getExtraPath(CATALOG_BASENAME, 'tomograms',
                                                                            'volume_%i' % tomoCounter, 'models')
                tomoCounter += 1

    def launchDynamoBoxingStep(self):
        tomoList = []
        for tomo in self.inputTomograms.get().iterItems():
            tomogram = tomo.clone()
            tomoList.append(tomogram)

        tomoProvider = DynamoTomogramProvider(tomoList, self._getExtraPath(), "txt")
        dynamoDialogCallingTime = datetime.datetime.now()
        self.dlg = DynamoTomoDialog(None, self._getExtraPath(),
                                    provider=tomoProvider,
                                    calledFromViewer=False)

        modelList = getDynamoModels(self._getExtraPath())
        if modelList:
            # Check if the modification file of the newest model file is higher than the time capture right before
            # calling the Dynamo dialog. In that case, it means that some modification was carried out by the user from
            # it and we have to ask if the changes should be saved
            if dynamoDialogCallingTime < getNewestModelModDate(modelList):
                # Open dialog to request confirmation to create output
                import tkinter as tk
                if askYesNo(Message.TITLE_SAVE_OUTPUT, Message.LABEL_SAVE_OUTPUT, tk.Frame()):
                    self._createOutput()
                    # Delete the .m generated files if requested
                if self.deleteGenMFiles.get():
                    pwutils.cleanPattern(self._getExtraPath('*.m'))

    def _createOutput(self):
        precedentsPointer = self.inputTomograms
        meshes, outCoords = createBoxingOutputObjects(self, precedentsPointer, boxSize=self.boxSize.get())
        # Define outputs and relations
        outputDict = {self._possibleOutputs.meshes.name: meshes}
        if outCoords:
            outputDict[self._possibleOutputs.coordinates.name] = outCoords
        self._defineOutputs(**outputDict)
        self._defineSourceRelation(precedentsPointer, meshes)
        if outCoords:
            self._defineSourceRelation(precedentsPointer, outCoords)
        self._updateOutputSet(self._possibleOutputs.meshes.name, meshes, state=meshes.STREAM_CLOSED)

    # --------------------------- DEFINE info functions ----------------------
    @staticmethod
    def getMethods(output):
        msg = 'User picked %d particles ' % output.getSize()
        msg += 'with a particle size of %s.' % output.getBoxSize()
        return msg

    def _methods(self):
        methodsMsgs = []
        if self.getOutputsSize() >= 1:
            for key, output in self.iterOutputAttributes():
                msg = self.getMethods(output)
                methodsMsgs.append("%s: %s" % (self.getObjectTag(output), msg))
        else:
            methodsMsgs.append(Message.TEXT_NO_OUTPUT_CO)

        return methodsMsgs

