"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SummarizationAppsyncStepfn = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const appsync = require("aws-cdk-lib/aws-appsync");
const ec2 = require("aws-cdk-lib/aws-ec2");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const sqs = require("aws-cdk-lib/aws-sqs");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const sfnTask = require("aws-cdk-lib/aws-stepfunctions-tasks");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const eventBridge = require("../../../common/helpers/eventbridge-helper");
const redisHelper = require("../../../common/helpers/redis-helper");
const s3BucketHelper = require("../../../common/helpers/s3-bucket-helper");
const vpcHelper = require("../../../common/helpers/vpc-helper");
class SummarizationAppsyncStepfn extends constructs_1.Construct {
    /**
       * @summary Constructs a new instance of the SummarizationAppsyncStepfn class.
       * @param {Construct} scope - represents the scope for all the resources.
       * @param {string} id - this is a a scope-unique id.
       * @param {SummarizationAppsyncStepfnProps} props - user provided props for the construct.
       * @since 0.0.0
       * @access public
       */
    constructor(scope, id, props) {
        super(scope, id);
        let stage = '-dev';
        if (props?.stage) {
            stage = props.stage;
        }
        // observability
        let lambda_tracing = lambda.Tracing.ACTIVE;
        let enable_xray = true;
        let api_log_config = {
            fieldLogLevel: appsync.FieldLogLevel.ALL,
            retention: logs.RetentionDays.TEN_YEARS,
        };
        if (props.observability == false) {
            enable_xray = false;
            lambda_tracing = lambda.Tracing.DISABLED;
            api_log_config = {
                fieldLogLevel: appsync.FieldLogLevel.NONE,
                retention: logs.RetentionDays.TEN_YEARS,
            };
        }
        ;
        // vpc
        if (props?.existingVpc) {
            this.vpc = props.existingVpc;
        }
        else {
            this.vpc = new ec2.Vpc(this, 'Vpc', props.vpcProps);
        }
        // Security group
        if (props?.existingSecurityGroup) {
            this.securityGroup = props.existingSecurityGroup;
        }
        else {
            this.securityGroup = new ec2.SecurityGroup(this, 'securityGroup', {
                vpc: this.vpc,
                allowAllOutbound: true,
            });
        }
        // vpc flowloggroup
        const logGroup = new logs.LogGroup(this, 'summarizationConstructLogGroup');
        const vpcFlowLogrole = new iam.Role(this, 'summarizationConstructRole', {
            assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
        });
        // vpc flowlogs
        new ec2.FlowLog(this, 'FlowLog', {
            resourceType: ec2.FlowLogResourceType.fromVpc(this.vpc),
            destination: ec2.FlowLogDestination.toCloudWatchLogs(logGroup, vpcFlowLogrole),
        });
        // bucket for storing server access logging
        const serverAccessLogBucket = new s3.Bucket(this, 'serverAccessLogBucket' + stage, {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            versioned: true,
            lifecycleRules: [{
                    expiration: aws_cdk_lib_1.Duration.days(90),
                }],
        });
        // bucket for input document
        s3BucketHelper.CheckS3Props({
            existingBucketObj: props.existingInputAssetsBucketObj,
            bucketProps: props.bucketInputsAssetsProps,
        });
        if (props?.existingInputAssetsBucketObj) {
            this.inputAssetBucket = props.existingInputAssetsBucketObj;
        }
        else if (props?.bucketInputsAssetsProps) {
            this.inputAssetBucket = new s3.Bucket(this, 'inputAssetsSummaryBucket' + stage, props.bucketInputsAssetsProps);
        }
        else {
            const bucketName = 'input-assets-summary-bucket' + stage + '-' + aws_cdk_lib_1.Aws.ACCOUNT_ID;
            this.inputAssetBucket = new s3.Bucket(this, 'inputAssetsSummaryBucket' + stage, {
                blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                encryption: s3.BucketEncryption.S3_MANAGED,
                bucketName: bucketName,
                serverAccessLogsBucket: serverAccessLogBucket,
                enforceSSL: true,
                versioned: true,
                lifecycleRules: [{
                        expiration: aws_cdk_lib_1.Duration.days(90),
                    }],
            });
        }
        // bucket for transformed document
        s3BucketHelper.CheckS3Props({
            existingBucketObj: props.existingProcessedAssetsBucketObj,
            bucketProps: props.bucketProcessedAssetsProps,
        });
        if (props?.existingProcessedAssetsBucketObj) {
            this.processedAssetBucket = props.existingProcessedAssetsBucketObj;
        }
        else if (props?.bucketProcessedAssetsProps) {
            this.processedAssetBucket = new s3.Bucket(this, 'processedAssetsSummaryBucket' + stage, props.bucketProcessedAssetsProps);
        }
        else {
            const bucketName = 'processed-assets-summary-bucket' + stage + '-' + aws_cdk_lib_1.Aws.ACCOUNT_ID;
            this.processedAssetBucket = new s3.Bucket(this, 'processedAssetsSummaryBucket' + stage, {
                blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                encryption: s3.BucketEncryption.S3_MANAGED,
                bucketName: bucketName,
                serverAccessLogsBucket: serverAccessLogBucket,
                enforceSSL: true,
                versioned: true,
                lifecycleRules: [{
                        expiration: aws_cdk_lib_1.Duration.days(90),
                    }],
            });
        }
        // set up redis cluster
        redisHelper.CheckRedisClusterProps(props);
        // build redis cluster only when cfnCacheClusterProps is set
        if (props?.cfnCacheClusterProps) {
            const redisSecurityGroup = redisHelper.getRedisSecurityGroup(this, {
                existingVpc: this.vpc,
            });
            const redisProps = {
                existingVpc: this.vpc,
                cfnCacheClusterProps: props.cfnCacheClusterProps,
                subnetIds: vpcHelper.getPrivateSubnetIDs(this.vpc),
                inboundSecurityGroup: this.securityGroup,
                redisSecurityGroup: redisSecurityGroup,
                redisPort: 8686,
            };
            this.redisCluster = redisHelper.buildRedisCluster(this, redisProps);
            redisHelper.setInboundRules(redisSecurityGroup, this.securityGroup, redisProps.redisPort);
        }
        else {
            this.redisCluster = props?.existingRedisCulster;
        }
        const redisHost = this.redisCluster?.attrRedisEndpointAddress;
        const redisPort = this.redisCluster?.attrRedisEndpointPort;
        eventBridge.CheckEventBridgeProps(props);
        // Create event bridge
        this.eventBridgeBus = eventBridge.buildEventBus(this, {
            existingEventBusInterface: props.existingBusInterface,
            eventBusProps: props.eventBusProps,
        });
        // make it generic for other auth config as well
        const authorizationConfig = {
            defaultAuthorization: {
                authorizationType: appsync.AuthorizationType.USER_POOL,
                userPoolConfig: { userPool: props.cognitoUserPool },
            },
            additionalAuthorizationModes: [
                {
                    authorizationType: appsync.AuthorizationType.IAM,
                },
            ],
        };
        const apiName = props.summaryApiName || 'summaryApi';
        // graphql api for summary. client invoke this api with given schema and cognito user pool auth.
        const summarizationGraphqlApi = new appsync.GraphqlApi(this, 'summarizationGraphqlApi' + stage, {
            name: apiName + stage,
            logConfig: api_log_config,
            schema: appsync.SchemaFile.fromAsset(path.join(__dirname, '../../../../resources/gen-ai/aws-summarization-appsync-stepfn/schema.graphql')),
            authorizationConfig: authorizationConfig,
            xrayEnabled: enable_xray,
        });
        this.graphqlApi = summarizationGraphqlApi;
        // If the user provides a mergedApi endpoint, the lambda
        // functions will use this endpoint to send their status updates
        const updateGraphQlApiEndpoint = !props.existingMergedApi ? summarizationGraphqlApi.graphqlUrl : props.existingMergedApi.attrGraphQlUrl;
        const updateGraphQlApiId = !props.existingMergedApi ? summarizationGraphqlApi.apiId : props.existingMergedApi.attrApiId;
        // bucket
        const transformedAssetBucketName = this.processedAssetBucket.bucketName;
        const inputAssetBucketName = this.inputAssetBucket.bucketName;
        const isFileTransformationRequired = props?.isFileTransformationRequired || 'false';
        // role
        const inputvalidatorLambdaRole = new iam.Role(this, 'inputvalidatorLambdaRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`],
                        })],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        inputvalidatorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:aws:ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        inputvalidatorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: [
                '*',
            ],
        }));
        inputvalidatorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject',
                's3:GetBucketLocation',
                's3:ListBucket',
                'appsync:GraphQL'],
            resources: ['arn:aws:s3:::' + inputAssetBucketName + '/*',
                'arn:aws:s3:::' + transformedAssetBucketName + '/*',
                'arn:aws:appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*'],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(inputvalidatorLambdaRole, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        // Lambda function to validate Input
        const inputValidatorLambda = new lambda.DockerImageFunction(this, 'inputValidatorLambda' + stage, {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-summarization-appsync-stepfn/input_validator')),
            functionName: 'summary_input_validator' + stage,
            description: 'Lambda function to validate input for summary api',
            vpc: this.vpc,
            tracing: lambda_tracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: 1769 * 1,
            timeout: aws_cdk_lib_1.Duration.minutes(5),
            role: inputvalidatorLambdaRole,
            environment: {
                GRAPHQL_URL: updateGraphQlApiEndpoint,
            },
        });
        // role
        const documentReaderLambdaRole = new iam.Role(this, 'documentReaderLambdaRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`],
                        })],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        documentReaderLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:aws:ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        documentReaderLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: [
                '*',
            ],
        }));
        documentReaderLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject',
                's3:GetBucketLocation',
                's3:ListBucket',
                's3:PutObject',
                'appsync:GraphQL'],
            resources: ['arn:aws:s3:::' + inputAssetBucketName + '/*',
                'arn:aws:s3:::' + transformedAssetBucketName + '/*',
                'arn:aws:appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*'],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(documentReaderLambdaRole, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used',
            },
        ], true);
        const documentReaderLambda = new lambda.DockerImageFunction(this, 'documentReaderLambda' + stage, {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-summarization-appsync-stepfn/document_reader')),
            functionName: 'summary_document_reader' + stage,
            description: 'Lambda function to read the input transformed document',
            vpc: this.vpc,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: 1769 * 1,
            tracing: lambda_tracing,
            timeout: aws_cdk_lib_1.Duration.minutes(5),
            role: documentReaderLambdaRole,
            environment: {
                REDIS_HOST: redisHost,
                REDIS_PORT: redisPort,
                TRANSFORMED_ASSET_BUCKET: transformedAssetBucketName,
                INPUT_ASSET_BUCKET: inputAssetBucketName,
                IS_FILE_TRANSFORMED: isFileTransformationRequired,
                GRAPHQL_URL: updateGraphQlApiEndpoint,
            },
        });
        const summaryChainType = props?.summaryChainType || 'stuff';
        // role
        const summaryGeneratorLambdaRole = new iam.Role(this, 'summaryGeneratorLambdaRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`],
                        })],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        summaryGeneratorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:aws:ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        summaryGeneratorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: [
                '*',
            ],
        }));
        summaryGeneratorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject',
                's3:GetBucketLocation',
                's3:ListBucket',
                's3:PutObject',
                'appsync:GraphQL',
                'bedrock:InvokeModel',
                'bedrock:InvokeModelWithResponseStream'],
            resources: ['arn:aws:s3:::' + inputAssetBucketName + '/*',
                'arn:aws:s3:::' + transformedAssetBucketName + '/*',
                'arn:aws:appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*',
                'arn:aws:bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model/*'],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(summaryGeneratorLambdaRole, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        const generateSummarylambda = new lambda.DockerImageFunction(this, 'generateSummarylambda' + stage, {
            functionName: 'summary_generator' + stage,
            description: 'Lambda function to generate the summary',
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-summarization-appsync-stepfn/summary_generator')),
            vpc: this.vpc,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: 1769 * 4,
            timeout: aws_cdk_lib_1.Duration.minutes(10),
            tracing: lambda_tracing,
            role: summaryGeneratorLambdaRole,
            environment: {
                REDIS_HOST: redisHost,
                REDIS_PORT: redisPort,
                ASSET_BUCKET_NAME: transformedAssetBucketName,
                GRAPHQL_URL: updateGraphQlApiEndpoint,
                SUMMARY_LLM_CHAIN_TYPE: summaryChainType,
            },
        });
        this.inputAssetBucket?.grantRead(generateSummarylambda);
        this.processedAssetBucket?.grantReadWrite(generateSummarylambda);
        this.inputAssetBucket?.grantRead(documentReaderLambda);
        this.processedAssetBucket?.grantReadWrite(documentReaderLambda);
        const enableOperationalMetric = props.enableOperationalMetric || true;
        const solution_id = 'genai_cdk_' + id;
        if (enableOperationalMetric) {
            documentReaderLambda.addEnvironment('AWS_SDK_UA_APP_ID', solution_id);
            generateSummarylambda.addEnvironment('AWS_SDK_UA_APP_ID', solution_id);
            inputValidatorLambda.addEnvironment('AWS_SDK_UA_APP_ID', solution_id);
        }
        ;
        // create datasource at appsync
        const SummaryStatusDataSource = new appsync.NoneDataSource(this, 'noneDataSource' + stage, {
            api: summarizationGraphqlApi,
            name: 'SummaryStatusDataSource',
        });
        SummaryStatusDataSource.createResolver('summaryResponseresolver' + stage, {
            typeName: 'Mutation',
            fieldName: 'updateSummaryJobStatus',
            requestMappingTemplate: appsync.MappingTemplate.fromString('{ "version": "2017-02-28", "payload": $util.toJson($context.args) }'),
            responseMappingTemplate: appsync.MappingTemplate.fromString('$util.toJson($context.result)'),
        });
        // step functions
        const inputValidationTask = new sfnTask.LambdaInvoke(this, 'Validate Input ', {
            lambdaFunction: inputValidatorLambda,
            resultPath: '$.validation_result',
        });
        const documentReaderTask = new sfnTask.LambdaInvoke(this, 'Read document and check summary in cache ', {
            lambdaFunction: documentReaderLambda,
            resultPath: '$.document_result',
        });
        const generateSummaryTask = new sfnTask.LambdaInvoke(this, 'Generate Summary with llm', {
            lambdaFunction: generateSummarylambda,
            resultPath: '$.summary_result',
        });
        const dlq = new sqs.Queue(this, 'dlq', {
            queueName: 'summarydlq' + stage,
            retentionPeriod: aws_cdk_lib_1.Duration.days(7),
            enforceSSL: true,
        });
        const jobFailed = new sfn.Fail(this, 'Failed', {
            comment: 'AWS summary Job failed',
        });
        const jobSuccess = new sfn.Succeed(this, 'succeeded', {
            comment: 'AWS summary Job succeeded',
        });
        // step function choice steps
        const validateInputChoice = new sfn.Choice(this, 'is Valid Parameters?', {
            outputPath: '$.validation_result.Payload.files',
        });
        const summaryfromCacheChoice = new sfn.Choice(this, 'is Summary in Cache?', {});
        const fileStatusForSummarization = new sfn.Choice(this, 'is file status ready for summarization?', {
            outputPath: '$.document_result.Payload',
        });
        // step function, run files in parallel
        const runFilesInparallel = new sfn.Map(this, 'Run Files in Parallel', {
            maxConcurrency: 100,
        }).iterator(documentReaderTask.next(summaryfromCacheChoice
            .when(sfn.Condition.booleanEquals('$.document_result.Payload.is_summary_available', true), jobSuccess).otherwise(fileStatusForSummarization.when(sfn.Condition.stringMatches('$.document_result.Payload.status', 'Error'), jobSuccess).otherwise(generateSummaryTask.next(jobSuccess)))));
        const summarizationLogGroup = new logs.LogGroup(this, 'summarizationLogGroup', {});
        // step function definition
        const definition = inputValidationTask.next(validateInputChoice
            .when(sfn.Condition.booleanEquals('$.validation_result.Payload.isValid', false), jobFailed)
            .otherwise(runFilesInparallel));
        // step function
        const summarizationStepFunction = new sfn.StateMachine(this, 'summarizationStepFunction', {
            definitionBody: sfn.DefinitionBody.fromChainable(definition),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            logs: {
                destination: summarizationLogGroup,
                level: sfn.LogLevel.ALL,
            },
            tracingEnabled: enable_xray,
        });
        this.stateMachine = summarizationStepFunction;
        // event bridge datasource for summarization api
        const eventBridgeDataSource = summarizationGraphqlApi.addEventBridgeDataSource('eventBridgeDataSource', this.eventBridgeBus);
        this.eventBridgeBus.grantPutEventsTo(eventBridgeDataSource.grantPrincipal);
        // add resolver on summary graphql schema
        eventBridgeDataSource.createResolver('generateSummary', {
            typeName: 'Mutation',
            fieldName: 'generateSummary',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
         {
          "version": "2018-05-29",
          "operation": "PutEvents",
          "events": [{
              "source": "summary",
              "detail": $util.toJson($context.arguments),
              "detailType": "genAIdemo"
          }
          ]
  } 
  `),
            responseMappingTemplate: appsync.MappingTemplate.fromString('#if($ctx.error)$utilerror($ctx.error.message, $ctx.error.type, $ctx.result) #end $util.toJson($ctx.result)'),
        });
        new events.Rule(this, 'SummaryMutationRule', {
            description: 'Summary Mutation Rule',
            eventBus: this.eventBridgeBus,
            eventPattern: {
                source: ['summary'],
            },
            targets: [
                new targets.SfnStateMachine(summarizationStepFunction, {
                    deadLetterQueue: dlq,
                    retryAttempts: 1,
                }),
            ],
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
SummarizationAppsyncStepfn[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.SummarizationAppsyncStepfn", version: "0.0.89" };
exports.SummarizationAppsyncStepfn = SummarizationAppsyncStepfn;
//# sourceMappingURL=data:application/json;base64,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