"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RagAppsyncStepfnOpensearch = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const appsync = require("aws-cdk-lib/aws-appsync");
const ec2 = require("aws-cdk-lib/aws-ec2");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const stepfn = require("aws-cdk-lib/aws-stepfunctions");
const stepfn_task = require("aws-cdk-lib/aws-stepfunctions-tasks");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const s3_bucket_helper = require("../../../common/helpers/s3-bucket-helper");
const vpc_helper = require("../../../common/helpers/vpc-helper");
/**
   * @summary The RagApiGatewayOpensearch class.
   */
class RagAppsyncStepfnOpensearch extends constructs_1.Construct {
    /**
       * @summary Constructs a new instance of the RagAppsyncStepfnOpensearch class.
       * @param {cdk.App} scope - represents the scope for all the resources.
       * @param {string} id - this is a a scope-unique id.
       * @param {RagAppsyncStepfnOpensearchProps} props - user provided props for the construct.
       * @since 0.0.0
       * @access public
       */
    constructor(scope, id, props) {
        super(scope, id);
        // stage
        let stage = '-dev';
        if (props?.stage) {
            stage = props.stage;
        }
        // observability
        let lambda_tracing = lambda.Tracing.ACTIVE;
        let enable_xray = true;
        let api_log_config = {
            fieldLogLevel: appsync.FieldLogLevel.ALL,
            retention: logs.RetentionDays.TEN_YEARS,
        };
        if (props.observability == false) {
            enable_xray = false;
            lambda_tracing = lambda.Tracing.DISABLED;
            api_log_config = {
                fieldLogLevel: appsync.FieldLogLevel.NONE,
                retention: logs.RetentionDays.TEN_YEARS,
            };
        }
        ;
        vpc_helper.CheckVpcProps(props);
        s3_bucket_helper.CheckS3Props({
            existingBucketObj: props.existingInputAssetsBucketObj,
            bucketProps: props.bucketInputsAssetsProps,
        });
        s3_bucket_helper.CheckS3Props({
            existingBucketObj: props.existingProcessedAssetsBucketObj,
            bucketProps: props.bucketProcessedAssetsProps,
        });
        if (props?.existingVpc) {
            this.vpc = props.existingVpc;
        }
        else {
            this.vpc = new ec2.Vpc(this, 'Vpc', props.vpcProps);
        }
        // Security group
        if (props?.existingSecurityGroup) {
            this.securityGroup = props.existingSecurityGroup;
        }
        else {
            this.securityGroup = new ec2.SecurityGroup(this, 'securityGroup', {
                vpc: this.vpc,
                allowAllOutbound: true,
                securityGroupName: 'securityGroup' + stage,
            });
        }
        // vpc flowloggroup
        const logGroup = new logs.LogGroup(this, 'ingestionConstructLogGroup');
        const role = new iam.Role(this, 'ingestionConstructRole', {
            assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
        });
        // vpc flowlogs
        new ec2.FlowLog(this, 'FlowLog', {
            resourceType: ec2.FlowLogResourceType.fromVpc(this.vpc),
            destination: ec2.FlowLogDestination.toCloudWatchLogs(logGroup, role),
        });
        // bucket for storing server access logging
        const serverAccessLogBucket = new s3.Bucket(this, 'serverAccessLogBucket' + stage, {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            versioned: true,
            lifecycleRules: [{
                    expiration: aws_cdk_lib_1.Duration.days(90),
                }],
        });
        // Bucket containing the inputs assets (documents - multiple modalities) uploaded by the user
        let inputAssetsBucket;
        if (!props.existingInputAssetsBucketObj) {
            let tmpBucket;
            if (!props.bucketInputsAssetsProps) {
                tmpBucket = new s3.Bucket(this, 'inputAssetsBucket' + stage, {
                    blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                    encryption: s3.BucketEncryption.S3_MANAGED,
                    bucketName: 'input-assets-bucket' + stage + '-' + aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    serverAccessLogsBucket: serverAccessLogBucket,
                    enforceSSL: true,
                    versioned: true,
                    lifecycleRules: [{
                            expiration: aws_cdk_lib_1.Duration.days(90),
                        }],
                });
            }
            else {
                tmpBucket = new s3.Bucket(this, 'InputAssetsBucket' + stage, props.bucketInputsAssetsProps);
            }
            inputAssetsBucket = tmpBucket;
            this.s3InputAssetsBucket = tmpBucket;
        }
        else {
            inputAssetsBucket = props.existingInputAssetsBucketObj;
        }
        // this is the one we manipulate, we know it exists
        this.s3InputAssetsBucketInterface = inputAssetsBucket;
        // Bucket containing the processed assets (documents - text format) uploaded by the user
        let processedAssetsBucket;
        if (!props.existingProcessedAssetsBucketObj) {
            let tmpBucket;
            if (!props.bucketInputsAssetsProps) {
                tmpBucket = new s3.Bucket(this, 'processedAssetsBucket' + stage, {
                    blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                    encryption: s3.BucketEncryption.S3_MANAGED,
                    bucketName: 'processed-assets-bucket' + stage + '-' + aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    serverAccessLogsBucket: serverAccessLogBucket,
                    enforceSSL: true,
                    versioned: true,
                    lifecycleRules: [{
                            expiration: aws_cdk_lib_1.Duration.days(90),
                        }],
                });
            }
            else {
                tmpBucket = new s3.Bucket(this, 'processedAssetsBucket' + stage, props.bucketProcessedAssetsProps);
            }
            processedAssetsBucket = tmpBucket;
            this.s3ProcessedAssetsBucket = tmpBucket;
        }
        else {
            processedAssetsBucket = props.existingProcessedAssetsBucketObj;
        }
        // this is the one we manipulate, we know it exists
        this.s3ProcessedAssetsBucketInterface = processedAssetsBucket;
        // GraphQL API
        const ingestion_graphql_api = new appsync.GraphqlApi(this, 'ingestionGraphqlApi', {
            name: 'ingestionGraphqlApi' + stage,
            schema: appsync.SchemaFile.fromAsset(path.join(__dirname, '../../../../resources/gen-ai/aws-rag-appsync-stepfn-opensearch/schema.graphql')),
            authorizationConfig: {
                defaultAuthorization: {
                    authorizationType: appsync.AuthorizationType.USER_POOL,
                    userPoolConfig: { userPool: props.cognitoUserPool },
                },
                additionalAuthorizationModes: [
                    {
                        authorizationType: appsync.AuthorizationType.IAM,
                    },
                ],
            },
            xrayEnabled: enable_xray,
            logConfig: api_log_config,
        });
        this.graphqlApi = ingestion_graphql_api;
        // If the user provides a mergedApi endpoint, the lambda
        // functions will use this endpoint to send their status updates
        const updateGraphQlApiEndpoint = !props.existingMergedApi ? ingestion_graphql_api.graphqlUrl : props.existingMergedApi.attrGraphQlUrl;
        const updateGraphQlApiId = !props.existingMergedApi ? ingestion_graphql_api.apiId : props.existingMergedApi.attrApiId;
        const job_status_data_source = new appsync.NoneDataSource(this, 'NoneDataSourceIngestion', {
            api: this.graphqlApi,
            name: 'JobStatusDataSource',
        });
        job_status_data_source.createResolver('updateIngestionJobStatusResolver', {
            fieldName: 'updateIngestionJobStatus',
            typeName: 'Mutation',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
                    {
                        "version": "2017-02-28",
                        "payload": $util.toJson($context.args)
                    }
                    `),
            responseMappingTemplate: appsync.MappingTemplate.fromString('$util.toJson($context.result)'),
        });
        if (!props.existingBusInterface) {
            this.ingestionBus = new events.EventBus(this, 'ingestionEventBus' + stage, {
                eventBusName: 'ingestionEventBus' + stage,
            });
        }
        else {
            this.ingestionBus = props.existingBusInterface;
        }
        // create httpdatasource with ingestion_graphql_api
        const event_bridge_datasource = this.graphqlApi.addEventBridgeDataSource('ingestionEventBridgeDataSource' + stage, this.ingestionBus, {
            name: 'ingestionEventBridgeDataSource' + stage,
        });
        // Lambda function used to validate inputs in the step function
        const validate_input_function = new lambda.DockerImageFunction(this, 'lambda_function_validation_input' + stage, {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-rag-appsync-stepfn-opensearch/input_validation/src')),
            functionName: 'ingestion_input_validation_docker' + stage,
            description: 'Lambda function for validating input files formats',
            vpc: this.vpc,
            tracing: lambda_tracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: 1769 * 4,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            environment: {
                GRAPHQL_URL: updateGraphQlApiEndpoint,
            },
        });
        // Add GraphQl permissions to the IAM role for the Lambda function
        validate_input_function.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'appsync:GraphQL',
            ],
            resources: [
                'arn:aws:appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*',
            ],
        }));
        // The lambda will pull documents from the input bucket, transform them, and upload
        // the artifacts to the processed bucket
        // we don't use grant read here since it has no effect in case of existing buckets provided by the user
        const s3_transformer_job_function_role = new iam.Role(this, 's3_transformer_job_function_role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`],
                        })],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:aws:ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: [
                '*',
            ],
        }));
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:GetObject',
                's3:GetObject*',
                's3:GetBucket*',
                's3:List*',
            ],
            resources: [
                'arn:aws:s3:::' + this.s3InputAssetsBucketInterface?.bucketName,
                'arn:aws:s3:::' + this.s3InputAssetsBucketInterface?.bucketName + '/*',
            ],
        }));
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:PutObjectRetention',
                's3:List*',
                's3:GetBucket*',
                's3:Abort*',
                's3:DeleteObject*',
                's3:PutObjectLegalHold',
                's3:PutObjectTagging',
                's3:PutObjectVersionTagging',
                's3:PutObject',
                's3:GetObject*'],
            resources: [
                'arn:aws:s3:::' + this.s3ProcessedAssetsBucketInterface?.bucketName,
                'arn:aws:s3:::' + this.s3ProcessedAssetsBucketInterface?.bucketName + '/*',
            ],
        }));
        // Add GraphQl permissions to the IAM role for the Lambda function
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'appsync:GraphQL',
            ],
            resources: [
                'arn:aws:appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*',
            ],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(s3_transformer_job_function_role, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        const s3_transformer_job_function = new lambda.DockerImageFunction(this, 'lambda_function_s3_file_transformer' + stage, {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-rag-appsync-stepfn-opensearch/s3_file_transformer/src')),
            functionName: 's3_file_transformer_docker' + stage,
            description: 'Lambda function for converting files from their input format to text',
            vpc: this.vpc,
            tracing: lambda_tracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: 1769 * 4,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: s3_transformer_job_function_role,
            environment: {
                INPUT_BUCKET: this.s3InputAssetsBucketInterface.bucketName,
                OUTPUT_BUCKET: this.s3ProcessedAssetsBucketInterface.bucketName,
                GRAPHQL_URL: updateGraphQlApiEndpoint,
            },
        });
        let SecretId = 'NONE';
        if (props.openSearchSecret) {
            SecretId = props.openSearchSecret.secretName;
        }
        const embeddings_job_function_role = new iam.Role(this, 'embeddings_job_function_role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`],
                        })],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:aws:ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: [
                '*',
            ],
        }));
        embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:GetObject',
                's3:GetObject*',
                's3:GetBucket*',
                's3:List*',
            ],
            resources: [
                'arn:aws:s3:::' + this.s3ProcessedAssetsBucketInterface?.bucketName,
                'arn:aws:s3:::' + this.s3ProcessedAssetsBucketInterface?.bucketName + '/*',
            ],
        }));
        embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['es:*'],
            resources: [
                'arn:aws:es:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':domain/' + props.existingOpensearchDomain.domainName + '/*',
                'arn:aws:es:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':domain/' + props.existingOpensearchDomain.domainName,
            ],
        }));
        // Add Amazon Bedrock permissions to the IAM role for the Lambda function
        embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['bedrock:*'],
            resources: [
                'arn:aws:bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model',
                'arn:aws:bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model/*',
            ],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(embeddings_job_function_role, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        // The lambda will access the opensearch credentials
        if (props.openSearchSecret) {
            props.openSearchSecret.grantRead(embeddings_job_function_role);
        }
        // Lambda function performing the embedding job
        const embeddings_job_function = new lambda.DockerImageFunction(this, 'lambda_function_embeddings_job' + stage, {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-rag-appsync-stepfn-opensearch/embeddings_job/src')),
            functionName: 'embeddings_job_docker' + stage,
            description: 'Lambda function for creating documents chunks, embeddings and storing them in Amazon Opensearch',
            vpc: this.vpc,
            tracing: lambda_tracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: 1769 * 4,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: embeddings_job_function_role,
            environment: {
                OUTPUT_BUCKET: this.s3ProcessedAssetsBucketInterface.bucketName,
                GRAPHQL_URL: updateGraphQlApiEndpoint,
                OPENSEARCH_INDEX: props.openSearchIndexName,
                OPENSEARCH_DOMAIN_ENDPOINT: props.existingOpensearchDomain.domainEndpoint,
                OPENSEARCH_SECRET_ID: SecretId,
            },
        });
        const enableOperationalMetric = props.enableOperationalMetric || true;
        const solution_id = 'genai_cdk_' + id;
        if (enableOperationalMetric) {
            embeddings_job_function.addEnvironment('AWS_SDK_UA_APP_ID', solution_id);
            s3_transformer_job_function.addEnvironment('AWS_SDK_UA_APP_ID', solution_id);
            validate_input_function.addEnvironment('AWS_SDK_UA_APP_ID', solution_id);
        }
        ;
        // Add GraphQl permissions to the IAM role for the Lambda function
        embeddings_job_function.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'appsync:GraphQL',
            ],
            resources: [
                'arn:aws:appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*',
            ],
        }));
        // Step function definition
        const inputValidationTask = new stepfn_task.LambdaInvoke(this, 'Validate Ingestion Input', {
            lambdaFunction: validate_input_function,
            resultPath: '$.validation_result',
        });
        const fileTransformationTask = new stepfn_task.LambdaInvoke(this, 'Download and transform document to raw text', {
            lambdaFunction: s3_transformer_job_function,
            resultPath: '$.s3_transformer_result',
        });
        const embeddingsTask = new stepfn_task.LambdaInvoke(this, 'Generate embeddings from processed documents and store them', {
            lambdaFunction: embeddings_job_function,
            resultPath: '$',
        });
        const validate_input_choice = new stepfn.Choice(this, 'Is Valid Ingestion Parameters?', {
            outputPath: '$.validation_result.Payload.files',
        });
        const run_files_in_parallel = new stepfn.Map(this, 'Map State', {
            maxConcurrency: 100,
        }).iterator(fileTransformationTask);
        const jobFailed = new stepfn.Fail(this, 'Job Failed', {
            cause: 'Validation job failed',
            error: 'DescribeJob returned FAILED',
        });
        const definition = inputValidationTask.next(validate_input_choice.when(stepfn.Condition.booleanEquals('$.validation_result.Payload.isValid', false), jobFailed).otherwise(run_files_in_parallel.next(embeddingsTask)));
        const ingestion_step_function = new stepfn.StateMachine(this, 'IngestionStateMachine', {
            stateMachineName: 'IngestionStateMachine' + stage,
            definitionBody: stepfn.DefinitionBody.fromChainable(definition),
            timeout: aws_cdk_lib_1.Duration.minutes(30),
            logs: {
                destination: new logs.LogGroup(this, 'ingestionStepFunctionLogGroup'),
                level: stepfn.LogLevel.ALL,
            },
            tracingEnabled: enable_xray,
        });
        this.stateMachine = ingestion_step_function;
        this.ingestionBus.grantPutEventsTo(event_bridge_datasource.grantPrincipal);
        event_bridge_datasource.createResolver('ingestDocumentResolver', {
            fieldName: 'ingestDocuments',
            typeName: 'Mutation',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
                    {
                        "version": "2018-05-29",
                        "operation": "PutEvents",
                        "events": [{
                            "source": "ingestion",
                            "detail": $util.toJson($context.arguments),
                            "detailType": "genAIdemo"
                        }
                        ]
                    } 
                    `),
            responseMappingTemplate: appsync.MappingTemplate.fromString(`
                    #if($ctx.error)
                        $util.error($ctx.error.message, $ctx.error.type, $ctx.result)
                    #end
                        $util.toJson($ctx.result)
                    `),
        });
        const rule = new events.Rule(this, 'ingestionRule' + stage, {
            description: 'Rule to trigger ingestion function',
            eventBus: this.ingestionBus,
            eventPattern: {
                source: ['ingestion'],
            },
        });
        rule.addTarget(new targets.SfnStateMachine(this.stateMachine));
    }
}
_a = JSII_RTTI_SYMBOL_1;
RagAppsyncStepfnOpensearch[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.RagAppsyncStepfnOpensearch", version: "0.0.89" };
exports.RagAppsyncStepfnOpensearch = RagAppsyncStepfnOpensearch;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi9zcmMvcGF0dGVybnMvZ2VuLWFpL2F3cy1yYWctYXBwc3luYy1zdGVwZm4tb3BlbnNlYXJjaC9pbmRleC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBOzs7Ozs7Ozs7OztHQVdHO0FBQ0gsNkJBQTZCO0FBQzdCLDZDQUE0QztBQUM1QyxtREFBbUQ7QUFFbkQsMkNBQTJDO0FBQzNDLGlEQUFpRDtBQUNqRCwwREFBMEQ7QUFDMUQsMkNBQTJDO0FBQzNDLGlEQUFpRDtBQUNqRCw2Q0FBNkM7QUFFN0MseUNBQXlDO0FBRXpDLHdEQUF3RDtBQUN4RCxtRUFBbUU7QUFDbkUscUNBQTBDO0FBQzFDLDJDQUF1QztBQUN2Qyw2RUFBNkU7QUFDN0UsaUVBQWlFO0FBdUhqRTs7S0FFSztBQUNMLE1BQWEsMEJBQTJCLFNBQVEsc0JBQVM7SUEwQ3ZEOzs7Ozs7O1NBT0s7SUFDTCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXNDO1FBQzlFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsUUFBUTtRQUNSLElBQUksS0FBSyxHQUFHLE1BQU0sQ0FBQztRQUNuQixJQUFJLEtBQUssRUFBRSxLQUFLLEVBQUU7WUFDaEIsS0FBSyxHQUFHLEtBQUssQ0FBQyxLQUFLLENBQUM7U0FDckI7UUFFRCxnQkFBZ0I7UUFDaEIsSUFBSSxjQUFjLEdBQUcsTUFBTSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUM7UUFDM0MsSUFBSSxXQUFXLEdBQUcsSUFBSSxDQUFDO1FBQ3ZCLElBQUksY0FBYyxHQUFHO1lBQ25CLGFBQWEsRUFBRSxPQUFPLENBQUMsYUFBYSxDQUFDLEdBQUc7WUFDeEMsU0FBUyxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUztTQUN4QyxDQUFDO1FBQ0YsSUFBSSxLQUFLLENBQUMsYUFBYSxJQUFJLEtBQUssRUFBRTtZQUNoQyxXQUFXLEdBQUcsS0FBSyxDQUFDO1lBQ3BCLGNBQWMsR0FBRyxNQUFNLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQztZQUN6QyxjQUFjLEdBQUc7Z0JBQ2YsYUFBYSxFQUFFLE9BQU8sQ0FBQyxhQUFhLENBQUMsSUFBSTtnQkFDekMsU0FBUyxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUzthQUN4QyxDQUFDO1NBQ0g7UUFBQSxDQUFDO1FBRUYsVUFBVSxDQUFDLGFBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNoQyxnQkFBZ0IsQ0FBQyxZQUFZLENBQUM7WUFDNUIsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLDRCQUE0QjtZQUNyRCxXQUFXLEVBQUUsS0FBSyxDQUFDLHVCQUF1QjtTQUMzQyxDQUFDLENBQUM7UUFDSCxnQkFBZ0IsQ0FBQyxZQUFZLENBQUM7WUFDNUIsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLGdDQUFnQztZQUN6RCxXQUFXLEVBQUUsS0FBSyxDQUFDLDBCQUEwQjtTQUM5QyxDQUFDLENBQUM7UUFFSCxJQUFJLEtBQUssRUFBRSxXQUFXLEVBQUU7WUFDdEIsSUFBSSxDQUFDLEdBQUcsR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDO1NBQzlCO2FBQU07WUFDTCxJQUFJLENBQUMsR0FBRyxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQztTQUNyRDtRQUVELGlCQUFpQjtRQUNqQixJQUFJLEtBQUssRUFBRSxxQkFBcUIsRUFBRTtZQUNoQyxJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQztTQUNsRDthQUFNO1lBQ0wsSUFBSSxDQUFDLGFBQWEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQ3hDLElBQUksRUFDSixlQUFlLEVBQ2Y7Z0JBQ0UsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO2dCQUNiLGdCQUFnQixFQUFFLElBQUk7Z0JBQ3RCLGlCQUFpQixFQUFFLGVBQWUsR0FBQyxLQUFLO2FBQ3pDLENBQ0YsQ0FBQztTQUNIO1FBRUQsbUJBQW1CO1FBQ25CLE1BQU0sUUFBUSxHQUFHLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsNEJBQTRCLENBQUMsQ0FBQztRQUN2RSxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLHdCQUF3QixFQUFFO1lBQ3hELFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyw2QkFBNkIsQ0FBQztTQUNuRSxDQUFDLENBQUM7UUFFSCxlQUFlO1FBQ2YsSUFBSSxHQUFHLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDL0IsWUFBWSxFQUFFLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQztZQUN2RCxXQUFXLEVBQUUsR0FBRyxDQUFDLGtCQUFrQixDQUFDLGdCQUFnQixDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUM7U0FDckUsQ0FBQyxDQUFDO1FBRUgsMkNBQTJDO1FBQzNDLE1BQU0scUJBQXFCLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksRUFDOUMsdUJBQXVCLEdBQUMsS0FBSyxFQUM3QjtZQUNFLGlCQUFpQixFQUFFLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTO1lBQ2pELFVBQVUsRUFBRSxFQUFFLENBQUMsZ0JBQWdCLENBQUMsVUFBVTtZQUMxQyxTQUFTLEVBQUUsSUFBSTtZQUNmLGNBQWMsRUFBRSxDQUFDO29CQUNmLFVBQVUsRUFBRSxzQkFBUSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUM7aUJBQzlCLENBQUM7U0FDSCxDQUFDLENBQUM7UUFFTCw2RkFBNkY7UUFDN0YsSUFBSSxpQkFBNkIsQ0FBQztRQUVsQyxJQUFJLENBQUMsS0FBSyxDQUFDLDRCQUE0QixFQUFFO1lBQ3ZDLElBQUksU0FBb0IsQ0FBQztZQUN6QixJQUFJLENBQUMsS0FBSyxDQUFDLHVCQUF1QixFQUFFO2dCQUNsQyxTQUFTLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxtQkFBbUIsR0FBQyxLQUFLLEVBQ3ZEO29CQUNFLGlCQUFpQixFQUFFLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTO29CQUNqRCxVQUFVLEVBQUUsRUFBRSxDQUFDLGdCQUFnQixDQUFDLFVBQVU7b0JBQzFDLFVBQVUsRUFBRSxxQkFBcUIsR0FBQyxLQUFLLEdBQUMsR0FBRyxHQUFDLGlCQUFHLENBQUMsVUFBVTtvQkFDMUQsc0JBQXNCLEVBQUUscUJBQXFCO29CQUM3QyxVQUFVLEVBQUUsSUFBSTtvQkFDaEIsU0FBUyxFQUFFLElBQUk7b0JBQ2YsY0FBYyxFQUFFLENBQUM7NEJBQ2YsVUFBVSxFQUFFLHNCQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQzt5QkFDOUIsQ0FBQztpQkFDSCxDQUFDLENBQUM7YUFDTjtpQkFBTTtnQkFDTCxTQUFTLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxtQkFBbUIsR0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLHVCQUF1QixDQUFDLENBQUM7YUFDM0Y7WUFDRCxpQkFBaUIsR0FBRyxTQUFTLENBQUM7WUFDOUIsSUFBSSxDQUFDLG1CQUFtQixHQUFHLFNBQVMsQ0FBQztTQUN0QzthQUFNO1lBQ0wsaUJBQWlCLEdBQUcsS0FBSyxDQUFDLDRCQUE0QixDQUFDO1NBQ3hEO1FBRUQsbURBQW1EO1FBQ25ELElBQUksQ0FBQyw0QkFBNEIsR0FBRyxpQkFBaUIsQ0FBQztRQUV0RCx3RkFBd0Y7UUFDeEYsSUFBSSxxQkFBaUMsQ0FBQztRQUV0QyxJQUFJLENBQUMsS0FBSyxDQUFDLGdDQUFnQyxFQUFFO1lBQzNDLElBQUksU0FBb0IsQ0FBQztZQUN6QixJQUFJLENBQUMsS0FBSyxDQUFDLHVCQUF1QixFQUFFO2dCQUNsQyxTQUFTLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSx1QkFBdUIsR0FBQyxLQUFLLEVBQzNEO29CQUNFLGlCQUFpQixFQUFFLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTO29CQUNqRCxVQUFVLEVBQUUsRUFBRSxDQUFDLGdCQUFnQixDQUFDLFVBQVU7b0JBQzFDLFVBQVUsRUFBRSx5QkFBeUIsR0FBQyxLQUFLLEdBQUMsR0FBRyxHQUFDLGlCQUFHLENBQUMsVUFBVTtvQkFDOUQsc0JBQXNCLEVBQUUscUJBQXFCO29CQUM3QyxVQUFVLEVBQUUsSUFBSTtvQkFDaEIsU0FBUyxFQUFFLElBQUk7b0JBQ2YsY0FBYyxFQUFFLENBQUM7NEJBQ2YsVUFBVSxFQUFFLHNCQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQzt5QkFDOUIsQ0FBQztpQkFDSCxDQUFDLENBQUM7YUFDTjtpQkFBTTtnQkFDTCxTQUFTLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSx1QkFBdUIsR0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLDBCQUEwQixDQUFDLENBQUM7YUFDbEc7WUFDRCxxQkFBcUIsR0FBRyxTQUFTLENBQUM7WUFDbEMsSUFBSSxDQUFDLHVCQUF1QixHQUFHLFNBQVMsQ0FBQztTQUMxQzthQUFNO1lBQ0wscUJBQXFCLEdBQUcsS0FBSyxDQUFDLGdDQUFnQyxDQUFDO1NBQ2hFO1FBRUQsbURBQW1EO1FBQ25ELElBQUksQ0FBQyxnQ0FBZ0MsR0FBRyxxQkFBcUIsQ0FBQztRQUU5RCxjQUFjO1FBQ2QsTUFBTSxxQkFBcUIsR0FBRyxJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQ2xELElBQUksRUFDSixxQkFBcUIsRUFDckI7WUFDRSxJQUFJLEVBQUUscUJBQXFCLEdBQUMsS0FBSztZQUNqQyxNQUFNLEVBQUUsT0FBTyxDQUFDLFVBQVUsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsK0VBQStFLENBQUMsQ0FBQztZQUMzSSxtQkFBbUIsRUFBRTtnQkFDbkIsb0JBQW9CLEVBQUU7b0JBQ3BCLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTO29CQUN0RCxjQUFjLEVBQUUsRUFBRSxRQUFRLEVBQUUsS0FBSyxDQUFDLGVBQWUsRUFBRTtpQkFDcEQ7Z0JBQ0QsNEJBQTRCLEVBQUU7b0JBQzVCO3dCQUNFLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxHQUFHO3FCQUNqRDtpQkFDRjthQUNGO1lBQ0QsV0FBVyxFQUFFLFdBQVc7WUFDeEIsU0FBUyxFQUFFLGNBQWM7U0FDMUIsQ0FDRixDQUFDO1FBRUYsSUFBSSxDQUFDLFVBQVUsR0FBQyxxQkFBcUIsQ0FBQztRQUV0Qyx3REFBd0Q7UUFDeEQsZ0VBQWdFO1FBQ2hFLE1BQU0sd0JBQXdCLEdBQUcsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxDQUFDLHFCQUFxQixDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLGNBQWMsQ0FBQztRQUN0SSxNQUFNLGtCQUFrQixHQUFHLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTLENBQUM7UUFFdEgsTUFBTSxzQkFBc0IsR0FBRyxJQUFJLE9BQU8sQ0FBQyxjQUFjLENBQ3ZELElBQUksRUFDSix5QkFBeUIsRUFDekI7WUFDRSxHQUFHLEVBQUUsSUFBSSxDQUFDLFVBQVU7WUFDcEIsSUFBSSxFQUFFLHFCQUFxQjtTQUM1QixDQUNGLENBQUM7UUFFRixzQkFBc0IsQ0FBQyxjQUFjLENBQ25DLGtDQUFrQyxFQUNsQztZQUNFLFNBQVMsRUFBRSwwQkFBMEI7WUFDckMsUUFBUSxFQUFFLFVBQVU7WUFDcEIsc0JBQXNCLEVBQUUsT0FBTyxDQUFDLGVBQWUsQ0FBQyxVQUFVLENBQ3hEOzs7OztxQkFLVyxDQUNaO1lBQ0QsdUJBQXVCLEVBQUUsT0FBTyxDQUFDLGVBQWUsQ0FBQyxVQUFVLENBQUMsK0JBQStCLENBQUM7U0FDN0YsQ0FFRixDQUFDO1FBRUYsSUFBSSxDQUFDLEtBQUssQ0FBQyxvQkFBb0IsRUFBRTtZQUMvQixJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsbUJBQW1CLEdBQUMsS0FBSyxFQUNyRTtnQkFDRSxZQUFZLEVBQUUsbUJBQW1CLEdBQUMsS0FBSzthQUN4QyxDQUNGLENBQUM7U0FDSDthQUFNO1lBQ0wsSUFBSSxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUMsb0JBQW9CLENBQUM7U0FDaEQ7UUFFRCxtREFBbUQ7UUFDbkQsTUFBTSx1QkFBdUIsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLHdCQUF3QixDQUN0RSxnQ0FBZ0MsR0FBQyxLQUFLLEVBQ3RDLElBQUksQ0FBQyxZQUFZLEVBQ2pCO1lBQ0UsSUFBSSxFQUFFLGdDQUFnQyxHQUFDLEtBQUs7U0FDN0MsQ0FDRixDQUFDO1FBRUYsK0RBQStEO1FBQy9ELE1BQU0sdUJBQXVCLEdBQUcsSUFBSSxNQUFNLENBQUMsbUJBQW1CLENBQzVELElBQUksRUFDSixrQ0FBa0MsR0FBQyxLQUFLLEVBQ3hDO1lBQ0UsSUFBSSxFQUFFLE1BQU0sQ0FBQyxlQUFlLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLDJFQUEyRSxDQUFDLENBQUM7WUFDOUksWUFBWSxFQUFFLG1DQUFtQyxHQUFDLEtBQUs7WUFDdkQsV0FBVyxFQUFFLG9EQUFvRDtZQUNqRSxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7WUFDYixPQUFPLEVBQUUsY0FBYztZQUN2QixVQUFVLEVBQUUsRUFBRSxVQUFVLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxtQkFBbUIsRUFBRTtZQUM5RCxjQUFjLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDO1lBQ3BDLFVBQVUsRUFBRSxJQUFLLEdBQUcsQ0FBQztZQUNyQixPQUFPLEVBQUUsc0JBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1lBQzdCLFdBQVcsRUFBRTtnQkFDWCxXQUFXLEVBQUUsd0JBQXdCO2FBQ3RDO1NBQ0YsQ0FDRixDQUFDO1FBRUYsa0VBQWtFO1FBQ2xFLHVCQUF1QixDQUFDLGVBQWUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDOUQsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixPQUFPLEVBQUU7Z0JBQ1AsaUJBQWlCO2FBQ2xCO1lBQ0QsU0FBUyxFQUFFO2dCQUNULGtCQUFrQixHQUFFLGlCQUFHLENBQUMsTUFBTSxHQUFDLEdBQUcsR0FBQyxpQkFBRyxDQUFDLFVBQVUsR0FBQyxRQUFRLEdBQUMsa0JBQWtCLEdBQUMsSUFBSTthQUNuRjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBQ0osbUZBQW1GO1FBQ25GLHdDQUF3QztRQUN4Qyx1R0FBdUc7UUFDdkcsTUFBTSxnQ0FBZ0MsR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLGtDQUFrQyxFQUFFO1lBQzlGLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxzQkFBc0IsQ0FBQztZQUMzRCxjQUFjLEVBQUU7Z0JBQ2QsK0JBQStCLEVBQUUsSUFBSSxHQUFHLENBQUMsY0FBYyxDQUFDO29CQUN0RCxVQUFVLEVBQUUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7NEJBQ25DLE9BQU8sRUFBRTtnQ0FDUCxxQkFBcUI7Z0NBQ3JCLHNCQUFzQjtnQ0FDdEIsbUJBQW1COzZCQUNwQjs0QkFDRCxTQUFTLEVBQUUsQ0FBQyxPQUFPLGlCQUFHLENBQUMsU0FBUyxTQUFTLGlCQUFHLENBQUMsTUFBTSxJQUFJLGlCQUFHLENBQUMsVUFBVSwwQkFBMEIsQ0FBQzt5QkFDakcsQ0FBQyxDQUFDO2lCQUNKLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQztRQUVILCtGQUErRjtRQUMvRixnQ0FBZ0MsQ0FBQyxXQUFXLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ25FLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsT0FBTyxFQUFFO2dCQUNQLDRCQUE0QjtnQkFDNUIsNEJBQTRCO2dCQUM1Qiw4QkFBOEI7Z0JBQzlCLGdDQUFnQzthQUNqQztZQUNELFNBQVMsRUFBRTtnQkFDVCxjQUFjLEdBQUMsaUJBQUcsQ0FBQyxNQUFNLEdBQUMsR0FBRyxHQUFDLGlCQUFHLENBQUMsVUFBVSxHQUFDLE1BQU07YUFDcEQ7U0FDRixDQUFDLENBQUMsQ0FBQztRQUNKLHVEQUF1RDtRQUN2RCxpR0FBaUc7UUFDakcsZ0NBQWdDLENBQUMsV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUNuRSxNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRTtnQkFDUCwrQkFBK0I7YUFDaEM7WUFDRCxTQUFTLEVBQUU7Z0JBQ1QsR0FBRzthQUNKO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixnQ0FBZ0MsQ0FBQyxXQUFXLENBQzFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRTtnQkFDUCxjQUFjO2dCQUNkLGVBQWU7Z0JBQ2YsZUFBZTtnQkFDZixVQUFVO2FBQ1g7WUFDRCxTQUFTLEVBQUU7Z0JBQ1QsZUFBZSxHQUFHLElBQUksQ0FBQyw0QkFBNEIsRUFBRSxVQUFVO2dCQUMvRCxlQUFlLEdBQUcsSUFBSSxDQUFDLDRCQUE0QixFQUFFLFVBQVUsR0FBRyxJQUFJO2FBQ3ZFO1NBQ0YsQ0FBQyxDQUNILENBQUM7UUFFRixnQ0FBZ0MsQ0FBQyxXQUFXLENBQzFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRSxDQUFDLHVCQUF1QjtnQkFDL0IsVUFBVTtnQkFDVixlQUFlO2dCQUNmLFdBQVc7Z0JBQ1gsa0JBQWtCO2dCQUNsQix1QkFBdUI7Z0JBQ3ZCLHFCQUFxQjtnQkFDckIsNEJBQTRCO2dCQUM1QixjQUFjO2dCQUNkLGVBQWUsQ0FBQztZQUNsQixTQUFTLEVBQUU7Z0JBQ1QsZUFBZSxHQUFHLElBQUksQ0FBQyxnQ0FBZ0MsRUFBRSxVQUFVO2dCQUNuRSxlQUFlLEdBQUcsSUFBSSxDQUFDLGdDQUFnQyxFQUFFLFVBQVUsR0FBRyxJQUFJO2FBQzNFO1NBQ0YsQ0FBQyxDQUNILENBQUM7UUFHRixrRUFBa0U7UUFDbEUsZ0NBQWdDLENBQUMsV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUNuRSxNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRTtnQkFDUCxpQkFBaUI7YUFDbEI7WUFDRCxTQUFTLEVBQUU7Z0JBQ1Qsa0JBQWtCLEdBQUUsaUJBQUcsQ0FBQyxNQUFNLEdBQUMsR0FBRyxHQUFDLGlCQUFHLENBQUMsVUFBVSxHQUFDLFFBQVEsR0FBQyxrQkFBa0IsR0FBQyxJQUFJO2FBQ25GO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSix5QkFBZSxDQUFDLHVCQUF1QixDQUNyQyxnQ0FBZ0MsRUFDaEM7WUFDRTtnQkFDRSxFQUFFLEVBQUUsbUJBQW1CO2dCQUN2QixNQUFNLEVBQUUsc0NBQXNDO2FBQy9DO1NBQ0YsRUFDRCxJQUFJLENBQ0wsQ0FBQztRQUVGLE1BQU0sMkJBQTJCLEdBQUcsSUFBSSxNQUFNLENBQUMsbUJBQW1CLENBQ2hFLElBQUksRUFDSixxQ0FBcUMsR0FBQyxLQUFLLEVBQzNDO1lBQ0UsSUFBSSxFQUFFLE1BQU0sQ0FBQyxlQUFlLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLDhFQUE4RSxDQUFDLENBQUM7WUFDakosWUFBWSxFQUFFLDRCQUE0QixHQUFDLEtBQUs7WUFDaEQsV0FBVyxFQUFFLHNFQUFzRTtZQUNuRixHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7WUFDYixPQUFPLEVBQUUsY0FBYztZQUN2QixVQUFVLEVBQUUsRUFBRSxVQUFVLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxtQkFBbUIsRUFBRTtZQUM5RCxjQUFjLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDO1lBQ3BDLFVBQVUsRUFBRSxJQUFLLEdBQUcsQ0FBQztZQUNyQixPQUFPLEVBQUUsc0JBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1lBQzdCLElBQUksRUFBRSxnQ0FBZ0M7WUFDdEMsV0FBVyxFQUFFO2dCQUNYLFlBQVksRUFBRSxJQUFJLENBQUMsNEJBQTRCLENBQUMsVUFBVTtnQkFDMUQsYUFBYSxFQUFFLElBQUksQ0FBQyxnQ0FBZ0MsQ0FBQyxVQUFVO2dCQUMvRCxXQUFXLEVBQUUsd0JBQXdCO2FBQ3RDO1NBQ0YsQ0FDRixDQUFDO1FBR0YsSUFBSSxRQUFRLEdBQUcsTUFBTSxDQUFDO1FBQ3RCLElBQUksS0FBSyxDQUFDLGdCQUFnQixFQUFFO1lBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVLENBQUM7U0FBQztRQUczRSxNQUFNLDRCQUE0QixHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsOEJBQThCLEVBQUU7WUFDdEYsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO1lBQzNELGNBQWMsRUFBRTtnQkFDZCwrQkFBK0IsRUFBRSxJQUFJLEdBQUcsQ0FBQyxjQUFjLENBQUM7b0JBQ3RELFVBQVUsRUFBRSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQzs0QkFDbkMsT0FBTyxFQUFFO2dDQUNQLHFCQUFxQjtnQ0FDckIsc0JBQXNCO2dDQUN0QixtQkFBbUI7NkJBQ3BCOzRCQUNELFNBQVMsRUFBRSxDQUFDLE9BQU8saUJBQUcsQ0FBQyxTQUFTLFNBQVMsaUJBQUcsQ0FBQyxNQUFNLElBQUksaUJBQUcsQ0FBQyxVQUFVLDBCQUEwQixDQUFDO3lCQUNqRyxDQUFDLENBQUM7aUJBQ0osQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsK0ZBQStGO1FBQy9GLDRCQUE0QixDQUFDLFdBQVcsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDL0QsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixPQUFPLEVBQUU7Z0JBQ1AsNEJBQTRCO2dCQUM1Qiw0QkFBNEI7Z0JBQzVCLDhCQUE4QjtnQkFDOUIsZ0NBQWdDO2FBQ2pDO1lBQ0QsU0FBUyxFQUFFO2dCQUNULGNBQWMsR0FBQyxpQkFBRyxDQUFDLE1BQU0sR0FBQyxHQUFHLEdBQUMsaUJBQUcsQ0FBQyxVQUFVLEdBQUMsTUFBTTthQUNwRDtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBQ0osdURBQXVEO1FBQ3ZELGlHQUFpRztRQUNqRyw0QkFBNEIsQ0FBQyxXQUFXLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQy9ELE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsT0FBTyxFQUFFO2dCQUNQLCtCQUErQjthQUNoQztZQUNELFNBQVMsRUFBRTtnQkFDVCxHQUFHO2FBQ0o7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLDRCQUE0QixDQUFDLFdBQVcsQ0FDdEMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3RCLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsT0FBTyxFQUFFO2dCQUNQLGNBQWM7Z0JBQ2QsZUFBZTtnQkFDZixlQUFlO2dCQUNmLFVBQVU7YUFDWDtZQUNELFNBQVMsRUFBRTtnQkFDVCxlQUFlLEdBQUcsSUFBSSxDQUFDLGdDQUFnQyxFQUFFLFVBQVU7Z0JBQ25FLGVBQWUsR0FBRyxJQUFJLENBQUMsZ0NBQWdDLEVBQUUsVUFBVSxHQUFHLElBQUk7YUFDM0U7U0FDRixDQUFDLENBQ0gsQ0FBQztRQUVGLDRCQUE0QixDQUFDLFdBQVcsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDL0QsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixPQUFPLEVBQUUsQ0FBQyxNQUFNLENBQUM7WUFDakIsU0FBUyxFQUFFO2dCQUNULGFBQWEsR0FBQyxpQkFBRyxDQUFDLE1BQU0sR0FBQyxHQUFHLEdBQUMsaUJBQUcsQ0FBQyxVQUFVLEdBQUMsVUFBVSxHQUFDLEtBQUssQ0FBQyx3QkFBd0IsQ0FBQyxVQUFVLEdBQUMsSUFBSTtnQkFDckcsYUFBYSxHQUFDLGlCQUFHLENBQUMsTUFBTSxHQUFDLEdBQUcsR0FBQyxpQkFBRyxDQUFDLFVBQVUsR0FBQyxVQUFVLEdBQUMsS0FBSyxDQUFDLHdCQUF3QixDQUFDLFVBQVU7YUFDakc7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLHlFQUF5RTtRQUN6RSw0QkFBNEIsQ0FBQyxXQUFXLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQy9ELE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsT0FBTyxFQUFFLENBQUMsV0FBVyxDQUFDO1lBQ3RCLFNBQVMsRUFBRTtnQkFDVCxrQkFBa0IsR0FBQyxpQkFBRyxDQUFDLE1BQU0sR0FBQyxvQkFBb0I7Z0JBQ2xELGtCQUFrQixHQUFDLGlCQUFHLENBQUMsTUFBTSxHQUFDLHNCQUFzQjthQUNyRDtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUoseUJBQWUsQ0FBQyx1QkFBdUIsQ0FDckMsNEJBQTRCLEVBQzVCO1lBQ0U7Z0JBQ0UsRUFBRSxFQUFFLG1CQUFtQjtnQkFDdkIsTUFBTSxFQUFFLHNDQUFzQzthQUMvQztTQUNGLEVBQ0QsSUFBSSxDQUNMLENBQUM7UUFHRixvREFBb0Q7UUFDcEQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEVBQUU7WUFBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxDQUFDLDRCQUE0QixDQUFDLENBQUM7U0FBQztRQUM3RiwrQ0FBK0M7UUFDL0MsTUFBTSx1QkFBdUIsR0FBRyxJQUFJLE1BQU0sQ0FBQyxtQkFBbUIsQ0FDNUQsSUFBSSxFQUNKLGdDQUFnQyxHQUFDLEtBQUssRUFDdEM7WUFDRSxJQUFJLEVBQUUsTUFBTSxDQUFDLGVBQWUsQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUseUVBQXlFLENBQUMsQ0FBQztZQUM1SSxZQUFZLEVBQUUsdUJBQXVCLEdBQUMsS0FBSztZQUMzQyxXQUFXLEVBQUUsaUdBQWlHO1lBQzlHLEdBQUcsRUFBRSxJQUFJLENBQUMsR0FBRztZQUNiLE9BQU8sRUFBRSxjQUFjO1lBQ3ZCLFVBQVUsRUFBRSxFQUFFLFVBQVUsRUFBRSxHQUFHLENBQUMsVUFBVSxDQUFDLG1CQUFtQixFQUFFO1lBQzlELGNBQWMsRUFBRSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUM7WUFDcEMsVUFBVSxFQUFFLElBQUssR0FBRyxDQUFDO1lBQ3JCLE9BQU8sRUFBRSxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7WUFDN0IsSUFBSSxFQUFFLDRCQUE0QjtZQUNsQyxXQUFXLEVBQUU7Z0JBQ1gsYUFBYSxFQUFFLElBQUksQ0FBQyxnQ0FBZ0MsQ0FBQyxVQUFVO2dCQUMvRCxXQUFXLEVBQUUsd0JBQXdCO2dCQUNyQyxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsbUJBQW1CO2dCQUMzQywwQkFBMEIsRUFBRSxLQUFLLENBQUMsd0JBQXdCLENBQUMsY0FBYztnQkFDekUsb0JBQW9CLEVBQUUsUUFBUTthQUMvQjtTQUNGLENBQ0YsQ0FBQztRQUdGLE1BQU0sdUJBQXVCLEdBQUcsS0FBSyxDQUFDLHVCQUF1QixJQUFJLElBQUksQ0FBQztRQUN0RSxNQUFNLFdBQVcsR0FBRyxZQUFZLEdBQUMsRUFBRSxDQUFDO1FBRXBDLElBQUksdUJBQXVCLEVBQUU7WUFDM0IsdUJBQXVCLENBQUMsY0FBYyxDQUNwQyxtQkFBbUIsRUFBRSxXQUFXLENBQ2pDLENBQUM7WUFDRiwyQkFBMkIsQ0FBQyxjQUFjLENBQ3hDLG1CQUFtQixFQUFFLFdBQVcsQ0FDakMsQ0FBQztZQUNGLHVCQUF1QixDQUFDLGNBQWMsQ0FDcEMsbUJBQW1CLEVBQUUsV0FBVyxDQUNqQyxDQUFDO1NBQ0g7UUFBQSxDQUFDO1FBRUYsa0VBQWtFO1FBQ2xFLHVCQUF1QixDQUFDLGVBQWUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDOUQsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixPQUFPLEVBQUU7Z0JBQ1AsaUJBQWlCO2FBQ2xCO1lBQ0QsU0FBUyxFQUFFO2dCQUNULGtCQUFrQixHQUFFLGlCQUFHLENBQUMsTUFBTSxHQUFDLEdBQUcsR0FBQyxpQkFBRyxDQUFDLFVBQVUsR0FBQyxRQUFRLEdBQUMsa0JBQWtCLEdBQUMsSUFBSTthQUNuRjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosMkJBQTJCO1FBQzNCLE1BQU0sbUJBQW1CLEdBQUcsSUFBSSxXQUFXLENBQUMsWUFBWSxDQUN0RCxJQUFJLEVBQ0osMEJBQTBCLEVBQzFCO1lBQ0UsY0FBYyxFQUFFLHVCQUF1QjtZQUN2QyxVQUFVLEVBQUUscUJBQXFCO1NBQ2xDLENBQ0YsQ0FBQztRQUVGLE1BQU0sc0JBQXNCLEdBQUcsSUFBSSxXQUFXLENBQUMsWUFBWSxDQUN6RCxJQUFJLEVBQ0osNkNBQTZDLEVBQzdDO1lBQ0UsY0FBYyxFQUFFLDJCQUEyQjtZQUMzQyxVQUFVLEVBQUUseUJBQXlCO1NBQ3RDLENBQ0YsQ0FBQztRQUVGLE1BQU0sY0FBYyxHQUFHLElBQUksV0FBVyxDQUFDLFlBQVksQ0FDakQsSUFBSSxFQUNKLDZEQUE2RCxFQUM3RDtZQUNFLGNBQWMsRUFBRSx1QkFBdUI7WUFDdkMsVUFBVSxFQUFFLEdBQUc7U0FDaEIsQ0FDRixDQUFDO1FBRUYsTUFBTSxxQkFBcUIsR0FBRyxJQUFJLE1BQU0sQ0FBQyxNQUFNLENBQzdDLElBQUksRUFDSixnQ0FBZ0MsRUFDaEM7WUFDRSxVQUFVLEVBQUUsbUNBQW1DO1NBQ2hELENBQ0YsQ0FBQztRQUVGLE1BQU0scUJBQXFCLEdBQUcsSUFBSSxNQUFNLENBQUMsR0FBRyxDQUMxQyxJQUFJLEVBQ0osV0FBVyxFQUNYO1lBQ0UsY0FBYyxFQUFFLEdBQUc7U0FDcEIsQ0FDRixDQUFDLFFBQVEsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBRW5DLE1BQU0sU0FBUyxHQUFHLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFO1lBQ3BELEtBQUssRUFBRSx1QkFBdUI7WUFDOUIsS0FBSyxFQUFFLDZCQUE2QjtTQUNyQyxDQUFDLENBQUM7UUFFSCxNQUFNLFVBQVUsR0FBRyxtQkFBbUIsQ0FBQyxJQUFJLENBQUMscUJBQXFCLENBQUMsSUFBSSxDQUNwRSxNQUFNLENBQUMsU0FBUyxDQUFDLGFBQWEsQ0FBQyxxQ0FBcUMsRUFBRSxLQUFLLENBQUMsRUFBRSxTQUFTLENBQUMsQ0FBQyxTQUFTLENBQUMscUJBQXFCLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUVsSixNQUFNLHVCQUF1QixHQUFHLElBQUksTUFBTSxDQUFDLFlBQVksQ0FDckQsSUFBSSxFQUNKLHVCQUF1QixFQUN2QjtZQUNFLGdCQUFnQixFQUFFLHVCQUF1QixHQUFDLEtBQUs7WUFDL0MsY0FBYyxFQUFFLE1BQU0sQ0FBQyxjQUFjLENBQUMsYUFBYSxDQUFDLFVBQVUsQ0FBQztZQUMvRCxPQUFPLEVBQUUsc0JBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1lBQzdCLElBQUksRUFBRTtnQkFDSixXQUFXLEVBQUUsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSwrQkFBK0IsQ0FBQztnQkFDckUsS0FBSyxFQUFFLE1BQU0sQ0FBQyxRQUFRLENBQUMsR0FBRzthQUMzQjtZQUNELGNBQWMsRUFBRSxXQUFXO1NBQzVCLENBQ0YsQ0FBQztRQUVGLElBQUksQ0FBQyxZQUFZLEdBQUMsdUJBQXVCLENBQUM7UUFFMUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxnQkFBZ0IsQ0FBQyx1QkFBdUIsQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUUzRSx1QkFBdUIsQ0FBQyxjQUFjLENBQ3BDLHdCQUF3QixFQUN4QjtZQUNFLFNBQVMsRUFBRSxpQkFBaUI7WUFDNUIsUUFBUSxFQUFFLFVBQVU7WUFDcEIsc0JBQXNCLEVBQUUsT0FBTyxDQUFDLGVBQWUsQ0FBQyxVQUFVLENBQ3hEOzs7Ozs7Ozs7OztxQkFXVyxDQUNaO1lBQ0QsdUJBQXVCLEVBQUUsT0FBTyxDQUFDLGVBQWUsQ0FBQyxVQUFVLENBQ3pEOzs7OztxQkFLVyxDQUNaO1NBQ0YsQ0FDRixDQUFDO1FBRUYsTUFBTSxJQUFJLEdBQUcsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUMxQixJQUFJLEVBQ0osZUFBZSxHQUFDLEtBQUssRUFDckI7WUFDRSxXQUFXLEVBQUUsb0NBQW9DO1lBQ2pELFFBQVEsRUFBRSxJQUFJLENBQUMsWUFBWTtZQUMzQixZQUFZLEVBQUU7Z0JBQ1osTUFBTSxFQUFFLENBQUMsV0FBVyxDQUFDO2FBQ3RCO1NBQ0YsQ0FDRixDQUFDO1FBRUYsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLE9BQU8sQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUM7SUFFakUsQ0FBQzs7OztBQTNxQlUsZ0VBQTBCIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiAgQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBEdXJhdGlvbiwgQXdzIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0ICogYXMgYXBwc3luYyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtYXBwc3luYyc7XG5pbXBvcnQgKiBhcyBjb2duaXRvIGZyb20gJ2F3cy1jZGstbGliL2F3cy1jb2duaXRvJztcbmltcG9ydCAqIGFzIGVjMiBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZWMyJztcbmltcG9ydCAqIGFzIGV2ZW50cyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZXZlbnRzJztcbmltcG9ydCAqIGFzIHRhcmdldHMgZnJvbSAnYXdzLWNkay1saWIvYXdzLWV2ZW50cy10YXJnZXRzJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIGxvZ3MgZnJvbSAnYXdzLWNkay1saWIvYXdzLWxvZ3MnO1xuaW1wb3J0ICogYXMgb3BlbnNlYXJjaHNlcnZpY2UgZnJvbSAnYXdzLWNkay1saWIvYXdzLW9wZW5zZWFyY2hzZXJ2aWNlJztcbmltcG9ydCAqIGFzIHMzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1zMyc7XG5pbXBvcnQgKiBhcyBzZWNyZXQgZnJvbSAnYXdzLWNkay1saWIvYXdzLXNlY3JldHNtYW5hZ2VyJztcbmltcG9ydCAqIGFzIHN0ZXBmbiBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc3RlcGZ1bmN0aW9ucyc7XG5pbXBvcnQgKiBhcyBzdGVwZm5fdGFzayBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc3RlcGZ1bmN0aW9ucy10YXNrcyc7XG5pbXBvcnQgeyBOYWdTdXBwcmVzc2lvbnMgfSBmcm9tICdjZGstbmFnJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0ICogYXMgczNfYnVja2V0X2hlbHBlciBmcm9tICcuLi8uLi8uLi9jb21tb24vaGVscGVycy9zMy1idWNrZXQtaGVscGVyJztcbmltcG9ydCAqIGFzIHZwY19oZWxwZXIgZnJvbSAnLi4vLi4vLi4vY29tbW9uL2hlbHBlcnMvdnBjLWhlbHBlcic7XG5cbi8qKlxuICogVGhlIHByb3BlcnRpZXMgZm9yIHRoZSBSYWdBcHBzeW5jU3RlcGZuT3BlbnNlYXJjaCBjbGFzcy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBSYWdBcHBzeW5jU3RlcGZuT3BlbnNlYXJjaFByb3BzIHtcbiAgLyoqXG4gICAqIE9wdGlvbmFsIGN1c3RvbSBwcm9wZXJ0aWVzIGZvciBhIFZQQyB0aGUgY29uc3RydWN0IHdpbGwgY3JlYXRlLiBUaGlzIFZQQyB3aWxsXG4gICAqIGJlIHVzZWQgYnkgdGhlIExhbWJkYSBmdW5jdGlvbnMgdGhlIGNvbnN0cnVjdCBjcmVhdGVzLiBQcm92aWRpbmdcbiAgICogYm90aCB0aGlzIGFuZCBleGlzdGluZ1ZwYyBpcyBhbiBlcnJvci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBub25lXG4gICAqL1xuICByZWFkb25seSB2cGNQcm9wcz86IGVjMi5WcGNQcm9wcztcbiAgLyoqXG4gICAqIE9wdGlvbmFsIEFuIGV4aXN0aW5nIFZQQyBpbiB3aGljaCB0byBkZXBsb3kgdGhlIGNvbnN0cnVjdC4gUHJvdmlkaW5nIGJvdGggdGhpcyBhbmRcbiAgICogdnBjUHJvcHMgaXMgYW4gZXJyb3IuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgZXhpc3RpbmdWcGM/OiBlYzIuSVZwYztcbiAgLyoqXG4gICAqIE9wdGlvbmFsIGV4aXN0aW5nIHNlY3VyaXR5IGdyb3VwIGFsbG93aW5nIGFjY2VzcyB0byBvcGVuc2VhcmNoLiBVc2VkIGJ5IHRoZSBsYW1iZGEgZnVuY3Rpb25zXG4gICAqIGJ1aWx0IGJ5IHRoaXMgY29uc3RydWN0LiBJZiBub3QgcHJvdmlkZWQsIHRoZSBjb25zdHJ1Y3Qgd2lsbCBjcmVhdGUgb25lLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGV4aXN0aW5nU2VjdXJpdHlHcm91cD86IGVjMi5JU2VjdXJpdHlHcm91cDtcbiAgLyoqXG4gICAqIE9wdGlvbmFsIEV4aXN0aW5nIGluc3RhbmNlIG9mIGFuIEV2ZW50QnJpZGdlIGJ1cy4gSWYgbm90IHByb3ZpZGVkLCB0aGUgY29uc3RydWN0IHdpbGwgY3JlYXRlIG9uZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lXG4gICAqL1xuICByZWFkb25seSBleGlzdGluZ0J1c0ludGVyZmFjZT86IGV2ZW50cy5JRXZlbnRCdXM7XG4gIC8qKlxuICAgKiBFeGlzdGluZyBpbnN0YW5jZSBvZiBTMyBCdWNrZXQgb2JqZWN0LCBwcm92aWRpbmcgYm90aCB0aGlzIGFuZCBgYnVja2V0SW5wdXRzQXNzZXRzUHJvcHNgIHdpbGwgY2F1c2UgYW4gZXJyb3IuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgZXhpc3RpbmdJbnB1dEFzc2V0c0J1Y2tldE9iaj86IHMzLklCdWNrZXQ7XG4gIC8qKlxuICAgKiBPcHRpb25hbCB1c2VyIHByb3ZpZGVkIHByb3BzIHRvIG92ZXJyaWRlIHRoZSBkZWZhdWx0IHByb3BzIGZvciB0aGUgUzMgQnVja2V0LlxuICAgKiBQcm92aWRpbmcgYm90aCB0aGlzIGFuZCBgZXhpc3RpbmdJbnB1dEFzc2V0c0J1Y2tldE9iamAgd2lsbCBjYXVzZSBhbiBlcnJvci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBEZWZhdWx0IHByb3BzIGFyZSB1c2VkXG4gICAqL1xuICByZWFkb25seSBidWNrZXRJbnB1dHNBc3NldHNQcm9wcz86IHMzLkJ1Y2tldFByb3BzO1xuICAvKipcbiAgICogRXhpc3RpbmcgaW5zdGFuY2Ugb2YgUzMgQnVja2V0IG9iamVjdCwgcHJvdmlkaW5nIGJvdGggdGhpcyBhbmQgYGJ1Y2tldFByb2Nlc3NlZEFzc2V0c1Byb3BzYCB3aWxsIGNhdXNlIGFuIGVycm9yLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGV4aXN0aW5nUHJvY2Vzc2VkQXNzZXRzQnVja2V0T2JqPzogczMuSUJ1Y2tldDtcbiAgLyoqXG4gICAqIE9wdGlvbmFsIHVzZXIgcHJvdmlkZWQgcHJvcHMgdG8gb3ZlcnJpZGUgdGhlIGRlZmF1bHQgcHJvcHMgZm9yIHRoZSBTMyBCdWNrZXQuXG4gICAqIFByb3ZpZGluZyBib3RoIHRoaXMgYW5kIGBleGlzdGluZ1Byb2Nlc3NlZEFzc2V0c0J1Y2tldE9iamAgd2lsbCBjYXVzZSBhbiBlcnJvci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBEZWZhdWx0IHByb3BzIGFyZSB1c2VkXG4gICAqL1xuICByZWFkb25seSBidWNrZXRQcm9jZXNzZWRBc3NldHNQcm9wcz86IHMzLkJ1Y2tldFByb3BzO1xuICAvKipcbiAgICAgKiBFeGlzdGluZyBBbWF6b24gT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpbi5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgICAqL1xuICByZWFkb25seSBleGlzdGluZ09wZW5zZWFyY2hEb21haW46IG9wZW5zZWFyY2hzZXJ2aWNlLklEb21haW47XG4gIC8qKlxuICAgKiBJbmRleCBuYW1lIGZvciB0aGUgT3BlblNlYXJjaCBTZXJ2aWNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IG9wZW5TZWFyY2hJbmRleE5hbWU6IHN0cmluZztcbiAgLyoqXG4gICAgICogT3B0aW9uYWwuIFNlY3JldHNNYW5hZ2VyIHNlY3JldCB0byBhdXRoZW50aWNhdGUgYWdhaW5zdCB0aGUgT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpbiBpZlxuICAgICAqIGRvbWFpbiBpcyBjb25maWd1cmVkIHdpdGggVXNlcm5hbWUvUGFzc3dvcmQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICAgKi9cbiAgcmVhZG9ubHkgb3BlblNlYXJjaFNlY3JldD86IHNlY3JldC5JU2VjcmV0O1xuICAvKipcbiAgICogRXhpc3RpbmcgbWVyZ2VkIEFwcHN5bmMgR3JhcGhRTCBhcGkuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgZXhpc3RpbmdNZXJnZWRBcGk/OiBhcHBzeW5jLkNmbkdyYXBoUUxBcGk7XG4gIC8qKlxuICAgKiBDb2duaXRvIHVzZXIgcG9vbCB1c2VkIGZvciBhdXRoZW50aWNhdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lXG4gICAqL1xuICByZWFkb25seSBjb2duaXRvVXNlclBvb2w6IGNvZ25pdG8uSVVzZXJQb29sO1xuICAvKipcbiAgICogVmFsdWUgd2lsbCBiZSBhcHBlbmRlZCB0byByZXNvdXJjZXMgbmFtZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBfZGV2XG4gICAqL1xuICByZWFkb25seSBzdGFnZT86IHN0cmluZztcblxuICAvKipcbiAgICogT3B0aW9uYWwuQ0RLIGNvbnN0cnVjdHMgcHJvdmlkZWQgY29sbGVjdHMgYW5vbnltb3VzIG9wZXJhdGlvbmFsXG4gICAqIG1ldHJpY3MgdG8gaGVscCBBV1MgaW1wcm92ZSB0aGUgcXVhbGl0eSBhbmQgZmVhdHVyZXMgb2YgdGhlXG4gICAqIGNvbnN0cnVjdHMuIERhdGEgY29sbGVjdGlvbiBpcyBzdWJqZWN0IHRvIHRoZSBBV1MgUHJpdmFjeSBQb2xpY3lcbiAgICogKGh0dHBzOi8vYXdzLmFtYXpvbi5jb20vcHJpdmFjeS8pLiBUbyBvcHQgb3V0IG9mIHRoaXMgZmVhdHVyZSxcbiAgICogc2ltcGx5IGRpc2FibGUgaXQgYnkgc2V0dGluZyB0aGUgY29uc3RydWN0IHByb3BlcnR5XG4gICAqIFwiZW5hYmxlT3BlcmF0aW9uYWxNZXRyaWNcIiB0byBmYWxzZSBmb3IgZWFjaCBjb25zdHJ1Y3QgdXNlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLXRydWVcbiAgICovXG4gIHJlYWRvbmx5IGVuYWJsZU9wZXJhdGlvbmFsTWV0cmljPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogRW5hYmxlIG9ic2VydmFiaWxpdHkuIFdhcm5pbmc6IGFzc29jaWF0ZWQgY29zdCB3aXRoIHRoZSBzZXJ2aWNlc1xuICAgKiB1c2VkLiBCZXN0IHByYWN0aWNlIHRvIGVuYWJsZSBieSBkZWZhdWx0LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHRydWVcbiAgICovXG4gIHJlYWRvbmx5IG9ic2VydmFiaWxpdHk/OiBib29sZWFuO1xufVxuXG4vKipcbiAgICogQHN1bW1hcnkgVGhlIFJhZ0FwaUdhdGV3YXlPcGVuc2VhcmNoIGNsYXNzLlxuICAgKi9cbmV4cG9ydCBjbGFzcyBSYWdBcHBzeW5jU3RlcGZuT3BlbnNlYXJjaCBleHRlbmRzIENvbnN0cnVjdCB7XG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBpbnN0YW5jZSBvZiBlYzIuSVZwYyB1c2VkIGJ5IHRoZSBjb25zdHJ1Y3RcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB2cGM6IGVjMi5JVnBjO1xuICAvKipcbiAgICogUmV0dXJucyB0aGUgaW5zdGFuY2Ugb2YgZWMyLklTZWN1cml0eUdyb3VwIHVzZWQgYnkgdGhlIGNvbnN0cnVjdFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXA6IGVjMi5JU2VjdXJpdHlHcm91cDtcbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIGluc3RhbmNlIG9mIGV2ZW50cy5JRXZlbnRCdXMgdXNlZCBieSB0aGUgY29uc3RydWN0XG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgaW5nZXN0aW9uQnVzOiBldmVudHMuSUV2ZW50QnVzO1xuICAvKipcbiAgICogUmV0dXJucyBhbiBpbnN0YW5jZSBvZiBzMy5JQnVja2V0IGNyZWF0ZWQgYnkgdGhlIGNvbnN0cnVjdFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHMzSW5wdXRBc3NldHNCdWNrZXRJbnRlcmZhY2U6IHMzLklCdWNrZXQ7XG4gIC8qKlxuICAgKiBSZXR1cm5zIGFuIGluc3RhbmNlIG9mIHMzLkJ1Y2tldCBjcmVhdGVkIGJ5IHRoZSBjb25zdHJ1Y3QuXG4gICAqIElNUE9SVEFOVDogSWYgZXhpc3RpbmdJbnB1dEFzc2V0c0J1Y2tldE9iaiB3YXMgcHJvdmlkZWQgaW4gUGF0dGVybiBDb25zdHJ1Y3QgUHJvcHMsXG4gICAqIHRoaXMgcHJvcGVydHkgd2lsbCBiZSB1bmRlZmluZWRcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBzM0lucHV0QXNzZXRzQnVja2V0PzogczMuQnVja2V0O1xuICAvKipcbiAgICogUmV0dXJucyBhbiBpbnN0YW5jZSBvZiBzMy5JQnVja2V0IGNyZWF0ZWQgYnkgdGhlIGNvbnN0cnVjdFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHMzUHJvY2Vzc2VkQXNzZXRzQnVja2V0SW50ZXJmYWNlOiBzMy5JQnVja2V0O1xuICAvKipcbiAgICogUmV0dXJucyBhbiBpbnN0YW5jZSBvZiBzMy5CdWNrZXQgY3JlYXRlZCBieSB0aGUgY29uc3RydWN0LlxuICAgKiBJTVBPUlRBTlQ6IElmIGV4aXN0aW5nUHJvY2Vzc2VkQXNzZXRzQnVja2V0T2JqIHdhcyBwcm92aWRlZCBpbiBQYXR0ZXJuIENvbnN0cnVjdCBQcm9wcyxcbiAgICogdGhpcyBwcm9wZXJ0eSB3aWxsIGJlIHVuZGVmaW5lZFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHMzUHJvY2Vzc2VkQXNzZXRzQnVja2V0PzogczMuQnVja2V0O1xuICAvKipcbiAgICogUmV0dXJucyBhbiBpbnN0YW5jZSBvZiBhcHBzeW5jLklHcmFwaHFsQXBpIGNyZWF0ZWQgYnkgdGhlIGNvbnN0cnVjdFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGdyYXBocWxBcGk6IGFwcHN5bmMuSUdyYXBocWxBcGk7XG4gIC8qKlxuICAgKiBSZXR1cm5zIGFuIGluc3RhbmNlIG9mIHN0ZXBmbi5TdGF0ZU1hY2hpbmUgY3JlYXRlZCBieSB0aGUgY29uc3RydWN0XG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgc3RhdGVNYWNoaW5lOiBzdGVwZm4uU3RhdGVNYWNoaW5lO1xuXG4gIC8qKlxuICAgICAqIEBzdW1tYXJ5IENvbnN0cnVjdHMgYSBuZXcgaW5zdGFuY2Ugb2YgdGhlIFJhZ0FwcHN5bmNTdGVwZm5PcGVuc2VhcmNoIGNsYXNzLlxuICAgICAqIEBwYXJhbSB7Y2RrLkFwcH0gc2NvcGUgLSByZXByZXNlbnRzIHRoZSBzY29wZSBmb3IgYWxsIHRoZSByZXNvdXJjZXMuXG4gICAgICogQHBhcmFtIHtzdHJpbmd9IGlkIC0gdGhpcyBpcyBhIGEgc2NvcGUtdW5pcXVlIGlkLlxuICAgICAqIEBwYXJhbSB7UmFnQXBwc3luY1N0ZXBmbk9wZW5zZWFyY2hQcm9wc30gcHJvcHMgLSB1c2VyIHByb3ZpZGVkIHByb3BzIGZvciB0aGUgY29uc3RydWN0LlxuICAgICAqIEBzaW5jZSAwLjAuMFxuICAgICAqIEBhY2Nlc3MgcHVibGljXG4gICAgICovXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBSYWdBcHBzeW5jU3RlcGZuT3BlbnNlYXJjaFByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIC8vIHN0YWdlXG4gICAgbGV0IHN0YWdlID0gJy1kZXYnO1xuICAgIGlmIChwcm9wcz8uc3RhZ2UpIHtcbiAgICAgIHN0YWdlID0gcHJvcHMuc3RhZ2U7XG4gICAgfVxuXG4gICAgLy8gb2JzZXJ2YWJpbGl0eVxuICAgIGxldCBsYW1iZGFfdHJhY2luZyA9IGxhbWJkYS5UcmFjaW5nLkFDVElWRTtcbiAgICBsZXQgZW5hYmxlX3hyYXkgPSB0cnVlO1xuICAgIGxldCBhcGlfbG9nX2NvbmZpZyA9IHtcbiAgICAgIGZpZWxkTG9nTGV2ZWw6IGFwcHN5bmMuRmllbGRMb2dMZXZlbC5BTEwsXG4gICAgICByZXRlbnRpb246IGxvZ3MuUmV0ZW50aW9uRGF5cy5URU5fWUVBUlMsXG4gICAgfTtcbiAgICBpZiAocHJvcHMub2JzZXJ2YWJpbGl0eSA9PSBmYWxzZSkge1xuICAgICAgZW5hYmxlX3hyYXkgPSBmYWxzZTtcbiAgICAgIGxhbWJkYV90cmFjaW5nID0gbGFtYmRhLlRyYWNpbmcuRElTQUJMRUQ7XG4gICAgICBhcGlfbG9nX2NvbmZpZyA9IHtcbiAgICAgICAgZmllbGRMb2dMZXZlbDogYXBwc3luYy5GaWVsZExvZ0xldmVsLk5PTkUsXG4gICAgICAgIHJldGVudGlvbjogbG9ncy5SZXRlbnRpb25EYXlzLlRFTl9ZRUFSUyxcbiAgICAgIH07XG4gICAgfTtcblxuICAgIHZwY19oZWxwZXIuQ2hlY2tWcGNQcm9wcyhwcm9wcyk7XG4gICAgczNfYnVja2V0X2hlbHBlci5DaGVja1MzUHJvcHMoe1xuICAgICAgZXhpc3RpbmdCdWNrZXRPYmo6IHByb3BzLmV4aXN0aW5nSW5wdXRBc3NldHNCdWNrZXRPYmosXG4gICAgICBidWNrZXRQcm9wczogcHJvcHMuYnVja2V0SW5wdXRzQXNzZXRzUHJvcHMsXG4gICAgfSk7XG4gICAgczNfYnVja2V0X2hlbHBlci5DaGVja1MzUHJvcHMoe1xuICAgICAgZXhpc3RpbmdCdWNrZXRPYmo6IHByb3BzLmV4aXN0aW5nUHJvY2Vzc2VkQXNzZXRzQnVja2V0T2JqLFxuICAgICAgYnVja2V0UHJvcHM6IHByb3BzLmJ1Y2tldFByb2Nlc3NlZEFzc2V0c1Byb3BzLFxuICAgIH0pO1xuXG4gICAgaWYgKHByb3BzPy5leGlzdGluZ1ZwYykge1xuICAgICAgdGhpcy52cGMgPSBwcm9wcy5leGlzdGluZ1ZwYztcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy52cGMgPSBuZXcgZWMyLlZwYyh0aGlzLCAnVnBjJywgcHJvcHMudnBjUHJvcHMpO1xuICAgIH1cblxuICAgIC8vIFNlY3VyaXR5IGdyb3VwXG4gICAgaWYgKHByb3BzPy5leGlzdGluZ1NlY3VyaXR5R3JvdXApIHtcbiAgICAgIHRoaXMuc2VjdXJpdHlHcm91cCA9IHByb3BzLmV4aXN0aW5nU2VjdXJpdHlHcm91cDtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy5zZWN1cml0eUdyb3VwID0gbmV3IGVjMi5TZWN1cml0eUdyb3VwKFxuICAgICAgICB0aGlzLFxuICAgICAgICAnc2VjdXJpdHlHcm91cCcsXG4gICAgICAgIHtcbiAgICAgICAgICB2cGM6IHRoaXMudnBjLFxuICAgICAgICAgIGFsbG93QWxsT3V0Ym91bmQ6IHRydWUsXG4gICAgICAgICAgc2VjdXJpdHlHcm91cE5hbWU6ICdzZWN1cml0eUdyb3VwJytzdGFnZSxcbiAgICAgICAgfSxcbiAgICAgICk7XG4gICAgfVxuXG4gICAgLy8gdnBjIGZsb3dsb2dncm91cFxuICAgIGNvbnN0IGxvZ0dyb3VwID0gbmV3IGxvZ3MuTG9nR3JvdXAodGhpcywgJ2luZ2VzdGlvbkNvbnN0cnVjdExvZ0dyb3VwJyk7XG4gICAgY29uc3Qgcm9sZSA9IG5ldyBpYW0uUm9sZSh0aGlzLCAnaW5nZXN0aW9uQ29uc3RydWN0Um9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCd2cGMtZmxvdy1sb2dzLmFtYXpvbmF3cy5jb20nKSxcbiAgICB9KTtcblxuICAgIC8vIHZwYyBmbG93bG9nc1xuICAgIG5ldyBlYzIuRmxvd0xvZyh0aGlzLCAnRmxvd0xvZycsIHtcbiAgICAgIHJlc291cmNlVHlwZTogZWMyLkZsb3dMb2dSZXNvdXJjZVR5cGUuZnJvbVZwYyh0aGlzLnZwYyksXG4gICAgICBkZXN0aW5hdGlvbjogZWMyLkZsb3dMb2dEZXN0aW5hdGlvbi50b0Nsb3VkV2F0Y2hMb2dzKGxvZ0dyb3VwLCByb2xlKSxcbiAgICB9KTtcblxuICAgIC8vIGJ1Y2tldCBmb3Igc3RvcmluZyBzZXJ2ZXIgYWNjZXNzIGxvZ2dpbmdcbiAgICBjb25zdCBzZXJ2ZXJBY2Nlc3NMb2dCdWNrZXQgPSBuZXcgczMuQnVja2V0KHRoaXMsXG4gICAgICAnc2VydmVyQWNjZXNzTG9nQnVja2V0JytzdGFnZSxcbiAgICAgIHtcbiAgICAgICAgYmxvY2tQdWJsaWNBY2Nlc3M6IHMzLkJsb2NrUHVibGljQWNjZXNzLkJMT0NLX0FMTCxcbiAgICAgICAgZW5jcnlwdGlvbjogczMuQnVja2V0RW5jcnlwdGlvbi5TM19NQU5BR0VELFxuICAgICAgICB2ZXJzaW9uZWQ6IHRydWUsXG4gICAgICAgIGxpZmVjeWNsZVJ1bGVzOiBbe1xuICAgICAgICAgIGV4cGlyYXRpb246IER1cmF0aW9uLmRheXMoOTApLFxuICAgICAgICB9XSxcbiAgICAgIH0pO1xuXG4gICAgLy8gQnVja2V0IGNvbnRhaW5pbmcgdGhlIGlucHV0cyBhc3NldHMgKGRvY3VtZW50cyAtIG11bHRpcGxlIG1vZGFsaXRpZXMpIHVwbG9hZGVkIGJ5IHRoZSB1c2VyXG4gICAgbGV0IGlucHV0QXNzZXRzQnVja2V0OiBzMy5JQnVja2V0O1xuXG4gICAgaWYgKCFwcm9wcy5leGlzdGluZ0lucHV0QXNzZXRzQnVja2V0T2JqKSB7XG4gICAgICBsZXQgdG1wQnVja2V0OiBzMy5CdWNrZXQ7XG4gICAgICBpZiAoIXByb3BzLmJ1Y2tldElucHV0c0Fzc2V0c1Byb3BzKSB7XG4gICAgICAgIHRtcEJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQodGhpcywgJ2lucHV0QXNzZXRzQnVja2V0JytzdGFnZSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBibG9ja1B1YmxpY0FjY2VzczogczMuQmxvY2tQdWJsaWNBY2Nlc3MuQkxPQ0tfQUxMLFxuICAgICAgICAgICAgZW5jcnlwdGlvbjogczMuQnVja2V0RW5jcnlwdGlvbi5TM19NQU5BR0VELFxuICAgICAgICAgICAgYnVja2V0TmFtZTogJ2lucHV0LWFzc2V0cy1idWNrZXQnK3N0YWdlKyctJytBd3MuQUNDT1VOVF9JRCxcbiAgICAgICAgICAgIHNlcnZlckFjY2Vzc0xvZ3NCdWNrZXQ6IHNlcnZlckFjY2Vzc0xvZ0J1Y2tldCxcbiAgICAgICAgICAgIGVuZm9yY2VTU0w6IHRydWUsXG4gICAgICAgICAgICB2ZXJzaW9uZWQ6IHRydWUsXG4gICAgICAgICAgICBsaWZlY3ljbGVSdWxlczogW3tcbiAgICAgICAgICAgICAgZXhwaXJhdGlvbjogRHVyYXRpb24uZGF5cyg5MCksXG4gICAgICAgICAgICB9XSxcbiAgICAgICAgICB9KTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHRtcEJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQodGhpcywgJ0lucHV0QXNzZXRzQnVja2V0JytzdGFnZSwgcHJvcHMuYnVja2V0SW5wdXRzQXNzZXRzUHJvcHMpO1xuICAgICAgfVxuICAgICAgaW5wdXRBc3NldHNCdWNrZXQgPSB0bXBCdWNrZXQ7XG4gICAgICB0aGlzLnMzSW5wdXRBc3NldHNCdWNrZXQgPSB0bXBCdWNrZXQ7XG4gICAgfSBlbHNlIHtcbiAgICAgIGlucHV0QXNzZXRzQnVja2V0ID0gcHJvcHMuZXhpc3RpbmdJbnB1dEFzc2V0c0J1Y2tldE9iajtcbiAgICB9XG5cbiAgICAvLyB0aGlzIGlzIHRoZSBvbmUgd2UgbWFuaXB1bGF0ZSwgd2Uga25vdyBpdCBleGlzdHNcbiAgICB0aGlzLnMzSW5wdXRBc3NldHNCdWNrZXRJbnRlcmZhY2UgPSBpbnB1dEFzc2V0c0J1Y2tldDtcblxuICAgIC8vIEJ1Y2tldCBjb250YWluaW5nIHRoZSBwcm9jZXNzZWQgYXNzZXRzIChkb2N1bWVudHMgLSB0ZXh0IGZvcm1hdCkgdXBsb2FkZWQgYnkgdGhlIHVzZXJcbiAgICBsZXQgcHJvY2Vzc2VkQXNzZXRzQnVja2V0OiBzMy5JQnVja2V0O1xuXG4gICAgaWYgKCFwcm9wcy5leGlzdGluZ1Byb2Nlc3NlZEFzc2V0c0J1Y2tldE9iaikge1xuICAgICAgbGV0IHRtcEJ1Y2tldDogczMuQnVja2V0O1xuICAgICAgaWYgKCFwcm9wcy5idWNrZXRJbnB1dHNBc3NldHNQcm9wcykge1xuICAgICAgICB0bXBCdWNrZXQgPSBuZXcgczMuQnVja2V0KHRoaXMsICdwcm9jZXNzZWRBc3NldHNCdWNrZXQnK3N0YWdlLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIGJsb2NrUHVibGljQWNjZXNzOiBzMy5CbG9ja1B1YmxpY0FjY2Vzcy5CTE9DS19BTEwsXG4gICAgICAgICAgICBlbmNyeXB0aW9uOiBzMy5CdWNrZXRFbmNyeXB0aW9uLlMzX01BTkFHRUQsXG4gICAgICAgICAgICBidWNrZXROYW1lOiAncHJvY2Vzc2VkLWFzc2V0cy1idWNrZXQnK3N0YWdlKyctJytBd3MuQUNDT1VOVF9JRCxcbiAgICAgICAgICAgIHNlcnZlckFjY2Vzc0xvZ3NCdWNrZXQ6IHNlcnZlckFjY2Vzc0xvZ0J1Y2tldCxcbiAgICAgICAgICAgIGVuZm9yY2VTU0w6IHRydWUsXG4gICAgICAgICAgICB2ZXJzaW9uZWQ6IHRydWUsXG4gICAgICAgICAgICBsaWZlY3ljbGVSdWxlczogW3tcbiAgICAgICAgICAgICAgZXhwaXJhdGlvbjogRHVyYXRpb24uZGF5cyg5MCksXG4gICAgICAgICAgICB9XSxcbiAgICAgICAgICB9KTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHRtcEJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQodGhpcywgJ3Byb2Nlc3NlZEFzc2V0c0J1Y2tldCcrc3RhZ2UsIHByb3BzLmJ1Y2tldFByb2Nlc3NlZEFzc2V0c1Byb3BzKTtcbiAgICAgIH1cbiAgICAgIHByb2Nlc3NlZEFzc2V0c0J1Y2tldCA9IHRtcEJ1Y2tldDtcbiAgICAgIHRoaXMuczNQcm9jZXNzZWRBc3NldHNCdWNrZXQgPSB0bXBCdWNrZXQ7XG4gICAgfSBlbHNlIHtcbiAgICAgIHByb2Nlc3NlZEFzc2V0c0J1Y2tldCA9IHByb3BzLmV4aXN0aW5nUHJvY2Vzc2VkQXNzZXRzQnVja2V0T2JqO1xuICAgIH1cblxuICAgIC8vIHRoaXMgaXMgdGhlIG9uZSB3ZSBtYW5pcHVsYXRlLCB3ZSBrbm93IGl0IGV4aXN0c1xuICAgIHRoaXMuczNQcm9jZXNzZWRBc3NldHNCdWNrZXRJbnRlcmZhY2UgPSBwcm9jZXNzZWRBc3NldHNCdWNrZXQ7XG5cbiAgICAvLyBHcmFwaFFMIEFQSVxuICAgIGNvbnN0IGluZ2VzdGlvbl9ncmFwaHFsX2FwaSA9IG5ldyBhcHBzeW5jLkdyYXBocWxBcGkoXG4gICAgICB0aGlzLFxuICAgICAgJ2luZ2VzdGlvbkdyYXBocWxBcGknLFxuICAgICAge1xuICAgICAgICBuYW1lOiAnaW5nZXN0aW9uR3JhcGhxbEFwaScrc3RhZ2UsXG4gICAgICAgIHNjaGVtYTogYXBwc3luYy5TY2hlbWFGaWxlLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4vLi4vLi4vLi4vcmVzb3VyY2VzL2dlbi1haS9hd3MtcmFnLWFwcHN5bmMtc3RlcGZuLW9wZW5zZWFyY2gvc2NoZW1hLmdyYXBocWwnKSksXG4gICAgICAgIGF1dGhvcml6YXRpb25Db25maWc6IHtcbiAgICAgICAgICBkZWZhdWx0QXV0aG9yaXphdGlvbjoge1xuICAgICAgICAgICAgYXV0aG9yaXphdGlvblR5cGU6IGFwcHN5bmMuQXV0aG9yaXphdGlvblR5cGUuVVNFUl9QT09MLFxuICAgICAgICAgICAgdXNlclBvb2xDb25maWc6IHsgdXNlclBvb2w6IHByb3BzLmNvZ25pdG9Vc2VyUG9vbCB9LFxuICAgICAgICAgIH0sXG4gICAgICAgICAgYWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2RlczogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBhdXRob3JpemF0aW9uVHlwZTogYXBwc3luYy5BdXRob3JpemF0aW9uVHlwZS5JQU0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICAgIHhyYXlFbmFibGVkOiBlbmFibGVfeHJheSxcbiAgICAgICAgbG9nQ29uZmlnOiBhcGlfbG9nX2NvbmZpZyxcbiAgICAgIH0sXG4gICAgKTtcblxuICAgIHRoaXMuZ3JhcGhxbEFwaT1pbmdlc3Rpb25fZ3JhcGhxbF9hcGk7XG5cbiAgICAvLyBJZiB0aGUgdXNlciBwcm92aWRlcyBhIG1lcmdlZEFwaSBlbmRwb2ludCwgdGhlIGxhbWJkYVxuICAgIC8vIGZ1bmN0aW9ucyB3aWxsIHVzZSB0aGlzIGVuZHBvaW50IHRvIHNlbmQgdGhlaXIgc3RhdHVzIHVwZGF0ZXNcbiAgICBjb25zdCB1cGRhdGVHcmFwaFFsQXBpRW5kcG9pbnQgPSAhcHJvcHMuZXhpc3RpbmdNZXJnZWRBcGkgPyBpbmdlc3Rpb25fZ3JhcGhxbF9hcGkuZ3JhcGhxbFVybCA6IHByb3BzLmV4aXN0aW5nTWVyZ2VkQXBpLmF0dHJHcmFwaFFsVXJsO1xuICAgIGNvbnN0IHVwZGF0ZUdyYXBoUWxBcGlJZCA9ICFwcm9wcy5leGlzdGluZ01lcmdlZEFwaSA/IGluZ2VzdGlvbl9ncmFwaHFsX2FwaS5hcGlJZCA6IHByb3BzLmV4aXN0aW5nTWVyZ2VkQXBpLmF0dHJBcGlJZDtcblxuICAgIGNvbnN0IGpvYl9zdGF0dXNfZGF0YV9zb3VyY2UgPSBuZXcgYXBwc3luYy5Ob25lRGF0YVNvdXJjZShcbiAgICAgIHRoaXMsXG4gICAgICAnTm9uZURhdGFTb3VyY2VJbmdlc3Rpb24nLFxuICAgICAge1xuICAgICAgICBhcGk6IHRoaXMuZ3JhcGhxbEFwaSxcbiAgICAgICAgbmFtZTogJ0pvYlN0YXR1c0RhdGFTb3VyY2UnLFxuICAgICAgfSxcbiAgICApO1xuXG4gICAgam9iX3N0YXR1c19kYXRhX3NvdXJjZS5jcmVhdGVSZXNvbHZlcihcbiAgICAgICd1cGRhdGVJbmdlc3Rpb25Kb2JTdGF0dXNSZXNvbHZlcicsXG4gICAgICB7XG4gICAgICAgIGZpZWxkTmFtZTogJ3VwZGF0ZUluZ2VzdGlvbkpvYlN0YXR1cycsXG4gICAgICAgIHR5cGVOYW1lOiAnTXV0YXRpb24nLFxuICAgICAgICByZXF1ZXN0TWFwcGluZ1RlbXBsYXRlOiBhcHBzeW5jLk1hcHBpbmdUZW1wbGF0ZS5mcm9tU3RyaW5nKFxuICAgICAgICAgIGBcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgXCJ2ZXJzaW9uXCI6IFwiMjAxNy0wMi0yOFwiLFxuICAgICAgICAgICAgICAgICAgICAgICAgXCJwYXlsb2FkXCI6ICR1dGlsLnRvSnNvbigkY29udGV4dC5hcmdzKVxuICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgICAgIGAsXG4gICAgICAgICksXG4gICAgICAgIHJlc3BvbnNlTWFwcGluZ1RlbXBsYXRlOiBhcHBzeW5jLk1hcHBpbmdUZW1wbGF0ZS5mcm9tU3RyaW5nKCckdXRpbC50b0pzb24oJGNvbnRleHQucmVzdWx0KScpLFxuICAgICAgfSxcblxuICAgICk7XG5cbiAgICBpZiAoIXByb3BzLmV4aXN0aW5nQnVzSW50ZXJmYWNlKSB7XG4gICAgICB0aGlzLmluZ2VzdGlvbkJ1cyA9IG5ldyBldmVudHMuRXZlbnRCdXModGhpcywgJ2luZ2VzdGlvbkV2ZW50QnVzJytzdGFnZSxcbiAgICAgICAge1xuICAgICAgICAgIGV2ZW50QnVzTmFtZTogJ2luZ2VzdGlvbkV2ZW50QnVzJytzdGFnZSxcbiAgICAgICAgfSxcbiAgICAgICk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMuaW5nZXN0aW9uQnVzID0gcHJvcHMuZXhpc3RpbmdCdXNJbnRlcmZhY2U7XG4gICAgfVxuXG4gICAgLy8gY3JlYXRlIGh0dHBkYXRhc291cmNlIHdpdGggaW5nZXN0aW9uX2dyYXBocWxfYXBpXG4gICAgY29uc3QgZXZlbnRfYnJpZGdlX2RhdGFzb3VyY2UgPSB0aGlzLmdyYXBocWxBcGkuYWRkRXZlbnRCcmlkZ2VEYXRhU291cmNlKFxuICAgICAgJ2luZ2VzdGlvbkV2ZW50QnJpZGdlRGF0YVNvdXJjZScrc3RhZ2UsXG4gICAgICB0aGlzLmluZ2VzdGlvbkJ1cyxcbiAgICAgIHtcbiAgICAgICAgbmFtZTogJ2luZ2VzdGlvbkV2ZW50QnJpZGdlRGF0YVNvdXJjZScrc3RhZ2UsXG4gICAgICB9LFxuICAgICk7XG5cbiAgICAvLyBMYW1iZGEgZnVuY3Rpb24gdXNlZCB0byB2YWxpZGF0ZSBpbnB1dHMgaW4gdGhlIHN0ZXAgZnVuY3Rpb25cbiAgICBjb25zdCB2YWxpZGF0ZV9pbnB1dF9mdW5jdGlvbiA9IG5ldyBsYW1iZGEuRG9ja2VySW1hZ2VGdW5jdGlvbihcbiAgICAgIHRoaXMsXG4gICAgICAnbGFtYmRhX2Z1bmN0aW9uX3ZhbGlkYXRpb25faW5wdXQnK3N0YWdlLFxuICAgICAge1xuICAgICAgICBjb2RlOiBsYW1iZGEuRG9ja2VySW1hZ2VDb2RlLmZyb21JbWFnZUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi8uLi8uLi8uLi9sYW1iZGEvYXdzLXJhZy1hcHBzeW5jLXN0ZXBmbi1vcGVuc2VhcmNoL2lucHV0X3ZhbGlkYXRpb24vc3JjJykpLFxuICAgICAgICBmdW5jdGlvbk5hbWU6ICdpbmdlc3Rpb25faW5wdXRfdmFsaWRhdGlvbl9kb2NrZXInK3N0YWdlLFxuICAgICAgICBkZXNjcmlwdGlvbjogJ0xhbWJkYSBmdW5jdGlvbiBmb3IgdmFsaWRhdGluZyBpbnB1dCBmaWxlcyBmb3JtYXRzJyxcbiAgICAgICAgdnBjOiB0aGlzLnZwYyxcbiAgICAgICAgdHJhY2luZzogbGFtYmRhX3RyYWNpbmcsXG4gICAgICAgIHZwY1N1Ym5ldHM6IHsgc3VibmV0VHlwZTogZWMyLlN1Ym5ldFR5cGUuUFJJVkFURV9XSVRIX0VHUkVTUyB9LFxuICAgICAgICBzZWN1cml0eUdyb3VwczogW3RoaXMuc2VjdXJpdHlHcm91cF0sXG4gICAgICAgIG1lbW9yeVNpemU6IDFfNzY5ICogNCxcbiAgICAgICAgdGltZW91dDogRHVyYXRpb24ubWludXRlcygxNSksXG4gICAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgICAgR1JBUEhRTF9VUkw6IHVwZGF0ZUdyYXBoUWxBcGlFbmRwb2ludCxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgKTtcblxuICAgIC8vIEFkZCBHcmFwaFFsIHBlcm1pc3Npb25zIHRvIHRoZSBJQU0gcm9sZSBmb3IgdGhlIExhbWJkYSBmdW5jdGlvblxuICAgIHZhbGlkYXRlX2lucHV0X2Z1bmN0aW9uLmFkZFRvUm9sZVBvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICBhY3Rpb25zOiBbXG4gICAgICAgICdhcHBzeW5jOkdyYXBoUUwnLFxuICAgICAgXSxcbiAgICAgIHJlc291cmNlczogW1xuICAgICAgICAnYXJuOmF3czphcHBzeW5jOicrIEF3cy5SRUdJT04rJzonK0F3cy5BQ0NPVU5UX0lEKyc6YXBpcy8nK3VwZGF0ZUdyYXBoUWxBcGlJZCsnLyonLFxuICAgICAgXSxcbiAgICB9KSk7XG4gICAgLy8gVGhlIGxhbWJkYSB3aWxsIHB1bGwgZG9jdW1lbnRzIGZyb20gdGhlIGlucHV0IGJ1Y2tldCwgdHJhbnNmb3JtIHRoZW0sIGFuZCB1cGxvYWRcbiAgICAvLyB0aGUgYXJ0aWZhY3RzIHRvIHRoZSBwcm9jZXNzZWQgYnVja2V0XG4gICAgLy8gd2UgZG9uJ3QgdXNlIGdyYW50IHJlYWQgaGVyZSBzaW5jZSBpdCBoYXMgbm8gZWZmZWN0IGluIGNhc2Ugb2YgZXhpc3RpbmcgYnVja2V0cyBwcm92aWRlZCBieSB0aGUgdXNlclxuICAgIGNvbnN0IHMzX3RyYW5zZm9ybWVyX2pvYl9mdW5jdGlvbl9yb2xlID0gbmV3IGlhbS5Sb2xlKHRoaXMsICdzM190cmFuc2Zvcm1lcl9qb2JfZnVuY3Rpb25fcm9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdsYW1iZGEuYW1hem9uYXdzLmNvbScpLFxuICAgICAgaW5saW5lUG9saWNpZXM6IHtcbiAgICAgICAgTGFtYmRhRnVuY3Rpb25TZXJ2aWNlUm9sZVBvbGljeTogbmV3IGlhbS5Qb2xpY3lEb2N1bWVudCh7XG4gICAgICAgICAgc3RhdGVtZW50czogW25ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgICAgJ2xvZ3M6Q3JlYXRlTG9nR3JvdXAnLFxuICAgICAgICAgICAgICAnbG9nczpDcmVhdGVMb2dTdHJlYW0nLFxuICAgICAgICAgICAgICAnbG9nczpQdXRMb2dFdmVudHMnLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIHJlc291cmNlczogW2Bhcm46JHtBd3MuUEFSVElUSU9OfTpsb2dzOiR7QXdzLlJFR0lPTn06JHtBd3MuQUNDT1VOVF9JRH06bG9nLWdyb3VwOi9hd3MvbGFtYmRhLypgXSxcbiAgICAgICAgICB9KV0sXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIE1pbmltdW0gcGVybWlzc2lvbnMgZm9yIGEgTGFtYmRhIGZ1bmN0aW9uIHRvIGV4ZWN1dGUgd2hpbGUgYWNjZXNzaW5nIGEgcmVzb3VyY2Ugd2l0aGluIGEgVlBDXG4gICAgczNfdHJhbnNmb3JtZXJfam9iX2Z1bmN0aW9uX3JvbGUuYWRkVG9Qb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgYWN0aW9uczogW1xuICAgICAgICAnZWMyOkNyZWF0ZU5ldHdvcmtJbnRlcmZhY2UnLFxuICAgICAgICAnZWMyOkRlbGV0ZU5ldHdvcmtJbnRlcmZhY2UnLFxuICAgICAgICAnZWMyOkFzc2lnblByaXZhdGVJcEFkZHJlc3NlcycsXG4gICAgICAgICdlYzI6VW5hc3NpZ25Qcml2YXRlSXBBZGRyZXNzZXMnLFxuICAgICAgXSxcbiAgICAgIHJlc291cmNlczogW1xuICAgICAgICAnYXJuOmF3czplYzI6JytBd3MuUkVHSU9OKyc6JytBd3MuQUNDT1VOVF9JRCsnOiovKicsXG4gICAgICBdLFxuICAgIH0pKTtcbiAgICAvLyBEZWNyaWJlIG9ubHkgd29ya3MgaWYgaXQncyBhbGxvd2VkIG9uIGFsbCByZXNvdXJjZXMuXG4gICAgLy8gUmVmZXJlbmNlOiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGFtYmRhL2xhdGVzdC9kZy9jb25maWd1cmF0aW9uLXZwYy5odG1sI3ZwYy1wZXJtaXNzaW9uc1xuICAgIHMzX3RyYW5zZm9ybWVyX2pvYl9mdW5jdGlvbl9yb2xlLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgJ2VjMjpEZXNjcmliZU5ldHdvcmtJbnRlcmZhY2VzJyxcbiAgICAgIF0sXG4gICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgJyonLFxuICAgICAgXSxcbiAgICB9KSk7XG5cbiAgICBzM190cmFuc2Zvcm1lcl9qb2JfZnVuY3Rpb25fcm9sZS5hZGRUb1BvbGljeShcbiAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgJ3MzOkdldE9iamVjdCcsXG4gICAgICAgICAgJ3MzOkdldE9iamVjdConLFxuICAgICAgICAgICdzMzpHZXRCdWNrZXQqJyxcbiAgICAgICAgICAnczM6TGlzdConLFxuICAgICAgICBdLFxuICAgICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgICAnYXJuOmF3czpzMzo6OicgKyB0aGlzLnMzSW5wdXRBc3NldHNCdWNrZXRJbnRlcmZhY2U/LmJ1Y2tldE5hbWUsXG4gICAgICAgICAgJ2Fybjphd3M6czM6OjonICsgdGhpcy5zM0lucHV0QXNzZXRzQnVja2V0SW50ZXJmYWNlPy5idWNrZXROYW1lICsgJy8qJyxcbiAgICAgICAgXSxcbiAgICAgIH0pLFxuICAgICk7XG5cbiAgICBzM190cmFuc2Zvcm1lcl9qb2JfZnVuY3Rpb25fcm9sZS5hZGRUb1BvbGljeShcbiAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICBhY3Rpb25zOiBbJ3MzOlB1dE9iamVjdFJldGVudGlvbicsXG4gICAgICAgICAgJ3MzOkxpc3QqJyxcbiAgICAgICAgICAnczM6R2V0QnVja2V0KicsXG4gICAgICAgICAgJ3MzOkFib3J0KicsXG4gICAgICAgICAgJ3MzOkRlbGV0ZU9iamVjdConLFxuICAgICAgICAgICdzMzpQdXRPYmplY3RMZWdhbEhvbGQnLFxuICAgICAgICAgICdzMzpQdXRPYmplY3RUYWdnaW5nJyxcbiAgICAgICAgICAnczM6UHV0T2JqZWN0VmVyc2lvblRhZ2dpbmcnLFxuICAgICAgICAgICdzMzpQdXRPYmplY3QnLFxuICAgICAgICAgICdzMzpHZXRPYmplY3QqJ10sXG4gICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgICdhcm46YXdzOnMzOjo6JyArIHRoaXMuczNQcm9jZXNzZWRBc3NldHNCdWNrZXRJbnRlcmZhY2U/LmJ1Y2tldE5hbWUsXG4gICAgICAgICAgJ2Fybjphd3M6czM6OjonICsgdGhpcy5zM1Byb2Nlc3NlZEFzc2V0c0J1Y2tldEludGVyZmFjZT8uYnVja2V0TmFtZSArICcvKicsXG4gICAgICAgIF0sXG4gICAgICB9KSxcbiAgICApO1xuXG5cbiAgICAvLyBBZGQgR3JhcGhRbCBwZXJtaXNzaW9ucyB0byB0aGUgSUFNIHJvbGUgZm9yIHRoZSBMYW1iZGEgZnVuY3Rpb25cbiAgICBzM190cmFuc2Zvcm1lcl9qb2JfZnVuY3Rpb25fcm9sZS5hZGRUb1BvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICBhY3Rpb25zOiBbXG4gICAgICAgICdhcHBzeW5jOkdyYXBoUUwnLFxuICAgICAgXSxcbiAgICAgIHJlc291cmNlczogW1xuICAgICAgICAnYXJuOmF3czphcHBzeW5jOicrIEF3cy5SRUdJT04rJzonK0F3cy5BQ0NPVU5UX0lEKyc6YXBpcy8nK3VwZGF0ZUdyYXBoUWxBcGlJZCsnLyonLFxuICAgICAgXSxcbiAgICB9KSk7XG5cbiAgICBOYWdTdXBwcmVzc2lvbnMuYWRkUmVzb3VyY2VTdXBwcmVzc2lvbnMoXG4gICAgICBzM190cmFuc2Zvcm1lcl9qb2JfZnVuY3Rpb25fcm9sZSxcbiAgICAgIFtcbiAgICAgICAge1xuICAgICAgICAgIGlkOiAnQXdzU29sdXRpb25zLUlBTTUnLFxuICAgICAgICAgIHJlYXNvbjogJ0FXU0xhbWJkYUJhc2ljRXhlY3V0aW9uUm9sZSBpcyB1c2VkLicsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAgdHJ1ZSxcbiAgICApO1xuXG4gICAgY29uc3QgczNfdHJhbnNmb3JtZXJfam9iX2Z1bmN0aW9uID0gbmV3IGxhbWJkYS5Eb2NrZXJJbWFnZUZ1bmN0aW9uKFxuICAgICAgdGhpcyxcbiAgICAgICdsYW1iZGFfZnVuY3Rpb25fczNfZmlsZV90cmFuc2Zvcm1lcicrc3RhZ2UsXG4gICAgICB7XG4gICAgICAgIGNvZGU6IGxhbWJkYS5Eb2NrZXJJbWFnZUNvZGUuZnJvbUltYWdlQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJy4uLy4uLy4uLy4uL2xhbWJkYS9hd3MtcmFnLWFwcHN5bmMtc3RlcGZuLW9wZW5zZWFyY2gvczNfZmlsZV90cmFuc2Zvcm1lci9zcmMnKSksXG4gICAgICAgIGZ1bmN0aW9uTmFtZTogJ3MzX2ZpbGVfdHJhbnNmb3JtZXJfZG9ja2VyJytzdGFnZSxcbiAgICAgICAgZGVzY3JpcHRpb246ICdMYW1iZGEgZnVuY3Rpb24gZm9yIGNvbnZlcnRpbmcgZmlsZXMgZnJvbSB0aGVpciBpbnB1dCBmb3JtYXQgdG8gdGV4dCcsXG4gICAgICAgIHZwYzogdGhpcy52cGMsXG4gICAgICAgIHRyYWNpbmc6IGxhbWJkYV90cmFjaW5nLFxuICAgICAgICB2cGNTdWJuZXRzOiB7IHN1Ym5ldFR5cGU6IGVjMi5TdWJuZXRUeXBlLlBSSVZBVEVfV0lUSF9FR1JFU1MgfSxcbiAgICAgICAgc2VjdXJpdHlHcm91cHM6IFt0aGlzLnNlY3VyaXR5R3JvdXBdLFxuICAgICAgICBtZW1vcnlTaXplOiAxXzc2OSAqIDQsXG4gICAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoMTUpLFxuICAgICAgICByb2xlOiBzM190cmFuc2Zvcm1lcl9qb2JfZnVuY3Rpb25fcm9sZSxcbiAgICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgICBJTlBVVF9CVUNLRVQ6IHRoaXMuczNJbnB1dEFzc2V0c0J1Y2tldEludGVyZmFjZS5idWNrZXROYW1lLFxuICAgICAgICAgIE9VVFBVVF9CVUNLRVQ6IHRoaXMuczNQcm9jZXNzZWRBc3NldHNCdWNrZXRJbnRlcmZhY2UuYnVja2V0TmFtZSxcbiAgICAgICAgICBHUkFQSFFMX1VSTDogdXBkYXRlR3JhcGhRbEFwaUVuZHBvaW50LFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICApO1xuXG5cbiAgICBsZXQgU2VjcmV0SWQgPSAnTk9ORSc7XG4gICAgaWYgKHByb3BzLm9wZW5TZWFyY2hTZWNyZXQpIHtTZWNyZXRJZCA9IHByb3BzLm9wZW5TZWFyY2hTZWNyZXQuc2VjcmV0TmFtZTt9XG5cblxuICAgIGNvbnN0IGVtYmVkZGluZ3Nfam9iX2Z1bmN0aW9uX3JvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ2VtYmVkZGluZ3Nfam9iX2Z1bmN0aW9uX3JvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnbGFtYmRhLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIGlubGluZVBvbGljaWVzOiB7XG4gICAgICAgIExhbWJkYUZ1bmN0aW9uU2VydmljZVJvbGVQb2xpY3k6IG5ldyBpYW0uUG9saWN5RG9jdW1lbnQoe1xuICAgICAgICAgIHN0YXRlbWVudHM6IFtuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAgICdsb2dzOkNyZWF0ZUxvZ0dyb3VwJyxcbiAgICAgICAgICAgICAgJ2xvZ3M6Q3JlYXRlTG9nU3RyZWFtJyxcbiAgICAgICAgICAgICAgJ2xvZ3M6UHV0TG9nRXZlbnRzJyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICByZXNvdXJjZXM6IFtgYXJuOiR7QXdzLlBBUlRJVElPTn06bG9nczoke0F3cy5SRUdJT059OiR7QXdzLkFDQ09VTlRfSUR9OmxvZy1ncm91cDovYXdzL2xhbWJkYS8qYF0sXG4gICAgICAgICAgfSldLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBNaW5pbXVtIHBlcm1pc3Npb25zIGZvciBhIExhbWJkYSBmdW5jdGlvbiB0byBleGVjdXRlIHdoaWxlIGFjY2Vzc2luZyBhIHJlc291cmNlIHdpdGhpbiBhIFZQQ1xuICAgIGVtYmVkZGluZ3Nfam9iX2Z1bmN0aW9uX3JvbGUuYWRkVG9Qb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgYWN0aW9uczogW1xuICAgICAgICAnZWMyOkNyZWF0ZU5ldHdvcmtJbnRlcmZhY2UnLFxuICAgICAgICAnZWMyOkRlbGV0ZU5ldHdvcmtJbnRlcmZhY2UnLFxuICAgICAgICAnZWMyOkFzc2lnblByaXZhdGVJcEFkZHJlc3NlcycsXG4gICAgICAgICdlYzI6VW5hc3NpZ25Qcml2YXRlSXBBZGRyZXNzZXMnLFxuICAgICAgXSxcbiAgICAgIHJlc291cmNlczogW1xuICAgICAgICAnYXJuOmF3czplYzI6JytBd3MuUkVHSU9OKyc6JytBd3MuQUNDT1VOVF9JRCsnOiovKicsXG4gICAgICBdLFxuICAgIH0pKTtcbiAgICAvLyBEZWNyaWJlIG9ubHkgd29ya3MgaWYgaXQncyBhbGxvd2VkIG9uIGFsbCByZXNvdXJjZXMuXG4gICAgLy8gUmVmZXJlbmNlOiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGFtYmRhL2xhdGVzdC9kZy9jb25maWd1cmF0aW9uLXZwYy5odG1sI3ZwYy1wZXJtaXNzaW9uc1xuICAgIGVtYmVkZGluZ3Nfam9iX2Z1bmN0aW9uX3JvbGUuYWRkVG9Qb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgYWN0aW9uczogW1xuICAgICAgICAnZWMyOkRlc2NyaWJlTmV0d29ya0ludGVyZmFjZXMnLFxuICAgICAgXSxcbiAgICAgIHJlc291cmNlczogW1xuICAgICAgICAnKicsXG4gICAgICBdLFxuICAgIH0pKTtcblxuICAgIGVtYmVkZGluZ3Nfam9iX2Z1bmN0aW9uX3JvbGUuYWRkVG9Qb2xpY3koXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICdzMzpHZXRPYmplY3QnLFxuICAgICAgICAgICdzMzpHZXRPYmplY3QqJyxcbiAgICAgICAgICAnczM6R2V0QnVja2V0KicsXG4gICAgICAgICAgJ3MzOkxpc3QqJyxcbiAgICAgICAgXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgICAgJ2Fybjphd3M6czM6OjonICsgdGhpcy5zM1Byb2Nlc3NlZEFzc2V0c0J1Y2tldEludGVyZmFjZT8uYnVja2V0TmFtZSxcbiAgICAgICAgICAnYXJuOmF3czpzMzo6OicgKyB0aGlzLnMzUHJvY2Vzc2VkQXNzZXRzQnVja2V0SW50ZXJmYWNlPy5idWNrZXROYW1lICsgJy8qJyxcbiAgICAgICAgXSxcbiAgICAgIH0pLFxuICAgICk7XG5cbiAgICBlbWJlZGRpbmdzX2pvYl9mdW5jdGlvbl9yb2xlLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgIGFjdGlvbnM6IFsnZXM6KiddLFxuICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgICdhcm46YXdzOmVzOicrQXdzLlJFR0lPTisnOicrQXdzLkFDQ09VTlRfSUQrJzpkb21haW4vJytwcm9wcy5leGlzdGluZ09wZW5zZWFyY2hEb21haW4uZG9tYWluTmFtZSsnLyonLFxuICAgICAgICAnYXJuOmF3czplczonK0F3cy5SRUdJT04rJzonK0F3cy5BQ0NPVU5UX0lEKyc6ZG9tYWluLycrcHJvcHMuZXhpc3RpbmdPcGVuc2VhcmNoRG9tYWluLmRvbWFpbk5hbWUsXG4gICAgICBdLFxuICAgIH0pKTtcblxuICAgIC8vIEFkZCBBbWF6b24gQmVkcm9jayBwZXJtaXNzaW9ucyB0byB0aGUgSUFNIHJvbGUgZm9yIHRoZSBMYW1iZGEgZnVuY3Rpb25cbiAgICBlbWJlZGRpbmdzX2pvYl9mdW5jdGlvbl9yb2xlLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgIGFjdGlvbnM6IFsnYmVkcm9jazoqJ10sXG4gICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgJ2Fybjphd3M6YmVkcm9jazonK0F3cy5SRUdJT04rJzo6Zm91bmRhdGlvbi1tb2RlbCcsXG4gICAgICAgICdhcm46YXdzOmJlZHJvY2s6JytBd3MuUkVHSU9OKyc6OmZvdW5kYXRpb24tbW9kZWwvKicsXG4gICAgICBdLFxuICAgIH0pKTtcblxuICAgIE5hZ1N1cHByZXNzaW9ucy5hZGRSZXNvdXJjZVN1cHByZXNzaW9ucyhcbiAgICAgIGVtYmVkZGluZ3Nfam9iX2Z1bmN0aW9uX3JvbGUsXG4gICAgICBbXG4gICAgICAgIHtcbiAgICAgICAgICBpZDogJ0F3c1NvbHV0aW9ucy1JQU01JyxcbiAgICAgICAgICByZWFzb246ICdBV1NMYW1iZGFCYXNpY0V4ZWN1dGlvblJvbGUgaXMgdXNlZC4nLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICAgIHRydWUsXG4gICAgKTtcblxuXG4gICAgLy8gVGhlIGxhbWJkYSB3aWxsIGFjY2VzcyB0aGUgb3BlbnNlYXJjaCBjcmVkZW50aWFsc1xuICAgIGlmIChwcm9wcy5vcGVuU2VhcmNoU2VjcmV0KSB7cHJvcHMub3BlblNlYXJjaFNlY3JldC5ncmFudFJlYWQoZW1iZWRkaW5nc19qb2JfZnVuY3Rpb25fcm9sZSk7fVxuICAgIC8vIExhbWJkYSBmdW5jdGlvbiBwZXJmb3JtaW5nIHRoZSBlbWJlZGRpbmcgam9iXG4gICAgY29uc3QgZW1iZWRkaW5nc19qb2JfZnVuY3Rpb24gPSBuZXcgbGFtYmRhLkRvY2tlckltYWdlRnVuY3Rpb24oXG4gICAgICB0aGlzLFxuICAgICAgJ2xhbWJkYV9mdW5jdGlvbl9lbWJlZGRpbmdzX2pvYicrc3RhZ2UsXG4gICAgICB7XG4gICAgICAgIGNvZGU6IGxhbWJkYS5Eb2NrZXJJbWFnZUNvZGUuZnJvbUltYWdlQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJy4uLy4uLy4uLy4uL2xhbWJkYS9hd3MtcmFnLWFwcHN5bmMtc3RlcGZuLW9wZW5zZWFyY2gvZW1iZWRkaW5nc19qb2Ivc3JjJykpLFxuICAgICAgICBmdW5jdGlvbk5hbWU6ICdlbWJlZGRpbmdzX2pvYl9kb2NrZXInK3N0YWdlLFxuICAgICAgICBkZXNjcmlwdGlvbjogJ0xhbWJkYSBmdW5jdGlvbiBmb3IgY3JlYXRpbmcgZG9jdW1lbnRzIGNodW5rcywgZW1iZWRkaW5ncyBhbmQgc3RvcmluZyB0aGVtIGluIEFtYXpvbiBPcGVuc2VhcmNoJyxcbiAgICAgICAgdnBjOiB0aGlzLnZwYyxcbiAgICAgICAgdHJhY2luZzogbGFtYmRhX3RyYWNpbmcsXG4gICAgICAgIHZwY1N1Ym5ldHM6IHsgc3VibmV0VHlwZTogZWMyLlN1Ym5ldFR5cGUuUFJJVkFURV9XSVRIX0VHUkVTUyB9LFxuICAgICAgICBzZWN1cml0eUdyb3VwczogW3RoaXMuc2VjdXJpdHlHcm91cF0sXG4gICAgICAgIG1lbW9yeVNpemU6IDFfNzY5ICogNCxcbiAgICAgICAgdGltZW91dDogRHVyYXRpb24ubWludXRlcygxNSksXG4gICAgICAgIHJvbGU6IGVtYmVkZGluZ3Nfam9iX2Z1bmN0aW9uX3JvbGUsXG4gICAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgICAgT1VUUFVUX0JVQ0tFVDogdGhpcy5zM1Byb2Nlc3NlZEFzc2V0c0J1Y2tldEludGVyZmFjZS5idWNrZXROYW1lLFxuICAgICAgICAgIEdSQVBIUUxfVVJMOiB1cGRhdGVHcmFwaFFsQXBpRW5kcG9pbnQsXG4gICAgICAgICAgT1BFTlNFQVJDSF9JTkRFWDogcHJvcHMub3BlblNlYXJjaEluZGV4TmFtZSxcbiAgICAgICAgICBPUEVOU0VBUkNIX0RPTUFJTl9FTkRQT0lOVDogcHJvcHMuZXhpc3RpbmdPcGVuc2VhcmNoRG9tYWluLmRvbWFpbkVuZHBvaW50LFxuICAgICAgICAgIE9QRU5TRUFSQ0hfU0VDUkVUX0lEOiBTZWNyZXRJZCxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgKTtcblxuXG4gICAgY29uc3QgZW5hYmxlT3BlcmF0aW9uYWxNZXRyaWMgPSBwcm9wcy5lbmFibGVPcGVyYXRpb25hbE1ldHJpYyB8fCB0cnVlO1xuICAgIGNvbnN0IHNvbHV0aW9uX2lkID0gJ2dlbmFpX2Nka18nK2lkO1xuXG4gICAgaWYgKGVuYWJsZU9wZXJhdGlvbmFsTWV0cmljKSB7XG4gICAgICBlbWJlZGRpbmdzX2pvYl9mdW5jdGlvbi5hZGRFbnZpcm9ubWVudChcbiAgICAgICAgJ0FXU19TREtfVUFfQVBQX0lEJywgc29sdXRpb25faWQsXG4gICAgICApO1xuICAgICAgczNfdHJhbnNmb3JtZXJfam9iX2Z1bmN0aW9uLmFkZEVudmlyb25tZW50KFxuICAgICAgICAnQVdTX1NES19VQV9BUFBfSUQnLCBzb2x1dGlvbl9pZCxcbiAgICAgICk7XG4gICAgICB2YWxpZGF0ZV9pbnB1dF9mdW5jdGlvbi5hZGRFbnZpcm9ubWVudChcbiAgICAgICAgJ0FXU19TREtfVUFfQVBQX0lEJywgc29sdXRpb25faWQsXG4gICAgICApO1xuICAgIH07XG5cbiAgICAvLyBBZGQgR3JhcGhRbCBwZXJtaXNzaW9ucyB0byB0aGUgSUFNIHJvbGUgZm9yIHRoZSBMYW1iZGEgZnVuY3Rpb25cbiAgICBlbWJlZGRpbmdzX2pvYl9mdW5jdGlvbi5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgYWN0aW9uczogW1xuICAgICAgICAnYXBwc3luYzpHcmFwaFFMJyxcbiAgICAgIF0sXG4gICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgJ2Fybjphd3M6YXBwc3luYzonKyBBd3MuUkVHSU9OKyc6JytBd3MuQUNDT1VOVF9JRCsnOmFwaXMvJyt1cGRhdGVHcmFwaFFsQXBpSWQrJy8qJyxcbiAgICAgIF0sXG4gICAgfSkpO1xuXG4gICAgLy8gU3RlcCBmdW5jdGlvbiBkZWZpbml0aW9uXG4gICAgY29uc3QgaW5wdXRWYWxpZGF0aW9uVGFzayA9IG5ldyBzdGVwZm5fdGFzay5MYW1iZGFJbnZva2UoXG4gICAgICB0aGlzLFxuICAgICAgJ1ZhbGlkYXRlIEluZ2VzdGlvbiBJbnB1dCcsXG4gICAgICB7XG4gICAgICAgIGxhbWJkYUZ1bmN0aW9uOiB2YWxpZGF0ZV9pbnB1dF9mdW5jdGlvbixcbiAgICAgICAgcmVzdWx0UGF0aDogJyQudmFsaWRhdGlvbl9yZXN1bHQnLFxuICAgICAgfSxcbiAgICApO1xuXG4gICAgY29uc3QgZmlsZVRyYW5zZm9ybWF0aW9uVGFzayA9IG5ldyBzdGVwZm5fdGFzay5MYW1iZGFJbnZva2UoXG4gICAgICB0aGlzLFxuICAgICAgJ0Rvd25sb2FkIGFuZCB0cmFuc2Zvcm0gZG9jdW1lbnQgdG8gcmF3IHRleHQnLFxuICAgICAge1xuICAgICAgICBsYW1iZGFGdW5jdGlvbjogczNfdHJhbnNmb3JtZXJfam9iX2Z1bmN0aW9uLFxuICAgICAgICByZXN1bHRQYXRoOiAnJC5zM190cmFuc2Zvcm1lcl9yZXN1bHQnLFxuICAgICAgfSxcbiAgICApO1xuXG4gICAgY29uc3QgZW1iZWRkaW5nc1Rhc2sgPSBuZXcgc3RlcGZuX3Rhc2suTGFtYmRhSW52b2tlKFxuICAgICAgdGhpcyxcbiAgICAgICdHZW5lcmF0ZSBlbWJlZGRpbmdzIGZyb20gcHJvY2Vzc2VkIGRvY3VtZW50cyBhbmQgc3RvcmUgdGhlbScsXG4gICAgICB7XG4gICAgICAgIGxhbWJkYUZ1bmN0aW9uOiBlbWJlZGRpbmdzX2pvYl9mdW5jdGlvbixcbiAgICAgICAgcmVzdWx0UGF0aDogJyQnLFxuICAgICAgfSxcbiAgICApO1xuXG4gICAgY29uc3QgdmFsaWRhdGVfaW5wdXRfY2hvaWNlID0gbmV3IHN0ZXBmbi5DaG9pY2UoXG4gICAgICB0aGlzLFxuICAgICAgJ0lzIFZhbGlkIEluZ2VzdGlvbiBQYXJhbWV0ZXJzPycsXG4gICAgICB7XG4gICAgICAgIG91dHB1dFBhdGg6ICckLnZhbGlkYXRpb25fcmVzdWx0LlBheWxvYWQuZmlsZXMnLFxuICAgICAgfSxcbiAgICApO1xuXG4gICAgY29uc3QgcnVuX2ZpbGVzX2luX3BhcmFsbGVsID0gbmV3IHN0ZXBmbi5NYXAoXG4gICAgICB0aGlzLFxuICAgICAgJ01hcCBTdGF0ZScsXG4gICAgICB7XG4gICAgICAgIG1heENvbmN1cnJlbmN5OiAxMDAsXG4gICAgICB9LFxuICAgICkuaXRlcmF0b3IoZmlsZVRyYW5zZm9ybWF0aW9uVGFzayk7XG5cbiAgICBjb25zdCBqb2JGYWlsZWQgPSBuZXcgc3RlcGZuLkZhaWwodGhpcywgJ0pvYiBGYWlsZWQnLCB7XG4gICAgICBjYXVzZTogJ1ZhbGlkYXRpb24gam9iIGZhaWxlZCcsXG4gICAgICBlcnJvcjogJ0Rlc2NyaWJlSm9iIHJldHVybmVkIEZBSUxFRCcsXG4gICAgfSk7XG5cbiAgICBjb25zdCBkZWZpbml0aW9uID0gaW5wdXRWYWxpZGF0aW9uVGFzay5uZXh0KHZhbGlkYXRlX2lucHV0X2Nob2ljZS53aGVuKFxuICAgICAgc3RlcGZuLkNvbmRpdGlvbi5ib29sZWFuRXF1YWxzKCckLnZhbGlkYXRpb25fcmVzdWx0LlBheWxvYWQuaXNWYWxpZCcsIGZhbHNlKSwgam9iRmFpbGVkKS5vdGhlcndpc2UocnVuX2ZpbGVzX2luX3BhcmFsbGVsLm5leHQoZW1iZWRkaW5nc1Rhc2spKSk7XG5cbiAgICBjb25zdCBpbmdlc3Rpb25fc3RlcF9mdW5jdGlvbiA9IG5ldyBzdGVwZm4uU3RhdGVNYWNoaW5lKFxuICAgICAgdGhpcyxcbiAgICAgICdJbmdlc3Rpb25TdGF0ZU1hY2hpbmUnLFxuICAgICAge1xuICAgICAgICBzdGF0ZU1hY2hpbmVOYW1lOiAnSW5nZXN0aW9uU3RhdGVNYWNoaW5lJytzdGFnZSxcbiAgICAgICAgZGVmaW5pdGlvbkJvZHk6IHN0ZXBmbi5EZWZpbml0aW9uQm9keS5mcm9tQ2hhaW5hYmxlKGRlZmluaXRpb24pLFxuICAgICAgICB0aW1lb3V0OiBEdXJhdGlvbi5taW51dGVzKDMwKSxcbiAgICAgICAgbG9nczoge1xuICAgICAgICAgIGRlc3RpbmF0aW9uOiBuZXcgbG9ncy5Mb2dHcm91cCh0aGlzLCAnaW5nZXN0aW9uU3RlcEZ1bmN0aW9uTG9nR3JvdXAnKSxcbiAgICAgICAgICBsZXZlbDogc3RlcGZuLkxvZ0xldmVsLkFMTCxcbiAgICAgICAgfSxcbiAgICAgICAgdHJhY2luZ0VuYWJsZWQ6IGVuYWJsZV94cmF5LFxuICAgICAgfSxcbiAgICApO1xuXG4gICAgdGhpcy5zdGF0ZU1hY2hpbmU9aW5nZXN0aW9uX3N0ZXBfZnVuY3Rpb247XG5cbiAgICB0aGlzLmluZ2VzdGlvbkJ1cy5ncmFudFB1dEV2ZW50c1RvKGV2ZW50X2JyaWRnZV9kYXRhc291cmNlLmdyYW50UHJpbmNpcGFsKTtcblxuICAgIGV2ZW50X2JyaWRnZV9kYXRhc291cmNlLmNyZWF0ZVJlc29sdmVyKFxuICAgICAgJ2luZ2VzdERvY3VtZW50UmVzb2x2ZXInLFxuICAgICAge1xuICAgICAgICBmaWVsZE5hbWU6ICdpbmdlc3REb2N1bWVudHMnLFxuICAgICAgICB0eXBlTmFtZTogJ011dGF0aW9uJyxcbiAgICAgICAgcmVxdWVzdE1hcHBpbmdUZW1wbGF0ZTogYXBwc3luYy5NYXBwaW5nVGVtcGxhdGUuZnJvbVN0cmluZyhcbiAgICAgICAgICBgXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIFwidmVyc2lvblwiOiBcIjIwMTgtMDUtMjlcIixcbiAgICAgICAgICAgICAgICAgICAgICAgIFwib3BlcmF0aW9uXCI6IFwiUHV0RXZlbnRzXCIsXG4gICAgICAgICAgICAgICAgICAgICAgICBcImV2ZW50c1wiOiBbe1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFwic291cmNlXCI6IFwiaW5nZXN0aW9uXCIsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgXCJkZXRhaWxcIjogJHV0aWwudG9Kc29uKCRjb250ZXh0LmFyZ3VtZW50cyksXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgXCJkZXRhaWxUeXBlXCI6IFwiZ2VuQUlkZW1vXCJcbiAgICAgICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgICAgICAgfSBcbiAgICAgICAgICAgICAgICAgICAgYCxcbiAgICAgICAgKSxcbiAgICAgICAgcmVzcG9uc2VNYXBwaW5nVGVtcGxhdGU6IGFwcHN5bmMuTWFwcGluZ1RlbXBsYXRlLmZyb21TdHJpbmcoXG4gICAgICAgICAgYFxuICAgICAgICAgICAgICAgICAgICAjaWYoJGN0eC5lcnJvcilcbiAgICAgICAgICAgICAgICAgICAgICAgICR1dGlsLmVycm9yKCRjdHguZXJyb3IubWVzc2FnZSwgJGN0eC5lcnJvci50eXBlLCAkY3R4LnJlc3VsdClcbiAgICAgICAgICAgICAgICAgICAgI2VuZFxuICAgICAgICAgICAgICAgICAgICAgICAgJHV0aWwudG9Kc29uKCRjdHgucmVzdWx0KVxuICAgICAgICAgICAgICAgICAgICBgLFxuICAgICAgICApLFxuICAgICAgfSxcbiAgICApO1xuXG4gICAgY29uc3QgcnVsZSA9IG5ldyBldmVudHMuUnVsZShcbiAgICAgIHRoaXMsXG4gICAgICAnaW5nZXN0aW9uUnVsZScrc3RhZ2UsXG4gICAgICB7XG4gICAgICAgIGRlc2NyaXB0aW9uOiAnUnVsZSB0byB0cmlnZ2VyIGluZ2VzdGlvbiBmdW5jdGlvbicsXG4gICAgICAgIGV2ZW50QnVzOiB0aGlzLmluZ2VzdGlvbkJ1cyxcbiAgICAgICAgZXZlbnRQYXR0ZXJuOiB7XG4gICAgICAgICAgc291cmNlOiBbJ2luZ2VzdGlvbiddLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICApO1xuXG4gICAgcnVsZS5hZGRUYXJnZXQobmV3IHRhcmdldHMuU2ZuU3RhdGVNYWNoaW5lKHRoaXMuc3RhdGVNYWNoaW5lKSk7XG5cbiAgfVxufSJdfQ==