"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.QaAppsyncOpensearch = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const appsync = require("aws-cdk-lib/aws-appsync");
const ec2 = require("aws-cdk-lib/aws-ec2");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const s3_bucket_helper = require("../../../common/helpers/s3-bucket-helper");
const vpc_helper = require("../../../common/helpers/vpc-helper");
/**
   * @summary The QaAppsyncOpensearch class.
   */
class QaAppsyncOpensearch extends constructs_1.Construct {
    /**
       * @summary Constructs a new instance of the RagAppsyncStepfnOpensearch class.
       * @param {cdk.App} scope - represents the scope for all the resources.
       * @param {string} id - this is a a scope-unique id.
       * @param {QaAppsyncOpensearchProps} props - user provided props for the construct.
       * @since 0.0.0
       * @access public
       */
    constructor(scope, id, props) {
        super(scope, id);
        // stage
        let stage = '-dev';
        if (props?.stage) {
            stage = props.stage;
        }
        // observability
        let lambda_tracing = lambda.Tracing.ACTIVE;
        let enable_xray = true;
        let api_log_config = {
            fieldLogLevel: appsync.FieldLogLevel.ALL,
            retention: logs.RetentionDays.TEN_YEARS,
        };
        if (props.observability == false) {
            enable_xray = false;
            lambda_tracing = lambda.Tracing.DISABLED;
            api_log_config = {
                fieldLogLevel: appsync.FieldLogLevel.NONE,
                retention: logs.RetentionDays.TEN_YEARS,
            };
        }
        ;
        vpc_helper.CheckVpcProps(props);
        s3_bucket_helper.CheckS3Props({
            existingBucketObj: props.existingInputAssetsBucketObj,
            bucketProps: props.bucketInputsAssetsProps,
        });
        if (props?.existingVpc) {
            this.vpc = props.existingVpc;
        }
        else {
            this.vpc = new ec2.Vpc(this, 'Vpc', props.vpcProps);
        }
        // Security group
        if (props?.existingSecurityGroup) {
            this.securityGroup = props.existingSecurityGroup;
        }
        else {
            this.securityGroup = new ec2.SecurityGroup(this, 'securityGroup', {
                vpc: this.vpc,
                allowAllOutbound: true,
                securityGroupName: 'securityGroup' + stage,
            });
        }
        // vpc flowloggroup
        const logGroup = new logs.LogGroup(this, 'qaConstructLogGroup');
        const role = new iam.Role(this, 'qaConstructRole', {
            assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
        });
        // vpc flowlogs
        new ec2.FlowLog(this, 'FlowLog', {
            resourceType: ec2.FlowLogResourceType.fromVpc(this.vpc),
            destination: ec2.FlowLogDestination.toCloudWatchLogs(logGroup, role),
        });
        // bucket for storing server access logging
        const serverAccessLogBucket = new s3.Bucket(this, 'serverAccessLogBucket' + stage, {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            versioned: true,
            lifecycleRules: [{
                    expiration: aws_cdk_lib_1.Duration.days(90),
                }],
        });
        // Bucket containing the inputs assets (documents - text format) uploaded by the user
        let inputAssetsBucket;
        if (!props.existingInputAssetsBucketObj) {
            let tmpBucket;
            if (!props.bucketInputsAssetsProps) {
                tmpBucket = new s3.Bucket(this, 'inputAssetsQABucket' + stage, {
                    blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                    encryption: s3.BucketEncryption.S3_MANAGED,
                    bucketName: 'input-asset-qa-bucket' + stage + '-' + aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    serverAccessLogsBucket: serverAccessLogBucket,
                    enforceSSL: true,
                    versioned: true,
                    lifecycleRules: [{
                            expiration: aws_cdk_lib_1.Duration.days(90),
                        }],
                });
            }
            else {
                tmpBucket = new s3.Bucket(this, 'InputAssetsQABucket' + stage, props.bucketInputsAssetsProps);
            }
            inputAssetsBucket = tmpBucket;
            this.s3InputAssetsBucket = tmpBucket;
        }
        else {
            inputAssetsBucket = props.existingInputAssetsBucketObj;
        }
        // this is the one we manipulate, we know it exists
        this.s3InputAssetsBucketInterface = inputAssetsBucket;
        // GraphQL API
        const question_answering_graphql_api = new appsync.GraphqlApi(this, 'questionAnsweringGraphqlApi', {
            name: 'questionAnsweringGraphqlApi' + stage,
            schema: appsync.SchemaFile.fromAsset(path.join(__dirname, '../../../../resources/gen-ai/aws-qa-appsync-opensearch/schema.graphql')),
            authorizationConfig: {
                defaultAuthorization: {
                    authorizationType: appsync.AuthorizationType.USER_POOL,
                    userPoolConfig: { userPool: props.cognitoUserPool },
                },
                additionalAuthorizationModes: [
                    {
                        authorizationType: appsync.AuthorizationType.IAM,
                    },
                ],
            },
            xrayEnabled: enable_xray,
            logConfig: api_log_config,
        });
        this.graphqlApi = question_answering_graphql_api;
        // If the user provides a mergedApi endpoint, the lambda
        // functions will use this endpoint to send their status updates
        const updateGraphQlApiEndpoint = !props.existingMergedApi ? question_answering_graphql_api.graphqlUrl : props.existingMergedApi.attrGraphQlUrl;
        const updateGraphQlApiId = !props.existingMergedApi ? question_answering_graphql_api.apiId : props.existingMergedApi.attrApiId;
        const job_status_data_source = new appsync.NoneDataSource(this, 'NoneDataSourceQuestionAnswering', {
            api: this.graphqlApi,
            name: 'JobStatusDataSource',
        });
        job_status_data_source.createResolver('updateQAJobStatusResolver', {
            fieldName: 'updateQAJobStatus',
            typeName: 'Mutation',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
                          {
                              "version": "2017-02-28",
                              "payload": $util.toJson($context.args)
                          }
                          `),
            responseMappingTemplate: appsync.MappingTemplate.fromString('$util.toJson($context.result)'),
        });
        if (!props.existingBusInterface) {
            this.qaBus = new events.EventBus(this, 'questionAnsweringEventBus' + stage, {
                eventBusName: 'questionAnsweringEventBus' + stage,
            });
        }
        else {
            this.qaBus = props.existingBusInterface;
        }
        // create httpdatasource with question_answering_graphql_api
        const event_bridge_datasource = this.graphqlApi.addEventBridgeDataSource('questionAnsweringEventBridgeDataSource' + stage, this.qaBus, {
            name: 'questionAnsweringEventBridgeDataSource' + stage,
        });
        let SecretId = 'NONE';
        if (props.openSearchSecret) {
            SecretId = props.openSearchSecret.secretName;
        }
        // Lambda function used to validate inputs in the step function
        const question_answering_function_role = new iam.Role(this, 'question_answering_function_role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`],
                        })],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        question_answering_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:aws:ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        question_answering_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: [
                '*',
            ],
        }));
        // The lambda will access the opensearch credentials
        if (props.openSearchSecret) {
            props.openSearchSecret.grantRead(question_answering_function_role);
        }
        // The lambda will pull processed files and create embeddings
        question_answering_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:GetObject',
                's3:GetObject*',
                's3:GetBucket*',
                's3:List*',
            ],
            resources: [
                'arn:aws:s3:::' + this.s3InputAssetsBucketInterface?.bucketName,
                'arn:aws:s3:::' + this.s3InputAssetsBucketInterface?.bucketName + '/*',
            ],
        }));
        question_answering_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['es:*'],
            resources: [
                'arn:aws:es:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':domain/' + props.existingOpensearchDomain.domainName + '/*',
                'arn:aws:es:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':domain/' + props.existingOpensearchDomain.domainName,
            ],
        }));
        // Add Amazon Bedrock permissions to the IAM role for the Lambda function
        question_answering_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'bedrock:InvokeModel',
                'bedrock:InvokeModelWithResponseStream',
            ],
            resources: [
                'arn:aws:bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model',
                'arn:aws:bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model/*',
            ],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(question_answering_function_role, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        const question_answering_function = new lambda.DockerImageFunction(this, 'lambda_question_answering' + stage, {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-qa-appsync-opensearch/question_answering/src')),
            functionName: 'lambda_question_answering' + stage,
            description: 'Lambda function for question answering',
            vpc: this.vpc,
            tracing: lambda_tracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: 1769 * 4,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: question_answering_function_role,
            environment: {
                GRAPHQL_URL: updateGraphQlApiEndpoint,
                INPUT_BUCKET: this.s3InputAssetsBucketInterface.bucketName,
                OPENSEARCH_DOMAIN_ENDPOINT: props.existingOpensearchDomain.domainEndpoint,
                OPENSEARCH_INDEX: props.openSearchIndexName,
                OPENSEARCH_SECRET_ID: SecretId,
            },
        });
        const enableOperationalMetric = props.enableOperationalMetric || true;
        const solution_id = 'genai_cdk_' + id;
        if (enableOperationalMetric) {
            question_answering_function.addEnvironment('AWS_SDK_UA_APP_ID', solution_id);
        }
        ;
        // Add GraphQl permissions to the IAM role for the Lambda function
        question_answering_function.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'appsync:GraphQL',
            ],
            resources: [
                'arn:aws:appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*',
            ],
        }));
        this.qaBus.grantPutEventsTo(event_bridge_datasource.grantPrincipal);
        event_bridge_datasource.createResolver('QuestionAnsweringResolver', {
            fieldName: 'postQuestion',
            typeName: 'Mutation',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
                        {
                            "version": "2018-05-29",
                            "operation": "PutEvents",
                            "events": [{
                                "source": "questionanswering",
                                "detail": $util.toJson($context.arguments),
                                "detailType": "Question answering"
                            }
                            ]
                        } 
                        `),
            responseMappingTemplate: appsync.MappingTemplate.fromString(`
                        #if($ctx.error)
                            $util.error($ctx.error.message, $ctx.error.type, $ctx.result)
                        #end
                            $util.toJson($ctx.result)
                        `),
        });
        const rule = new events.Rule(this, 'QuestionAnsweringRule' + stage, {
            description: 'Rule to trigger question answering function',
            eventBus: this.qaBus,
            eventPattern: {
                source: ['questionanswering'],
            },
        });
        rule.addTarget(new targets.LambdaFunction(question_answering_function));
    }
}
_a = JSII_RTTI_SYMBOL_1;
QaAppsyncOpensearch[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.QaAppsyncOpensearch", version: "0.0.89" };
exports.QaAppsyncOpensearch = QaAppsyncOpensearch;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi9zcmMvcGF0dGVybnMvZ2VuLWFpL2F3cy1xYS1hcHBzeW5jLW9wZW5zZWFyY2gvaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQTs7Ozs7Ozs7Ozs7R0FXRztBQUNILDZCQUE2QjtBQUM3Qiw2Q0FBNEM7QUFDNUMsbURBQW1EO0FBRW5ELDJDQUEyQztBQUMzQyxpREFBaUQ7QUFDakQsMERBQTBEO0FBQzFELDJDQUEyQztBQUMzQyxpREFBaUQ7QUFDakQsNkNBQTZDO0FBRTdDLHlDQUF5QztBQUV6QyxxQ0FBMEM7QUFDMUMsMkNBQXVDO0FBQ3ZDLDZFQUE2RTtBQUM3RSxpRUFBaUU7QUEwR2pFOztLQUVLO0FBQ0wsTUFBYSxtQkFBb0IsU0FBUSxzQkFBUztJQTRCaEQ7Ozs7Ozs7U0FPSztJQUNMLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBK0I7UUFDdkUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixRQUFRO1FBQ1IsSUFBSSxLQUFLLEdBQUcsTUFBTSxDQUFDO1FBQ25CLElBQUksS0FBSyxFQUFFLEtBQUssRUFBRTtZQUNoQixLQUFLLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQztTQUNyQjtRQUVELGdCQUFnQjtRQUNoQixJQUFJLGNBQWMsR0FBRyxNQUFNLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQztRQUMzQyxJQUFJLFdBQVcsR0FBRyxJQUFJLENBQUM7UUFDdkIsSUFBSSxjQUFjLEdBQUc7WUFDbkIsYUFBYSxFQUFFLE9BQU8sQ0FBQyxhQUFhLENBQUMsR0FBRztZQUN4QyxTQUFTLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxTQUFTO1NBQ3hDLENBQUM7UUFDRixJQUFJLEtBQUssQ0FBQyxhQUFhLElBQUksS0FBSyxFQUFFO1lBQ2hDLFdBQVcsR0FBRyxLQUFLLENBQUM7WUFDcEIsY0FBYyxHQUFHLE1BQU0sQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDO1lBQ3pDLGNBQWMsR0FBRztnQkFDZixhQUFhLEVBQUUsT0FBTyxDQUFDLGFBQWEsQ0FBQyxJQUFJO2dCQUN6QyxTQUFTLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxTQUFTO2FBQ3hDLENBQUM7U0FDSDtRQUFBLENBQUM7UUFFRixVQUFVLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ2hDLGdCQUFnQixDQUFDLFlBQVksQ0FBQztZQUM1QixpQkFBaUIsRUFBRSxLQUFLLENBQUMsNEJBQTRCO1lBQ3JELFdBQVcsRUFBRSxLQUFLLENBQUMsdUJBQXVCO1NBQzNDLENBQUMsQ0FBQztRQUVILElBQUksS0FBSyxFQUFFLFdBQVcsRUFBRTtZQUN0QixJQUFJLENBQUMsR0FBRyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUM7U0FDOUI7YUFBTTtZQUNMLElBQUksQ0FBQyxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLElBQUksRUFBRSxLQUFLLEVBQUUsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQ3JEO1FBRUQsaUJBQWlCO1FBQ2pCLElBQUksS0FBSyxFQUFFLHFCQUFxQixFQUFFO1lBQ2hDLElBQUksQ0FBQyxhQUFhLEdBQUcsS0FBSyxDQUFDLHFCQUFxQixDQUFDO1NBQ2xEO2FBQU07WUFDTCxJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksR0FBRyxDQUFDLGFBQWEsQ0FDeEMsSUFBSSxFQUNKLGVBQWUsRUFDZjtnQkFDRSxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7Z0JBQ2IsZ0JBQWdCLEVBQUUsSUFBSTtnQkFDdEIsaUJBQWlCLEVBQUUsZUFBZSxHQUFDLEtBQUs7YUFDekMsQ0FDRixDQUFDO1NBQ0g7UUFFRCxtQkFBbUI7UUFDbkIsTUFBTSxRQUFRLEdBQUcsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxxQkFBcUIsQ0FBQyxDQUFDO1FBQ2hFLE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLEVBQUU7WUFDakQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLDZCQUE2QixDQUFDO1NBQ25FLENBQUMsQ0FBQztRQUVILGVBQWU7UUFDZixJQUFJLEdBQUcsQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUMvQixZQUFZLEVBQUUsR0FBRyxDQUFDLG1CQUFtQixDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDO1lBQ3ZELFdBQVcsRUFBRSxHQUFHLENBQUMsa0JBQWtCLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQztTQUNyRSxDQUFDLENBQUM7UUFFSCwyQ0FBMkM7UUFDM0MsTUFBTSxxQkFBcUIsR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUM5Qyx1QkFBdUIsR0FBQyxLQUFLLEVBQzdCO1lBQ0UsaUJBQWlCLEVBQUUsRUFBRSxDQUFDLGlCQUFpQixDQUFDLFNBQVM7WUFDakQsVUFBVSxFQUFFLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVO1lBQzFDLFVBQVUsRUFBRSxJQUFJO1lBQ2hCLFNBQVMsRUFBRSxJQUFJO1lBQ2YsY0FBYyxFQUFFLENBQUM7b0JBQ2YsVUFBVSxFQUFFLHNCQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztpQkFDOUIsQ0FBQztTQUNILENBQUMsQ0FBQztRQUVMLHFGQUFxRjtRQUNyRixJQUFJLGlCQUE2QixDQUFDO1FBRWxDLElBQUksQ0FBQyxLQUFLLENBQUMsNEJBQTRCLEVBQUU7WUFDdkMsSUFBSSxTQUFvQixDQUFDO1lBQ3pCLElBQUksQ0FBQyxLQUFLLENBQUMsdUJBQXVCLEVBQUU7Z0JBQ2xDLFNBQVMsR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLHFCQUFxQixHQUFDLEtBQUssRUFDekQ7b0JBQ0UsaUJBQWlCLEVBQUUsRUFBRSxDQUFDLGlCQUFpQixDQUFDLFNBQVM7b0JBQ2pELFVBQVUsRUFBRSxFQUFFLENBQUMsZ0JBQWdCLENBQUMsVUFBVTtvQkFDMUMsVUFBVSxFQUFFLHVCQUF1QixHQUFDLEtBQUssR0FBQyxHQUFHLEdBQUMsaUJBQUcsQ0FBQyxVQUFVO29CQUM1RCxzQkFBc0IsRUFBRSxxQkFBcUI7b0JBQzdDLFVBQVUsRUFBRSxJQUFJO29CQUNoQixTQUFTLEVBQUUsSUFBSTtvQkFDZixjQUFjLEVBQUUsQ0FBQzs0QkFDZixVQUFVLEVBQUUsc0JBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDO3lCQUM5QixDQUFDO2lCQUNILENBQUMsQ0FBQzthQUNOO2lCQUFNO2dCQUNMLFNBQVMsR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLHFCQUFxQixHQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsdUJBQXVCLENBQUMsQ0FBQzthQUM3RjtZQUNELGlCQUFpQixHQUFHLFNBQVMsQ0FBQztZQUM5QixJQUFJLENBQUMsbUJBQW1CLEdBQUcsU0FBUyxDQUFDO1NBQ3RDO2FBQU07WUFDTCxpQkFBaUIsR0FBRyxLQUFLLENBQUMsNEJBQTRCLENBQUM7U0FDeEQ7UUFFRCxtREFBbUQ7UUFDbkQsSUFBSSxDQUFDLDRCQUE0QixHQUFHLGlCQUFpQixDQUFDO1FBRXRELGNBQWM7UUFDZCxNQUFNLDhCQUE4QixHQUFHLElBQUksT0FBTyxDQUFDLFVBQVUsQ0FDM0QsSUFBSSxFQUNKLDZCQUE2QixFQUM3QjtZQUNFLElBQUksRUFBRSw2QkFBNkIsR0FBQyxLQUFLO1lBQ3pDLE1BQU0sRUFBRSxPQUFPLENBQUMsVUFBVSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSx1RUFBdUUsQ0FBQyxDQUFDO1lBQ25JLG1CQUFtQixFQUFFO2dCQUNuQixvQkFBb0IsRUFBRTtvQkFDcEIsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLGlCQUFpQixDQUFDLFNBQVM7b0JBQ3RELGNBQWMsRUFBRSxFQUFFLFFBQVEsRUFBRSxLQUFLLENBQUMsZUFBZSxFQUFFO2lCQUNwRDtnQkFDRCw0QkFBNEIsRUFBRTtvQkFDNUI7d0JBQ0UsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLGlCQUFpQixDQUFDLEdBQUc7cUJBQ2pEO2lCQUNGO2FBQ0Y7WUFDRCxXQUFXLEVBQUUsV0FBVztZQUN4QixTQUFTLEVBQUUsY0FBYztTQUMxQixDQUNGLENBQUM7UUFFRixJQUFJLENBQUMsVUFBVSxHQUFDLDhCQUE4QixDQUFDO1FBRS9DLHdEQUF3RDtRQUN4RCxnRUFBZ0U7UUFDaEUsTUFBTSx3QkFBd0IsR0FBRyxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsOEJBQThCLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUMsY0FBYyxDQUFDO1FBQy9JLE1BQU0sa0JBQWtCLEdBQUcsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxDQUFDLDhCQUE4QixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLFNBQVMsQ0FBQztRQUUvSCxNQUFNLHNCQUFzQixHQUFHLElBQUksT0FBTyxDQUFDLGNBQWMsQ0FDdkQsSUFBSSxFQUNKLGlDQUFpQyxFQUNqQztZQUNFLEdBQUcsRUFBRSxJQUFJLENBQUMsVUFBVTtZQUNwQixJQUFJLEVBQUUscUJBQXFCO1NBQzVCLENBQ0YsQ0FBQztRQUVGLHNCQUFzQixDQUFDLGNBQWMsQ0FDbkMsMkJBQTJCLEVBQzNCO1lBQ0UsU0FBUyxFQUFFLG1CQUFtQjtZQUM5QixRQUFRLEVBQUUsVUFBVTtZQUNwQixzQkFBc0IsRUFBRSxPQUFPLENBQUMsZUFBZSxDQUFDLFVBQVUsQ0FDeEQ7Ozs7OzJCQUtpQixDQUNsQjtZQUNELHVCQUF1QixFQUFFLE9BQU8sQ0FBQyxlQUFlLENBQUMsVUFBVSxDQUFDLCtCQUErQixDQUFDO1NBQzdGLENBQ0YsQ0FBQztRQUVGLElBQUksQ0FBQyxLQUFLLENBQUMsb0JBQW9CLEVBQUU7WUFDL0IsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLDJCQUEyQixHQUFDLEtBQUssRUFDdEU7Z0JBQ0UsWUFBWSxFQUFFLDJCQUEyQixHQUFDLEtBQUs7YUFDaEQsQ0FDRixDQUFDO1NBQ0g7YUFBTTtZQUNMLElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDLG9CQUFvQixDQUFDO1NBQ3pDO1FBRUQsNERBQTREO1FBQzVELE1BQU0sdUJBQXVCLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyx3QkFBd0IsQ0FDdEUsd0NBQXdDLEdBQUMsS0FBSyxFQUM5QyxJQUFJLENBQUMsS0FBSyxFQUNWO1lBQ0UsSUFBSSxFQUFFLHdDQUF3QyxHQUFDLEtBQUs7U0FDckQsQ0FDRixDQUFDO1FBRUYsSUFBSSxRQUFRLEdBQUcsTUFBTSxDQUFDO1FBQ3RCLElBQUksS0FBSyxDQUFDLGdCQUFnQixFQUFFO1lBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVLENBQUM7U0FBQztRQUUzRSwrREFBK0Q7UUFFL0QsTUFBTSxnQ0FBZ0MsR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLGtDQUFrQyxFQUFFO1lBQzlGLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxzQkFBc0IsQ0FBQztZQUMzRCxjQUFjLEVBQUU7Z0JBQ2QsK0JBQStCLEVBQUUsSUFBSSxHQUFHLENBQUMsY0FBYyxDQUFDO29CQUN0RCxVQUFVLEVBQUUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7NEJBQ25DLE9BQU8sRUFBRTtnQ0FDUCxxQkFBcUI7Z0NBQ3JCLHNCQUFzQjtnQ0FDdEIsbUJBQW1COzZCQUNwQjs0QkFDRCxTQUFTLEVBQUUsQ0FBQyxPQUFPLGlCQUFHLENBQUMsU0FBUyxTQUFTLGlCQUFHLENBQUMsTUFBTSxJQUFJLGlCQUFHLENBQUMsVUFBVSwwQkFBMEIsQ0FBQzt5QkFDakcsQ0FBQyxDQUFDO2lCQUNKLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQztRQUVILCtGQUErRjtRQUMvRixnQ0FBZ0MsQ0FBQyxXQUFXLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ25FLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsT0FBTyxFQUFFO2dCQUNQLDRCQUE0QjtnQkFDNUIsNEJBQTRCO2dCQUM1Qiw4QkFBOEI7Z0JBQzlCLGdDQUFnQzthQUNqQztZQUNELFNBQVMsRUFBRTtnQkFDVCxjQUFjLEdBQUMsaUJBQUcsQ0FBQyxNQUFNLEdBQUMsR0FBRyxHQUFDLGlCQUFHLENBQUMsVUFBVSxHQUFDLE1BQU07YUFDcEQ7U0FDRixDQUFDLENBQUMsQ0FBQztRQUNKLHVEQUF1RDtRQUN2RCxpR0FBaUc7UUFDakcsZ0NBQWdDLENBQUMsV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUNuRSxNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRTtnQkFDUCwrQkFBK0I7YUFDaEM7WUFDRCxTQUFTLEVBQUU7Z0JBQ1QsR0FBRzthQUNKO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixvREFBb0Q7UUFDcEQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEVBQUU7WUFBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxDQUFDLGdDQUFnQyxDQUFDLENBQUM7U0FBQztRQUVqRyw2REFBNkQ7UUFDN0QsZ0NBQWdDLENBQUMsV0FBVyxDQUMxQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixPQUFPLEVBQUU7Z0JBQ1AsY0FBYztnQkFDZCxlQUFlO2dCQUNmLGVBQWU7Z0JBQ2YsVUFBVTthQUNYO1lBQ0QsU0FBUyxFQUFFO2dCQUNULGVBQWUsR0FBRyxJQUFJLENBQUMsNEJBQTRCLEVBQUUsVUFBVTtnQkFDL0QsZUFBZSxHQUFHLElBQUksQ0FBQyw0QkFBNEIsRUFBRSxVQUFVLEdBQUcsSUFBSTthQUN2RTtTQUNGLENBQUMsQ0FDSCxDQUFDO1FBRUYsZ0NBQWdDLENBQUMsV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUNuRSxNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRSxDQUFDLE1BQU0sQ0FBQztZQUNqQixTQUFTLEVBQUU7Z0JBQ1QsYUFBYSxHQUFDLGlCQUFHLENBQUMsTUFBTSxHQUFDLEdBQUcsR0FBQyxpQkFBRyxDQUFDLFVBQVUsR0FBQyxVQUFVLEdBQUMsS0FBSyxDQUFDLHdCQUF3QixDQUFDLFVBQVUsR0FBQyxJQUFJO2dCQUNyRyxhQUFhLEdBQUMsaUJBQUcsQ0FBQyxNQUFNLEdBQUMsR0FBRyxHQUFDLGlCQUFHLENBQUMsVUFBVSxHQUFDLFVBQVUsR0FBQyxLQUFLLENBQUMsd0JBQXdCLENBQUMsVUFBVTthQUNqRztTQUNGLENBQUMsQ0FBQyxDQUFDO1FBR0oseUVBQXlFO1FBQ3pFLGdDQUFnQyxDQUFDLFdBQVcsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDbkUsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixPQUFPLEVBQUU7Z0JBQ1AscUJBQXFCO2dCQUNyQix1Q0FBdUM7YUFDeEM7WUFDRCxTQUFTLEVBQUU7Z0JBQ1Qsa0JBQWtCLEdBQUMsaUJBQUcsQ0FBQyxNQUFNLEdBQUMsb0JBQW9CO2dCQUNsRCxrQkFBa0IsR0FBQyxpQkFBRyxDQUFDLE1BQU0sR0FBQyxzQkFBc0I7YUFDckQ7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLHlCQUFlLENBQUMsdUJBQXVCLENBQ3JDLGdDQUFnQyxFQUNoQztZQUNFO2dCQUNFLEVBQUUsRUFBRSxtQkFBbUI7Z0JBQ3ZCLE1BQU0sRUFBRSxzQ0FBc0M7YUFDL0M7U0FDRixFQUNELElBQUksQ0FDTCxDQUFDO1FBRUYsTUFBTSwyQkFBMkIsR0FBRyxJQUFJLE1BQU0sQ0FBQyxtQkFBbUIsQ0FDaEUsSUFBSSxFQUNKLDJCQUEyQixHQUFDLEtBQUssRUFDakM7WUFDRSxJQUFJLEVBQUUsTUFBTSxDQUFDLGVBQWUsQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUscUVBQXFFLENBQUMsQ0FBQztZQUN4SSxZQUFZLEVBQUUsMkJBQTJCLEdBQUMsS0FBSztZQUMvQyxXQUFXLEVBQUUsd0NBQXdDO1lBQ3JELEdBQUcsRUFBRSxJQUFJLENBQUMsR0FBRztZQUNiLE9BQU8sRUFBRSxjQUFjO1lBQ3ZCLFVBQVUsRUFBRSxFQUFFLFVBQVUsRUFBRSxHQUFHLENBQUMsVUFBVSxDQUFDLG1CQUFtQixFQUFFO1lBQzlELGNBQWMsRUFBRSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUM7WUFDcEMsVUFBVSxFQUFFLElBQUssR0FBRyxDQUFDO1lBQ3JCLE9BQU8sRUFBRSxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7WUFDN0IsSUFBSSxFQUFFLGdDQUFnQztZQUN0QyxXQUFXLEVBQUU7Z0JBQ1gsV0FBVyxFQUFFLHdCQUF3QjtnQkFDckMsWUFBWSxFQUFFLElBQUksQ0FBQyw0QkFBNEIsQ0FBQyxVQUFVO2dCQUMxRCwwQkFBMEIsRUFBRSxLQUFLLENBQUMsd0JBQXdCLENBQUMsY0FBYztnQkFDekUsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjtnQkFDM0Msb0JBQW9CLEVBQUUsUUFBUTthQUMvQjtTQUNGLENBQ0YsQ0FBQztRQUdGLE1BQU0sdUJBQXVCLEdBQUcsS0FBSyxDQUFDLHVCQUF1QixJQUFJLElBQUksQ0FBQztRQUN0RSxNQUFNLFdBQVcsR0FBRyxZQUFZLEdBQUMsRUFBRSxDQUFDO1FBRXBDLElBQUksdUJBQXVCLEVBQUU7WUFDM0IsMkJBQTJCLENBQUMsY0FBYyxDQUN4QyxtQkFBbUIsRUFBRSxXQUFXLENBQ2pDLENBQUM7U0FDSDtRQUFBLENBQUM7UUFFRixrRUFBa0U7UUFDbEUsMkJBQTJCLENBQUMsZUFBZSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUNsRSxNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRTtnQkFDUCxpQkFBaUI7YUFDbEI7WUFDRCxTQUFTLEVBQUU7Z0JBQ1Qsa0JBQWtCLEdBQUUsaUJBQUcsQ0FBQyxNQUFNLEdBQUMsR0FBRyxHQUFDLGlCQUFHLENBQUMsVUFBVSxHQUFDLFFBQVEsR0FBQyxrQkFBa0IsR0FBQyxJQUFJO2FBQ25GO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLHVCQUF1QixDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBRXBFLHVCQUF1QixDQUFDLGNBQWMsQ0FDcEMsMkJBQTJCLEVBQzNCO1lBQ0UsU0FBUyxFQUFFLGNBQWM7WUFDekIsUUFBUSxFQUFFLFVBQVU7WUFDcEIsc0JBQXNCLEVBQUUsT0FBTyxDQUFDLGVBQWUsQ0FBQyxVQUFVLENBQ3hEOzs7Ozs7Ozs7Ozt5QkFXZSxDQUNoQjtZQUNELHVCQUF1QixFQUFFLE9BQU8sQ0FBQyxlQUFlLENBQUMsVUFBVSxDQUN6RDs7Ozs7eUJBS2UsQ0FDaEI7U0FDRixDQUNGLENBQUM7UUFFRixNQUFNLElBQUksR0FBRyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQzFCLElBQUksRUFDSix1QkFBdUIsR0FBQyxLQUFLLEVBQzdCO1lBQ0UsV0FBVyxFQUFFLDZDQUE2QztZQUMxRCxRQUFRLEVBQUUsSUFBSSxDQUFDLEtBQUs7WUFDcEIsWUFBWSxFQUFFO2dCQUNaLE1BQU0sRUFBRSxDQUFDLG1CQUFtQixDQUFDO2FBQzlCO1NBQ0YsQ0FDRixDQUFDO1FBRUYsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLE9BQU8sQ0FBQyxjQUFjLENBQUMsMkJBQTJCLENBQUMsQ0FBQyxDQUFDO0lBRTFFLENBQUM7Ozs7QUF6WlUsa0RBQW1CIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiAgQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBEdXJhdGlvbiwgQXdzIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0ICogYXMgYXBwc3luYyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtYXBwc3luYyc7XG5pbXBvcnQgKiBhcyBjb2duaXRvIGZyb20gJ2F3cy1jZGstbGliL2F3cy1jb2duaXRvJztcbmltcG9ydCAqIGFzIGVjMiBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZWMyJztcbmltcG9ydCAqIGFzIGV2ZW50cyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZXZlbnRzJztcbmltcG9ydCAqIGFzIHRhcmdldHMgZnJvbSAnYXdzLWNkay1saWIvYXdzLWV2ZW50cy10YXJnZXRzJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIGxvZ3MgZnJvbSAnYXdzLWNkay1saWIvYXdzLWxvZ3MnO1xuaW1wb3J0ICogYXMgb3BlbnNlYXJjaHNlcnZpY2UgZnJvbSAnYXdzLWNkay1saWIvYXdzLW9wZW5zZWFyY2hzZXJ2aWNlJztcbmltcG9ydCAqIGFzIHMzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1zMyc7XG5pbXBvcnQgKiBhcyBzZWNyZXQgZnJvbSAnYXdzLWNkay1saWIvYXdzLXNlY3JldHNtYW5hZ2VyJztcbmltcG9ydCB7IE5hZ1N1cHByZXNzaW9ucyB9IGZyb20gJ2Nkay1uYWcnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgKiBhcyBzM19idWNrZXRfaGVscGVyIGZyb20gJy4uLy4uLy4uL2NvbW1vbi9oZWxwZXJzL3MzLWJ1Y2tldC1oZWxwZXInO1xuaW1wb3J0ICogYXMgdnBjX2hlbHBlciBmcm9tICcuLi8uLi8uLi9jb21tb24vaGVscGVycy92cGMtaGVscGVyJztcblxuXG4vKipcbiAqIFRoZSBwcm9wZXJ0aWVzIGZvciB0aGUgUWFBcHBzeW5jT3BlbnNlYXJjaFByb3BzIGNsYXNzLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFFhQXBwc3luY09wZW5zZWFyY2hQcm9wcyB7XG4gIC8qKlxuICAgICAqIE9wdGlvbmFsIGN1c3RvbSBwcm9wZXJ0aWVzIGZvciBhIFZQQyB0aGUgY29uc3RydWN0IHdpbGwgY3JlYXRlLiBUaGlzIFZQQyB3aWxsXG4gICAgICogYmUgdXNlZCBieSB0aGUgTGFtYmRhIGZ1bmN0aW9ucyB0aGUgY29uc3RydWN0IGNyZWF0ZXMuIFByb3ZpZGluZ1xuICAgICAqIGJvdGggdGhpcyBhbmQgZXhpc3RpbmdWcGMgaXMgYW4gZXJyb3IuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIG5vbmVcbiAgICAgKi9cbiAgcmVhZG9ubHkgdnBjUHJvcHM/OiBlYzIuVnBjUHJvcHM7XG4gIC8qKlxuICAgICAqIE9wdGlvbmFsIEFuIGV4aXN0aW5nIFZQQyBpbiB3aGljaCB0byBkZXBsb3kgdGhlIGNvbnN0cnVjdC4gUHJvdmlkaW5nIGJvdGggdGhpcyBhbmRcbiAgICAgKiB2cGNQcm9wcyBpcyBhbiBlcnJvci5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gbm9uZVxuICAgICAqL1xuICByZWFkb25seSBleGlzdGluZ1ZwYz86IGVjMi5JVnBjO1xuICAvKipcbiAgICAgKiBPcHRpb25hbCBleGlzdGluZyBzZWN1cml0eSBncm91cCBhbGxvd2luZyBhY2Nlc3MgdG8gb3BlbnNlYXJjaC4gVXNlZCBieSB0aGUgbGFtYmRhIGZ1bmN0aW9uc1xuICAgICAqIGJ1aWx0IGJ5IHRoaXMgY29uc3RydWN0LiBJZiBub3QgcHJvdmlkZWQsIHRoZSBjb25zdHJ1Y3Qgd2lsbCBjcmVhdGUgb25lLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBub25lXG4gICAgICovXG4gIHJlYWRvbmx5IGV4aXN0aW5nU2VjdXJpdHlHcm91cD86IGVjMi5JU2VjdXJpdHlHcm91cDtcbiAgLyoqXG4gICAgICogT3B0aW9uYWwgRXhpc3RpbmcgaW5zdGFuY2Ugb2YgYW4gRXZlbnRCcmlkZ2UgYnVzLiBJZiBub3QgcHJvdmlkZWQsIHRoZSBjb25zdHJ1Y3Qgd2lsbCBjcmVhdGUgb25lLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBOb25lXG4gICAgICovXG4gIHJlYWRvbmx5IGV4aXN0aW5nQnVzSW50ZXJmYWNlPzogZXZlbnRzLklFdmVudEJ1cztcbiAgLyoqXG4gICAgICogRXhpc3RpbmcgaW5zdGFuY2Ugb2YgUzMgQnVja2V0IG9iamVjdCwgcHJvdmlkaW5nIGJvdGggdGhpcyBhbmQgYGJ1Y2tldElucHV0c0Fzc2V0c1Byb3BzYCB3aWxsIGNhdXNlIGFuIGVycm9yLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBOb25lXG4gICAgICovXG4gIHJlYWRvbmx5IGV4aXN0aW5nSW5wdXRBc3NldHNCdWNrZXRPYmo/OiBzMy5JQnVja2V0O1xuICAvKipcbiAgICAgKiBPcHRpb25hbCB1c2VyIHByb3ZpZGVkIHByb3BzIHRvIG92ZXJyaWRlIHRoZSBkZWZhdWx0IHByb3BzIGZvciB0aGUgUzMgQnVja2V0LlxuICAgICAqIFByb3ZpZGluZyBib3RoIHRoaXMgYW5kIGBleGlzdGluZ0lucHV0QXNzZXRzQnVja2V0T2JqYCB3aWxsIGNhdXNlIGFuIGVycm9yLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBEZWZhdWx0IHByb3BzIGFyZSB1c2VkXG4gICAgICovXG4gIHJlYWRvbmx5IGJ1Y2tldElucHV0c0Fzc2V0c1Byb3BzPzogczMuQnVja2V0UHJvcHM7XG4gIC8qKlxuICAgICAqIEV4aXN0aW5nIEFtYXpvbiBPcGVuU2VhcmNoIFNlcnZpY2UgZG9tYWluLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBOb25lXG4gICAgICovXG4gIHJlYWRvbmx5IGV4aXN0aW5nT3BlbnNlYXJjaERvbWFpbjogb3BlbnNlYXJjaHNlcnZpY2UuSURvbWFpbjtcbiAgLyoqXG4gICAgICogRGF0YSBJbmRleCBuYW1lIGZvciB0aGUgT3BlblNlYXJjaCBTZXJ2aWNlLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBOb25lXG4gICAgICovXG4gIHJlYWRvbmx5IG9wZW5TZWFyY2hJbmRleE5hbWU6IHN0cmluZztcbiAgLyoqXG4gICAgICogT3B0aW9uYWwuIFNlY3JldHNNYW5hZ2VyIHNlY3JldCB0byBhdXRoZW50aWNhdGUgYWdhaW5zdCB0aGUgT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpbiBpZlxuICAgICAqIGRvbWFpbiBpcyBjb25maWd1cmVkIHdpdGggVXNlcm5hbWUvUGFzc3dvcmQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICAgKi9cbiAgcmVhZG9ubHkgb3BlblNlYXJjaFNlY3JldD86IHNlY3JldC5JU2VjcmV0O1xuICAvKipcbiAgICogRXhpc3RpbmcgbWVyZ2VkIEFwcHN5bmMgR3JhcGhRTCBhcGkuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgZXhpc3RpbmdNZXJnZWRBcGk/OiBhcHBzeW5jLkNmbkdyYXBoUUxBcGk7XG4gIC8qKlxuICAgICAqIENvZ25pdG8gdXNlciBwb29sIHVzZWQgZm9yIGF1dGhlbnRpY2F0aW9uLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBOb25lXG4gICAgICovXG4gIHJlYWRvbmx5IGNvZ25pdG9Vc2VyUG9vbDogY29nbml0by5JVXNlclBvb2w7XG4gIC8qKlxuICAgICAqIFZhbHVlIHdpbGwgYmUgYXBwZW5kZWQgdG8gcmVzb3VyY2VzIG5hbWUuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIF9kZXZcbiAgICAgKi9cbiAgcmVhZG9ubHkgc3RhZ2U/OiBzdHJpbmc7XG4gIC8qKlxuICAgICAqIEVuYWJsZSBvYnNlcnZhYmlsaXR5LiBXYXJuaW5nOiBhc3NvY2lhdGVkIGNvc3Qgd2l0aCB0aGUgc2VydmljZXNcbiAgICAgKiB1c2VkLiBCZXN0IHByYWN0aXZlIHRvIGVuYWJsZSBieSBkZWZhdWx0LlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSB0cnVlXG4gICAgICovXG4gIHJlYWRvbmx5IG9ic2VydmFiaWxpdHk/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbC5DREsgY29uc3RydWN0cyBwcm92aWRlZCBjb2xsZWN0cyBhbm9ueW1vdXMgb3BlcmF0aW9uYWxcbiAgICogbWV0cmljcyB0byBoZWxwIEFXUyBpbXByb3ZlIHRoZSBxdWFsaXR5IGFuZCBmZWF0dXJlcyBvZiB0aGVcbiAgICogY29uc3RydWN0cy4gRGF0YSBjb2xsZWN0aW9uIGlzIHN1YmplY3QgdG8gdGhlIEFXUyBQcml2YWN5IFBvbGljeVxuICAgKiAoaHR0cHM6Ly9hd3MuYW1hem9uLmNvbS9wcml2YWN5LykuIFRvIG9wdCBvdXQgb2YgdGhpcyBmZWF0dXJlLFxuICAgKiBzaW1wbHkgZGlzYWJsZSBpdCBieSBzZXR0aW5nIHRoZSBjb25zdHJ1Y3QgcHJvcGVydHlcbiAgICogXCJlbmFibGVPcGVyYXRpb25hbE1ldHJpY1wiIHRvIGZhbHNlIGZvciBlYWNoIGNvbnN0cnVjdCB1c2VkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgZW5hYmxlT3BlcmF0aW9uYWxNZXRyaWM/OiBib29sZWFuO1xufVxuXG4vKipcbiAgICogQHN1bW1hcnkgVGhlIFFhQXBwc3luY09wZW5zZWFyY2ggY2xhc3MuXG4gICAqL1xuZXhwb3J0IGNsYXNzIFFhQXBwc3luY09wZW5zZWFyY2ggZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICAvKipcbiAgICogUmV0dXJucyB0aGUgaW5zdGFuY2Ugb2YgZWMyLklWcGMgdXNlZCBieSB0aGUgY29uc3RydWN0XG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdnBjOiBlYzIuSVZwYztcbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIGluc3RhbmNlIG9mIGVjMi5JU2VjdXJpdHlHcm91cCB1c2VkIGJ5IHRoZSBjb25zdHJ1Y3RcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBzZWN1cml0eUdyb3VwOiBlYzIuSVNlY3VyaXR5R3JvdXA7XG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBpbnN0YW5jZSBvZiBldmVudHMuSUV2ZW50QnVzIHVzZWQgYnkgdGhlIGNvbnN0cnVjdFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHFhQnVzOiBldmVudHMuSUV2ZW50QnVzO1xuICAvKipcbiAgICogUmV0dXJucyBhbiBpbnN0YW5jZSBvZiBzMy5JQnVja2V0IGNyZWF0ZWQgYnkgdGhlIGNvbnN0cnVjdFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHMzSW5wdXRBc3NldHNCdWNrZXRJbnRlcmZhY2U6IHMzLklCdWNrZXQ7XG4gIC8qKlxuICAgKiBSZXR1cm5zIGFuIGluc3RhbmNlIG9mIHMzLkJ1Y2tldCBjcmVhdGVkIGJ5IHRoZSBjb25zdHJ1Y3QuXG4gICAqIElNUE9SVEFOVDogSWYgZXhpc3RpbmdJbnB1dEFzc2V0c0J1Y2tldE9iaiB3YXMgcHJvdmlkZWQgaW4gUGF0dGVybiBDb25zdHJ1Y3QgUHJvcHMsXG4gICAqIHRoaXMgcHJvcGVydHkgd2lsbCBiZSB1bmRlZmluZWRcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBzM0lucHV0QXNzZXRzQnVja2V0PzogczMuQnVja2V0O1xuICAvKipcbiAgICogUmV0dXJucyBhbiBpbnN0YW5jZSBvZiBhcHBzeW5jLklHcmFwaHFsQXBpIGNyZWF0ZWQgYnkgdGhlIGNvbnN0cnVjdFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGdyYXBocWxBcGk6IGFwcHN5bmMuSUdyYXBocWxBcGk7XG5cbiAgLyoqXG4gICAgICogQHN1bW1hcnkgQ29uc3RydWN0cyBhIG5ldyBpbnN0YW5jZSBvZiB0aGUgUmFnQXBwc3luY1N0ZXBmbk9wZW5zZWFyY2ggY2xhc3MuXG4gICAgICogQHBhcmFtIHtjZGsuQXBwfSBzY29wZSAtIHJlcHJlc2VudHMgdGhlIHNjb3BlIGZvciBhbGwgdGhlIHJlc291cmNlcy5cbiAgICAgKiBAcGFyYW0ge3N0cmluZ30gaWQgLSB0aGlzIGlzIGEgYSBzY29wZS11bmlxdWUgaWQuXG4gICAgICogQHBhcmFtIHtRYUFwcHN5bmNPcGVuc2VhcmNoUHJvcHN9IHByb3BzIC0gdXNlciBwcm92aWRlZCBwcm9wcyBmb3IgdGhlIGNvbnN0cnVjdC5cbiAgICAgKiBAc2luY2UgMC4wLjBcbiAgICAgKiBAYWNjZXNzIHB1YmxpY1xuICAgICAqL1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogUWFBcHBzeW5jT3BlbnNlYXJjaFByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIC8vIHN0YWdlXG4gICAgbGV0IHN0YWdlID0gJy1kZXYnO1xuICAgIGlmIChwcm9wcz8uc3RhZ2UpIHtcbiAgICAgIHN0YWdlID0gcHJvcHMuc3RhZ2U7XG4gICAgfVxuXG4gICAgLy8gb2JzZXJ2YWJpbGl0eVxuICAgIGxldCBsYW1iZGFfdHJhY2luZyA9IGxhbWJkYS5UcmFjaW5nLkFDVElWRTtcbiAgICBsZXQgZW5hYmxlX3hyYXkgPSB0cnVlO1xuICAgIGxldCBhcGlfbG9nX2NvbmZpZyA9IHtcbiAgICAgIGZpZWxkTG9nTGV2ZWw6IGFwcHN5bmMuRmllbGRMb2dMZXZlbC5BTEwsXG4gICAgICByZXRlbnRpb246IGxvZ3MuUmV0ZW50aW9uRGF5cy5URU5fWUVBUlMsXG4gICAgfTtcbiAgICBpZiAocHJvcHMub2JzZXJ2YWJpbGl0eSA9PSBmYWxzZSkge1xuICAgICAgZW5hYmxlX3hyYXkgPSBmYWxzZTtcbiAgICAgIGxhbWJkYV90cmFjaW5nID0gbGFtYmRhLlRyYWNpbmcuRElTQUJMRUQ7XG4gICAgICBhcGlfbG9nX2NvbmZpZyA9IHtcbiAgICAgICAgZmllbGRMb2dMZXZlbDogYXBwc3luYy5GaWVsZExvZ0xldmVsLk5PTkUsXG4gICAgICAgIHJldGVudGlvbjogbG9ncy5SZXRlbnRpb25EYXlzLlRFTl9ZRUFSUyxcbiAgICAgIH07XG4gICAgfTtcblxuICAgIHZwY19oZWxwZXIuQ2hlY2tWcGNQcm9wcyhwcm9wcyk7XG4gICAgczNfYnVja2V0X2hlbHBlci5DaGVja1MzUHJvcHMoe1xuICAgICAgZXhpc3RpbmdCdWNrZXRPYmo6IHByb3BzLmV4aXN0aW5nSW5wdXRBc3NldHNCdWNrZXRPYmosXG4gICAgICBidWNrZXRQcm9wczogcHJvcHMuYnVja2V0SW5wdXRzQXNzZXRzUHJvcHMsXG4gICAgfSk7XG5cbiAgICBpZiAocHJvcHM/LmV4aXN0aW5nVnBjKSB7XG4gICAgICB0aGlzLnZwYyA9IHByb3BzLmV4aXN0aW5nVnBjO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLnZwYyA9IG5ldyBlYzIuVnBjKHRoaXMsICdWcGMnLCBwcm9wcy52cGNQcm9wcyk7XG4gICAgfVxuXG4gICAgLy8gU2VjdXJpdHkgZ3JvdXBcbiAgICBpZiAocHJvcHM/LmV4aXN0aW5nU2VjdXJpdHlHcm91cCkge1xuICAgICAgdGhpcy5zZWN1cml0eUdyb3VwID0gcHJvcHMuZXhpc3RpbmdTZWN1cml0eUdyb3VwO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLnNlY3VyaXR5R3JvdXAgPSBuZXcgZWMyLlNlY3VyaXR5R3JvdXAoXG4gICAgICAgIHRoaXMsXG4gICAgICAgICdzZWN1cml0eUdyb3VwJyxcbiAgICAgICAge1xuICAgICAgICAgIHZwYzogdGhpcy52cGMsXG4gICAgICAgICAgYWxsb3dBbGxPdXRib3VuZDogdHJ1ZSxcbiAgICAgICAgICBzZWN1cml0eUdyb3VwTmFtZTogJ3NlY3VyaXR5R3JvdXAnK3N0YWdlLFxuICAgICAgICB9LFxuICAgICAgKTtcbiAgICB9XG5cbiAgICAvLyB2cGMgZmxvd2xvZ2dyb3VwXG4gICAgY29uc3QgbG9nR3JvdXAgPSBuZXcgbG9ncy5Mb2dHcm91cCh0aGlzLCAncWFDb25zdHJ1Y3RMb2dHcm91cCcpO1xuICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ3FhQ29uc3RydWN0Um9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCd2cGMtZmxvdy1sb2dzLmFtYXpvbmF3cy5jb20nKSxcbiAgICB9KTtcblxuICAgIC8vIHZwYyBmbG93bG9nc1xuICAgIG5ldyBlYzIuRmxvd0xvZyh0aGlzLCAnRmxvd0xvZycsIHtcbiAgICAgIHJlc291cmNlVHlwZTogZWMyLkZsb3dMb2dSZXNvdXJjZVR5cGUuZnJvbVZwYyh0aGlzLnZwYyksXG4gICAgICBkZXN0aW5hdGlvbjogZWMyLkZsb3dMb2dEZXN0aW5hdGlvbi50b0Nsb3VkV2F0Y2hMb2dzKGxvZ0dyb3VwLCByb2xlKSxcbiAgICB9KTtcblxuICAgIC8vIGJ1Y2tldCBmb3Igc3RvcmluZyBzZXJ2ZXIgYWNjZXNzIGxvZ2dpbmdcbiAgICBjb25zdCBzZXJ2ZXJBY2Nlc3NMb2dCdWNrZXQgPSBuZXcgczMuQnVja2V0KHRoaXMsXG4gICAgICAnc2VydmVyQWNjZXNzTG9nQnVja2V0JytzdGFnZSxcbiAgICAgIHtcbiAgICAgICAgYmxvY2tQdWJsaWNBY2Nlc3M6IHMzLkJsb2NrUHVibGljQWNjZXNzLkJMT0NLX0FMTCxcbiAgICAgICAgZW5jcnlwdGlvbjogczMuQnVja2V0RW5jcnlwdGlvbi5TM19NQU5BR0VELFxuICAgICAgICBlbmZvcmNlU1NMOiB0cnVlLFxuICAgICAgICB2ZXJzaW9uZWQ6IHRydWUsXG4gICAgICAgIGxpZmVjeWNsZVJ1bGVzOiBbe1xuICAgICAgICAgIGV4cGlyYXRpb246IER1cmF0aW9uLmRheXMoOTApLFxuICAgICAgICB9XSxcbiAgICAgIH0pO1xuXG4gICAgLy8gQnVja2V0IGNvbnRhaW5pbmcgdGhlIGlucHV0cyBhc3NldHMgKGRvY3VtZW50cyAtIHRleHQgZm9ybWF0KSB1cGxvYWRlZCBieSB0aGUgdXNlclxuICAgIGxldCBpbnB1dEFzc2V0c0J1Y2tldDogczMuSUJ1Y2tldDtcblxuICAgIGlmICghcHJvcHMuZXhpc3RpbmdJbnB1dEFzc2V0c0J1Y2tldE9iaikge1xuICAgICAgbGV0IHRtcEJ1Y2tldDogczMuQnVja2V0O1xuICAgICAgaWYgKCFwcm9wcy5idWNrZXRJbnB1dHNBc3NldHNQcm9wcykge1xuICAgICAgICB0bXBCdWNrZXQgPSBuZXcgczMuQnVja2V0KHRoaXMsICdpbnB1dEFzc2V0c1FBQnVja2V0JytzdGFnZSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBibG9ja1B1YmxpY0FjY2VzczogczMuQmxvY2tQdWJsaWNBY2Nlc3MuQkxPQ0tfQUxMLFxuICAgICAgICAgICAgZW5jcnlwdGlvbjogczMuQnVja2V0RW5jcnlwdGlvbi5TM19NQU5BR0VELFxuICAgICAgICAgICAgYnVja2V0TmFtZTogJ2lucHV0LWFzc2V0LXFhLWJ1Y2tldCcrc3RhZ2UrJy0nK0F3cy5BQ0NPVU5UX0lELFxuICAgICAgICAgICAgc2VydmVyQWNjZXNzTG9nc0J1Y2tldDogc2VydmVyQWNjZXNzTG9nQnVja2V0LFxuICAgICAgICAgICAgZW5mb3JjZVNTTDogdHJ1ZSxcbiAgICAgICAgICAgIHZlcnNpb25lZDogdHJ1ZSxcbiAgICAgICAgICAgIGxpZmVjeWNsZVJ1bGVzOiBbe1xuICAgICAgICAgICAgICBleHBpcmF0aW9uOiBEdXJhdGlvbi5kYXlzKDkwKSxcbiAgICAgICAgICAgIH1dLFxuICAgICAgICAgIH0pO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgdG1wQnVja2V0ID0gbmV3IHMzLkJ1Y2tldCh0aGlzLCAnSW5wdXRBc3NldHNRQUJ1Y2tldCcrc3RhZ2UsIHByb3BzLmJ1Y2tldElucHV0c0Fzc2V0c1Byb3BzKTtcbiAgICAgIH1cbiAgICAgIGlucHV0QXNzZXRzQnVja2V0ID0gdG1wQnVja2V0O1xuICAgICAgdGhpcy5zM0lucHV0QXNzZXRzQnVja2V0ID0gdG1wQnVja2V0O1xuICAgIH0gZWxzZSB7XG4gICAgICBpbnB1dEFzc2V0c0J1Y2tldCA9IHByb3BzLmV4aXN0aW5nSW5wdXRBc3NldHNCdWNrZXRPYmo7XG4gICAgfVxuXG4gICAgLy8gdGhpcyBpcyB0aGUgb25lIHdlIG1hbmlwdWxhdGUsIHdlIGtub3cgaXQgZXhpc3RzXG4gICAgdGhpcy5zM0lucHV0QXNzZXRzQnVja2V0SW50ZXJmYWNlID0gaW5wdXRBc3NldHNCdWNrZXQ7XG5cbiAgICAvLyBHcmFwaFFMIEFQSVxuICAgIGNvbnN0IHF1ZXN0aW9uX2Fuc3dlcmluZ19ncmFwaHFsX2FwaSA9IG5ldyBhcHBzeW5jLkdyYXBocWxBcGkoXG4gICAgICB0aGlzLFxuICAgICAgJ3F1ZXN0aW9uQW5zd2VyaW5nR3JhcGhxbEFwaScsXG4gICAgICB7XG4gICAgICAgIG5hbWU6ICdxdWVzdGlvbkFuc3dlcmluZ0dyYXBocWxBcGknK3N0YWdlLFxuICAgICAgICBzY2hlbWE6IGFwcHN5bmMuU2NoZW1hRmlsZS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJy4uLy4uLy4uLy4uL3Jlc291cmNlcy9nZW4tYWkvYXdzLXFhLWFwcHN5bmMtb3BlbnNlYXJjaC9zY2hlbWEuZ3JhcGhxbCcpKSxcbiAgICAgICAgYXV0aG9yaXphdGlvbkNvbmZpZzoge1xuICAgICAgICAgIGRlZmF1bHRBdXRob3JpemF0aW9uOiB7XG4gICAgICAgICAgICBhdXRob3JpemF0aW9uVHlwZTogYXBwc3luYy5BdXRob3JpemF0aW9uVHlwZS5VU0VSX1BPT0wsXG4gICAgICAgICAgICB1c2VyUG9vbENvbmZpZzogeyB1c2VyUG9vbDogcHJvcHMuY29nbml0b1VzZXJQb29sIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICBhZGRpdGlvbmFsQXV0aG9yaXphdGlvbk1vZGVzOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIGF1dGhvcml6YXRpb25UeXBlOiBhcHBzeW5jLkF1dGhvcml6YXRpb25UeXBlLklBTSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgICAgeHJheUVuYWJsZWQ6IGVuYWJsZV94cmF5LFxuICAgICAgICBsb2dDb25maWc6IGFwaV9sb2dfY29uZmlnLFxuICAgICAgfSxcbiAgICApO1xuXG4gICAgdGhpcy5ncmFwaHFsQXBpPXF1ZXN0aW9uX2Fuc3dlcmluZ19ncmFwaHFsX2FwaTtcblxuICAgIC8vIElmIHRoZSB1c2VyIHByb3ZpZGVzIGEgbWVyZ2VkQXBpIGVuZHBvaW50LCB0aGUgbGFtYmRhXG4gICAgLy8gZnVuY3Rpb25zIHdpbGwgdXNlIHRoaXMgZW5kcG9pbnQgdG8gc2VuZCB0aGVpciBzdGF0dXMgdXBkYXRlc1xuICAgIGNvbnN0IHVwZGF0ZUdyYXBoUWxBcGlFbmRwb2ludCA9ICFwcm9wcy5leGlzdGluZ01lcmdlZEFwaSA/IHF1ZXN0aW9uX2Fuc3dlcmluZ19ncmFwaHFsX2FwaS5ncmFwaHFsVXJsIDogcHJvcHMuZXhpc3RpbmdNZXJnZWRBcGkuYXR0ckdyYXBoUWxVcmw7XG4gICAgY29uc3QgdXBkYXRlR3JhcGhRbEFwaUlkID0gIXByb3BzLmV4aXN0aW5nTWVyZ2VkQXBpID8gcXVlc3Rpb25fYW5zd2VyaW5nX2dyYXBocWxfYXBpLmFwaUlkIDogcHJvcHMuZXhpc3RpbmdNZXJnZWRBcGkuYXR0ckFwaUlkO1xuXG4gICAgY29uc3Qgam9iX3N0YXR1c19kYXRhX3NvdXJjZSA9IG5ldyBhcHBzeW5jLk5vbmVEYXRhU291cmNlKFxuICAgICAgdGhpcyxcbiAgICAgICdOb25lRGF0YVNvdXJjZVF1ZXN0aW9uQW5zd2VyaW5nJyxcbiAgICAgIHtcbiAgICAgICAgYXBpOiB0aGlzLmdyYXBocWxBcGksXG4gICAgICAgIG5hbWU6ICdKb2JTdGF0dXNEYXRhU291cmNlJyxcbiAgICAgIH0sXG4gICAgKTtcblxuICAgIGpvYl9zdGF0dXNfZGF0YV9zb3VyY2UuY3JlYXRlUmVzb2x2ZXIoXG4gICAgICAndXBkYXRlUUFKb2JTdGF0dXNSZXNvbHZlcicsXG4gICAgICB7XG4gICAgICAgIGZpZWxkTmFtZTogJ3VwZGF0ZVFBSm9iU3RhdHVzJyxcbiAgICAgICAgdHlwZU5hbWU6ICdNdXRhdGlvbicsXG4gICAgICAgIHJlcXVlc3RNYXBwaW5nVGVtcGxhdGU6IGFwcHN5bmMuTWFwcGluZ1RlbXBsYXRlLmZyb21TdHJpbmcoXG4gICAgICAgICAgYFxuICAgICAgICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcInZlcnNpb25cIjogXCIyMDE3LTAyLTI4XCIsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcInBheWxvYWRcIjogJHV0aWwudG9Kc29uKCRjb250ZXh0LmFyZ3MpXG4gICAgICAgICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgYCxcbiAgICAgICAgKSxcbiAgICAgICAgcmVzcG9uc2VNYXBwaW5nVGVtcGxhdGU6IGFwcHN5bmMuTWFwcGluZ1RlbXBsYXRlLmZyb21TdHJpbmcoJyR1dGlsLnRvSnNvbigkY29udGV4dC5yZXN1bHQpJyksXG4gICAgICB9LFxuICAgICk7XG5cbiAgICBpZiAoIXByb3BzLmV4aXN0aW5nQnVzSW50ZXJmYWNlKSB7XG4gICAgICB0aGlzLnFhQnVzID0gbmV3IGV2ZW50cy5FdmVudEJ1cyh0aGlzLCAncXVlc3Rpb25BbnN3ZXJpbmdFdmVudEJ1cycrc3RhZ2UsXG4gICAgICAgIHtcbiAgICAgICAgICBldmVudEJ1c05hbWU6ICdxdWVzdGlvbkFuc3dlcmluZ0V2ZW50QnVzJytzdGFnZSxcbiAgICAgICAgfSxcbiAgICAgICk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMucWFCdXMgPSBwcm9wcy5leGlzdGluZ0J1c0ludGVyZmFjZTtcbiAgICB9XG5cbiAgICAvLyBjcmVhdGUgaHR0cGRhdGFzb3VyY2Ugd2l0aCBxdWVzdGlvbl9hbnN3ZXJpbmdfZ3JhcGhxbF9hcGlcbiAgICBjb25zdCBldmVudF9icmlkZ2VfZGF0YXNvdXJjZSA9IHRoaXMuZ3JhcGhxbEFwaS5hZGRFdmVudEJyaWRnZURhdGFTb3VyY2UoXG4gICAgICAncXVlc3Rpb25BbnN3ZXJpbmdFdmVudEJyaWRnZURhdGFTb3VyY2UnK3N0YWdlLFxuICAgICAgdGhpcy5xYUJ1cyxcbiAgICAgIHtcbiAgICAgICAgbmFtZTogJ3F1ZXN0aW9uQW5zd2VyaW5nRXZlbnRCcmlkZ2VEYXRhU291cmNlJytzdGFnZSxcbiAgICAgIH0sXG4gICAgKTtcblxuICAgIGxldCBTZWNyZXRJZCA9ICdOT05FJztcbiAgICBpZiAocHJvcHMub3BlblNlYXJjaFNlY3JldCkge1NlY3JldElkID0gcHJvcHMub3BlblNlYXJjaFNlY3JldC5zZWNyZXROYW1lO31cblxuICAgIC8vIExhbWJkYSBmdW5jdGlvbiB1c2VkIHRvIHZhbGlkYXRlIGlucHV0cyBpbiB0aGUgc3RlcCBmdW5jdGlvblxuXG4gICAgY29uc3QgcXVlc3Rpb25fYW5zd2VyaW5nX2Z1bmN0aW9uX3JvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ3F1ZXN0aW9uX2Fuc3dlcmluZ19mdW5jdGlvbl9yb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2xhbWJkYS5hbWF6b25hd3MuY29tJyksXG4gICAgICBpbmxpbmVQb2xpY2llczoge1xuICAgICAgICBMYW1iZGFGdW5jdGlvblNlcnZpY2VSb2xlUG9saWN5OiBuZXcgaWFtLlBvbGljeURvY3VtZW50KHtcbiAgICAgICAgICBzdGF0ZW1lbnRzOiBbbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgICAnbG9nczpDcmVhdGVMb2dHcm91cCcsXG4gICAgICAgICAgICAgICdsb2dzOkNyZWF0ZUxvZ1N0cmVhbScsXG4gICAgICAgICAgICAgICdsb2dzOlB1dExvZ0V2ZW50cycsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgcmVzb3VyY2VzOiBbYGFybjoke0F3cy5QQVJUSVRJT059OmxvZ3M6JHtBd3MuUkVHSU9OfToke0F3cy5BQ0NPVU5UX0lEfTpsb2ctZ3JvdXA6L2F3cy9sYW1iZGEvKmBdLFxuICAgICAgICAgIH0pXSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gTWluaW11bSBwZXJtaXNzaW9ucyBmb3IgYSBMYW1iZGEgZnVuY3Rpb24gdG8gZXhlY3V0ZSB3aGlsZSBhY2Nlc3NpbmcgYSByZXNvdXJjZSB3aXRoaW4gYSBWUENcbiAgICBxdWVzdGlvbl9hbnN3ZXJpbmdfZnVuY3Rpb25fcm9sZS5hZGRUb1BvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICBhY3Rpb25zOiBbXG4gICAgICAgICdlYzI6Q3JlYXRlTmV0d29ya0ludGVyZmFjZScsXG4gICAgICAgICdlYzI6RGVsZXRlTmV0d29ya0ludGVyZmFjZScsXG4gICAgICAgICdlYzI6QXNzaWduUHJpdmF0ZUlwQWRkcmVzc2VzJyxcbiAgICAgICAgJ2VjMjpVbmFzc2lnblByaXZhdGVJcEFkZHJlc3NlcycsXG4gICAgICBdLFxuICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgICdhcm46YXdzOmVjMjonK0F3cy5SRUdJT04rJzonK0F3cy5BQ0NPVU5UX0lEKyc6Ki8qJyxcbiAgICAgIF0sXG4gICAgfSkpO1xuICAgIC8vIERlY3JpYmUgb25seSB3b3JrcyBpZiBpdCdzIGFsbG93ZWQgb24gYWxsIHJlc291cmNlcy5cbiAgICAvLyBSZWZlcmVuY2U6IGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9sYW1iZGEvbGF0ZXN0L2RnL2NvbmZpZ3VyYXRpb24tdnBjLmh0bWwjdnBjLXBlcm1pc3Npb25zXG4gICAgcXVlc3Rpb25fYW5zd2VyaW5nX2Z1bmN0aW9uX3JvbGUuYWRkVG9Qb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgYWN0aW9uczogW1xuICAgICAgICAnZWMyOkRlc2NyaWJlTmV0d29ya0ludGVyZmFjZXMnLFxuICAgICAgXSxcbiAgICAgIHJlc291cmNlczogW1xuICAgICAgICAnKicsXG4gICAgICBdLFxuICAgIH0pKTtcblxuICAgIC8vIFRoZSBsYW1iZGEgd2lsbCBhY2Nlc3MgdGhlIG9wZW5zZWFyY2ggY3JlZGVudGlhbHNcbiAgICBpZiAocHJvcHMub3BlblNlYXJjaFNlY3JldCkge3Byb3BzLm9wZW5TZWFyY2hTZWNyZXQuZ3JhbnRSZWFkKHF1ZXN0aW9uX2Fuc3dlcmluZ19mdW5jdGlvbl9yb2xlKTt9XG5cbiAgICAvLyBUaGUgbGFtYmRhIHdpbGwgcHVsbCBwcm9jZXNzZWQgZmlsZXMgYW5kIGNyZWF0ZSBlbWJlZGRpbmdzXG4gICAgcXVlc3Rpb25fYW5zd2VyaW5nX2Z1bmN0aW9uX3JvbGUuYWRkVG9Qb2xpY3koXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICdzMzpHZXRPYmplY3QnLFxuICAgICAgICAgICdzMzpHZXRPYmplY3QqJyxcbiAgICAgICAgICAnczM6R2V0QnVja2V0KicsXG4gICAgICAgICAgJ3MzOkxpc3QqJyxcbiAgICAgICAgXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgICAgJ2Fybjphd3M6czM6OjonICsgdGhpcy5zM0lucHV0QXNzZXRzQnVja2V0SW50ZXJmYWNlPy5idWNrZXROYW1lLFxuICAgICAgICAgICdhcm46YXdzOnMzOjo6JyArIHRoaXMuczNJbnB1dEFzc2V0c0J1Y2tldEludGVyZmFjZT8uYnVja2V0TmFtZSArICcvKicsXG4gICAgICAgIF0sXG4gICAgICB9KSxcbiAgICApO1xuXG4gICAgcXVlc3Rpb25fYW5zd2VyaW5nX2Z1bmN0aW9uX3JvbGUuYWRkVG9Qb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgYWN0aW9uczogWydlczoqJ10sXG4gICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgJ2Fybjphd3M6ZXM6JytBd3MuUkVHSU9OKyc6JytBd3MuQUNDT1VOVF9JRCsnOmRvbWFpbi8nK3Byb3BzLmV4aXN0aW5nT3BlbnNlYXJjaERvbWFpbi5kb21haW5OYW1lKycvKicsXG4gICAgICAgICdhcm46YXdzOmVzOicrQXdzLlJFR0lPTisnOicrQXdzLkFDQ09VTlRfSUQrJzpkb21haW4vJytwcm9wcy5leGlzdGluZ09wZW5zZWFyY2hEb21haW4uZG9tYWluTmFtZSxcbiAgICAgIF0sXG4gICAgfSkpO1xuXG5cbiAgICAvLyBBZGQgQW1hem9uIEJlZHJvY2sgcGVybWlzc2lvbnMgdG8gdGhlIElBTSByb2xlIGZvciB0aGUgTGFtYmRhIGZ1bmN0aW9uXG4gICAgcXVlc3Rpb25fYW5zd2VyaW5nX2Z1bmN0aW9uX3JvbGUuYWRkVG9Qb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgYWN0aW9uczogW1xuICAgICAgICAnYmVkcm9jazpJbnZva2VNb2RlbCcsXG4gICAgICAgICdiZWRyb2NrOkludm9rZU1vZGVsV2l0aFJlc3BvbnNlU3RyZWFtJyxcbiAgICAgIF0sXG4gICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgJ2Fybjphd3M6YmVkcm9jazonK0F3cy5SRUdJT04rJzo6Zm91bmRhdGlvbi1tb2RlbCcsXG4gICAgICAgICdhcm46YXdzOmJlZHJvY2s6JytBd3MuUkVHSU9OKyc6OmZvdW5kYXRpb24tbW9kZWwvKicsXG4gICAgICBdLFxuICAgIH0pKTtcblxuICAgIE5hZ1N1cHByZXNzaW9ucy5hZGRSZXNvdXJjZVN1cHByZXNzaW9ucyhcbiAgICAgIHF1ZXN0aW9uX2Fuc3dlcmluZ19mdW5jdGlvbl9yb2xlLFxuICAgICAgW1xuICAgICAgICB7XG4gICAgICAgICAgaWQ6ICdBd3NTb2x1dGlvbnMtSUFNNScsXG4gICAgICAgICAgcmVhc29uOiAnQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlIGlzIHVzZWQuJyxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgICB0cnVlLFxuICAgICk7XG5cbiAgICBjb25zdCBxdWVzdGlvbl9hbnN3ZXJpbmdfZnVuY3Rpb24gPSBuZXcgbGFtYmRhLkRvY2tlckltYWdlRnVuY3Rpb24oXG4gICAgICB0aGlzLFxuICAgICAgJ2xhbWJkYV9xdWVzdGlvbl9hbnN3ZXJpbmcnK3N0YWdlLFxuICAgICAge1xuICAgICAgICBjb2RlOiBsYW1iZGEuRG9ja2VySW1hZ2VDb2RlLmZyb21JbWFnZUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi8uLi8uLi8uLi9sYW1iZGEvYXdzLXFhLWFwcHN5bmMtb3BlbnNlYXJjaC9xdWVzdGlvbl9hbnN3ZXJpbmcvc3JjJykpLFxuICAgICAgICBmdW5jdGlvbk5hbWU6ICdsYW1iZGFfcXVlc3Rpb25fYW5zd2VyaW5nJytzdGFnZSxcbiAgICAgICAgZGVzY3JpcHRpb246ICdMYW1iZGEgZnVuY3Rpb24gZm9yIHF1ZXN0aW9uIGFuc3dlcmluZycsXG4gICAgICAgIHZwYzogdGhpcy52cGMsXG4gICAgICAgIHRyYWNpbmc6IGxhbWJkYV90cmFjaW5nLFxuICAgICAgICB2cGNTdWJuZXRzOiB7IHN1Ym5ldFR5cGU6IGVjMi5TdWJuZXRUeXBlLlBSSVZBVEVfV0lUSF9FR1JFU1MgfSxcbiAgICAgICAgc2VjdXJpdHlHcm91cHM6IFt0aGlzLnNlY3VyaXR5R3JvdXBdLFxuICAgICAgICBtZW1vcnlTaXplOiAxXzc2OSAqIDQsXG4gICAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoMTUpLFxuICAgICAgICByb2xlOiBxdWVzdGlvbl9hbnN3ZXJpbmdfZnVuY3Rpb25fcm9sZSxcbiAgICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgICBHUkFQSFFMX1VSTDogdXBkYXRlR3JhcGhRbEFwaUVuZHBvaW50LFxuICAgICAgICAgIElOUFVUX0JVQ0tFVDogdGhpcy5zM0lucHV0QXNzZXRzQnVja2V0SW50ZXJmYWNlLmJ1Y2tldE5hbWUsXG4gICAgICAgICAgT1BFTlNFQVJDSF9ET01BSU5fRU5EUE9JTlQ6IHByb3BzLmV4aXN0aW5nT3BlbnNlYXJjaERvbWFpbi5kb21haW5FbmRwb2ludCxcbiAgICAgICAgICBPUEVOU0VBUkNIX0lOREVYOiBwcm9wcy5vcGVuU2VhcmNoSW5kZXhOYW1lLFxuICAgICAgICAgIE9QRU5TRUFSQ0hfU0VDUkVUX0lEOiBTZWNyZXRJZCxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgKTtcblxuXG4gICAgY29uc3QgZW5hYmxlT3BlcmF0aW9uYWxNZXRyaWMgPSBwcm9wcy5lbmFibGVPcGVyYXRpb25hbE1ldHJpYyB8fCB0cnVlO1xuICAgIGNvbnN0IHNvbHV0aW9uX2lkID0gJ2dlbmFpX2Nka18nK2lkO1xuXG4gICAgaWYgKGVuYWJsZU9wZXJhdGlvbmFsTWV0cmljKSB7XG4gICAgICBxdWVzdGlvbl9hbnN3ZXJpbmdfZnVuY3Rpb24uYWRkRW52aXJvbm1lbnQoXG4gICAgICAgICdBV1NfU0RLX1VBX0FQUF9JRCcsIHNvbHV0aW9uX2lkLFxuICAgICAgKTtcbiAgICB9O1xuXG4gICAgLy8gQWRkIEdyYXBoUWwgcGVybWlzc2lvbnMgdG8gdGhlIElBTSByb2xlIGZvciB0aGUgTGFtYmRhIGZ1bmN0aW9uXG4gICAgcXVlc3Rpb25fYW5zd2VyaW5nX2Z1bmN0aW9uLmFkZFRvUm9sZVBvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICBhY3Rpb25zOiBbXG4gICAgICAgICdhcHBzeW5jOkdyYXBoUUwnLFxuICAgICAgXSxcbiAgICAgIHJlc291cmNlczogW1xuICAgICAgICAnYXJuOmF3czphcHBzeW5jOicrIEF3cy5SRUdJT04rJzonK0F3cy5BQ0NPVU5UX0lEKyc6YXBpcy8nK3VwZGF0ZUdyYXBoUWxBcGlJZCsnLyonLFxuICAgICAgXSxcbiAgICB9KSk7XG5cbiAgICB0aGlzLnFhQnVzLmdyYW50UHV0RXZlbnRzVG8oZXZlbnRfYnJpZGdlX2RhdGFzb3VyY2UuZ3JhbnRQcmluY2lwYWwpO1xuXG4gICAgZXZlbnRfYnJpZGdlX2RhdGFzb3VyY2UuY3JlYXRlUmVzb2x2ZXIoXG4gICAgICAnUXVlc3Rpb25BbnN3ZXJpbmdSZXNvbHZlcicsXG4gICAgICB7XG4gICAgICAgIGZpZWxkTmFtZTogJ3Bvc3RRdWVzdGlvbicsXG4gICAgICAgIHR5cGVOYW1lOiAnTXV0YXRpb24nLFxuICAgICAgICByZXF1ZXN0TWFwcGluZ1RlbXBsYXRlOiBhcHBzeW5jLk1hcHBpbmdUZW1wbGF0ZS5mcm9tU3RyaW5nKFxuICAgICAgICAgIGBcbiAgICAgICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcInZlcnNpb25cIjogXCIyMDE4LTA1LTI5XCIsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgXCJvcGVyYXRpb25cIjogXCJQdXRFdmVudHNcIixcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcImV2ZW50c1wiOiBbe1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcInNvdXJjZVwiOiBcInF1ZXN0aW9uYW5zd2VyaW5nXCIsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFwiZGV0YWlsXCI6ICR1dGlsLnRvSnNvbigkY29udGV4dC5hcmd1bWVudHMpLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcImRldGFpbFR5cGVcIjogXCJRdWVzdGlvbiBhbnN3ZXJpbmdcIlxuICAgICAgICAgICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgICAgICAgICAgICB9IFxuICAgICAgICAgICAgICAgICAgICAgICAgYCxcbiAgICAgICAgKSxcbiAgICAgICAgcmVzcG9uc2VNYXBwaW5nVGVtcGxhdGU6IGFwcHN5bmMuTWFwcGluZ1RlbXBsYXRlLmZyb21TdHJpbmcoXG4gICAgICAgICAgYFxuICAgICAgICAgICAgICAgICAgICAgICAgI2lmKCRjdHguZXJyb3IpXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgJHV0aWwuZXJyb3IoJGN0eC5lcnJvci5tZXNzYWdlLCAkY3R4LmVycm9yLnR5cGUsICRjdHgucmVzdWx0KVxuICAgICAgICAgICAgICAgICAgICAgICAgI2VuZFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICR1dGlsLnRvSnNvbigkY3R4LnJlc3VsdClcbiAgICAgICAgICAgICAgICAgICAgICAgIGAsXG4gICAgICAgICksXG4gICAgICB9LFxuICAgICk7XG5cbiAgICBjb25zdCBydWxlID0gbmV3IGV2ZW50cy5SdWxlKFxuICAgICAgdGhpcyxcbiAgICAgICdRdWVzdGlvbkFuc3dlcmluZ1J1bGUnK3N0YWdlLFxuICAgICAge1xuICAgICAgICBkZXNjcmlwdGlvbjogJ1J1bGUgdG8gdHJpZ2dlciBxdWVzdGlvbiBhbnN3ZXJpbmcgZnVuY3Rpb24nLFxuICAgICAgICBldmVudEJ1czogdGhpcy5xYUJ1cyxcbiAgICAgICAgZXZlbnRQYXR0ZXJuOiB7XG4gICAgICAgICAgc291cmNlOiBbJ3F1ZXN0aW9uYW5zd2VyaW5nJ10sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgICk7XG5cbiAgICBydWxlLmFkZFRhcmdldChuZXcgdGFyZ2V0cy5MYW1iZGFGdW5jdGlvbihxdWVzdGlvbl9hbnN3ZXJpbmdfZnVuY3Rpb24pKTtcblxuICB9XG59Il19