"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JumpStartSageMakerEndpoint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const iam = require("aws-cdk-lib/aws-iam");
const sagemaker = require("aws-cdk-lib/aws-sagemaker");
const core_1 = require("aws-cdk-lib/core");
const jumpstart_constants_1 = require("./private/jumpstart-constants");
const sagemaker_endpoint_base_1 = require("./sagemaker-endpoint-base");
/**
 * @summary The JumpStartSageMakerEndpoint class.
 */
class JumpStartSageMakerEndpoint extends sagemaker_endpoint_base_1.SageMakerEndpointBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.model = props.model;
        this.instanceType = props.instanceType;
        this.instanceCount = Math.max(1, props.instanceCount ?? 1);
        this.role = props.role ?? this.createSageMakerRole();
        this.grantPrincipal = this.role;
        this.startupHealthCheckTimeoutInSeconds = props.startupHealthCheckTimeoutInSeconds ?? 600;
        this.environment = props.environment;
        this.spec = this.model.bind();
        this.region = core_1.Stack.of(this).region;
        if (core_1.Token.isUnresolved(this.region)) {
            throw new Error('Region is unresolved. You should explicitly specify the region in the environment.');
        }
        const instanceType = this.verifyInstanceType();
        const instanseBaseType = instanceType.split('.')[1];
        let model;
        if (this.spec.modelPackageArns) {
            if (this.environment) {
                throw new Error('Environment variables are not supported for model packages.');
            }
            model = this.getModelFromPackage(scope, id);
        }
        else {
            const environment = this.buildEnvironment(instanceType);
            model = this.getModelFromArtifact(scope, id, instanceType, instanseBaseType, environment);
        }
        const endpointConfig = new sagemaker.CfnEndpointConfig(scope, `EndpointConfig-${id}`, {
            productionVariants: [
                {
                    instanceType,
                    initialVariantWeight: 1,
                    initialInstanceCount: this.instanceCount,
                    variantName: 'AllTraffic',
                    modelName: model.getAtt('ModelName').toString(),
                    containerStartupHealthCheckTimeoutInSeconds: this.startupHealthCheckTimeoutInSeconds,
                },
            ],
        });
        endpointConfig.addDependency(model);
        const endpoint = new sagemaker.CfnEndpoint(scope, `${this.spec.modelId}-endpoint-${id}`, {
            endpointConfigName: endpointConfig.getAtt('EndpointConfigName').toString(),
            endpointName: props.endpointName,
        });
        endpoint.addDependency(endpointConfig);
        this.cfnModel = model;
        this.cfnEndpoint = endpoint;
        this.cfnEndpointConfig = endpointConfig;
        this.endpointArn = endpoint.ref;
    }
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPolicy(statement);
    }
    grantInvoke(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['sagemaker:InvokeEndpoint'],
            resourceArns: [this.endpointArn],
        });
    }
    verifyInstanceType() {
        let instanceType = this.spec.defaultInstanceType;
        if (this.instanceType) {
            instanceType = this.instanceType.toString();
        }
        const supportedInstanceTypes = this.spec.instanceTypes;
        if (!supportedInstanceTypes.includes(instanceType)) {
            throw new Error(`The instance type ${instanceType} is not supported. Default instance type: ${this.spec.defaultInstanceType}. Supported instance types: ${supportedInstanceTypes.join(', ')}.`);
        }
        return instanceType;
    }
    buildEnvironment(instanceType) {
        const configEnvironment = this.spec.instanceVariants?.find((v) => v.instanceType === instanceType)?.environment;
        const environment = {
            ...(this.spec.environment ?? {}),
            ...configEnvironment,
            ...this.environment,
        };
        return environment;
    }
    getModelFromArtifact(scope, id, instanceType, instanceBaseType, environment) {
        const key = this.spec.prepackedArtifactKey ?? this.spec.artifactKey;
        const bucket = jumpstart_constants_1.JumpStartConstants.JUMPSTART_LAUNCHED_REGIONS[this.region]?.contentBucket;
        if (!bucket) {
            throw new Error(`JumpStart is not available in the region ${this.region}.`);
        }
        const modelDataUrl = `s3://${bucket}/${key}`;
        const imageUriKey = this.spec.instanceVariants
            ?.find((v) => v.instanceType === instanceBaseType)
            ?.imageUri?.replace('$', '');
        if (!imageUriKey) {
            throw new Error(`The image uri is not available for instance type ${instanceType}.`);
        }
        const image = this.spec.instanceAliases?.find((v) => v.region === this.region)?.aliases[imageUriKey];
        if (!image) {
            throw new Error(`The image uri is not available for instance type ${instanceType} in region ${this.region}.`);
        }
        const model = new sagemaker.CfnModel(scope, `${this.spec.modelId}-model-${id}`, {
            executionRoleArn: this.role.roleArn,
            enableNetworkIsolation: true,
            primaryContainer: {
                image,
                modelDataUrl,
                environment,
            },
            tags: [
                {
                    key: 'modelId',
                    value: this.spec.modelId,
                },
                {
                    key: 'modelVersion',
                    value: this.spec.version,
                },
            ],
        });
        return model;
    }
    getModelFromPackage(scope, id) {
        const modelPackageArns = this.spec.modelPackageArns || {};
        const supportedRegions = Object.keys(modelPackageArns);
        if (!supportedRegions.includes(this.region)) {
            throw new Error(`The model package is not available in the region ${this.region}. Supported regions: ${supportedRegions.join(', ')}.`);
        }
        const modelPackageArn = modelPackageArns[this.region];
        const model = new sagemaker.CfnModel(scope, `${this.spec.modelId}-model-${id}`, {
            executionRoleArn: this.role.roleArn,
            enableNetworkIsolation: true,
            primaryContainer: {
                modelPackageName: modelPackageArn,
            },
            tags: [
                {
                    key: 'modelId',
                    value: this.spec.modelId,
                },
                {
                    key: 'modelVersion',
                    value: this.spec.version,
                },
            ],
        });
        return model;
    }
}
_a = JSII_RTTI_SYMBOL_1;
JumpStartSageMakerEndpoint[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.JumpStartSageMakerEndpoint", version: "0.0.89" };
exports.JumpStartSageMakerEndpoint = JumpStartSageMakerEndpoint;
//# sourceMappingURL=data:application/json;base64,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