"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GenerateUtils = void 0;
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const fs = require("fs");
const https = require("https");
const path = require("path");
class GenerateUtils {
    static replaceAll(str, find, replace) {
        return str.replace(new RegExp(find, 'g'), replace);
    }
    static replaceAllBatch(str, find, replace) {
        for (const f of find) {
            str = str.replace(new RegExp(f, 'g'), replace);
        }
        return str;
    }
    static writeFileSyncWithDirs(filePath, data, options) {
        const dir = path.dirname(filePath);
        if (!fs.existsSync(dir)) {
            fs.mkdirSync(dir, { recursive: true });
        }
        fs.writeFileSync(filePath, data, options);
    }
    static async downloadJSON(options) {
        let retryCount = 0;
        let error;
        let result;
        while (retryCount < 5) {
            try {
                result = await new Promise((resolve, reject) => {
                    const request = https.get(options, (response) => {
                        if (response.statusCode === 200) {
                            let rawData = '';
                            response.setEncoding('utf8');
                            response.on('data', (chunk) => {
                                rawData += chunk;
                            });
                            response.on('error', (e) => {
                                reject(e);
                            });
                            response.on('end', () => {
                                try {
                                    const parsedData = JSON.parse(rawData);
                                    resolve([parsedData, response]);
                                }
                                catch (e) {
                                    reject(e);
                                }
                            });
                        }
                        else {
                            reject(new Error(`Request Failed. Status Code: ${response.statusCode}`));
                        }
                    });
                    request.end();
                });
                break;
            }
            catch (e) {
                error = e;
                retryCount++;
                console.log('Retrying in 1 second...');
                await new Promise((resolve) => setTimeout(resolve, 1000 * retryCount));
            }
        }
        if (error) {
            throw error;
        }
        return result;
    }
}
exports.GenerateUtils = GenerateUtils;
//# sourceMappingURL=data:application/json;base64,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