"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CheckRedisClusterProps = exports.setInboundRules = exports.getRedisSecurityGroup = exports.buildRedisCluster = void 0;
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const ec2 = require("aws-cdk-lib/aws-ec2");
const elasticache = require("aws-cdk-lib/aws-elasticache");
function buildRedisCluster(scope, props) {
    const cacheNodeType = props.cfnCacheClusterProps?.cacheNodeType || 'cache.r6g.xlarge';
    const engine = props.cfnCacheClusterProps?.engine || 'redis';
    const numCacheNodes = props.cfnCacheClusterProps?.numCacheNodes || 1;
    let redisCulster = new elasticache.CfnCacheCluster(scope, 'redisCluster', {
        cacheNodeType: cacheNodeType,
        engine: engine,
        numCacheNodes: numCacheNodes,
        cacheSubnetGroupName: getRedisSubnetGroup(scope, props).ref,
        vpcSecurityGroupIds: [props.redisSecurityGroup.securityGroupId],
        port: props.redisPort,
    });
    return redisCulster;
}
exports.buildRedisCluster = buildRedisCluster;
// get redis subnet group from existing vpc
function getRedisSubnetGroup(scope, props) {
    let redisSubnetGroup = new elasticache.CfnSubnetGroup(scope, 'redisSubnetGroup', {
        description: 'Redis subnet group',
        subnetIds: props.subnetIds,
    });
    return redisSubnetGroup;
}
// get redis security group from existing vpc
function getRedisSecurityGroup(scope, props) {
    let redisSecurityGroup = new ec2.SecurityGroup(scope, 'redisSecurityGroup', {
        vpc: props.existingVpc,
        allowAllOutbound: true,
        description: 'security group for elasticache',
    });
    return redisSecurityGroup;
}
exports.getRedisSecurityGroup = getRedisSecurityGroup;
function setInboundRules(redisSecurityGroup, sourceSecuritygroup, redisPort) {
    redisSecurityGroup.connections.allowFrom(sourceSecuritygroup, ec2.Port.tcp(redisPort));
}
exports.setInboundRules = setInboundRules;
function CheckRedisClusterProps(propsObject) {
    let errorMessages = '';
    let errorFound = false;
    if (propsObject.existingRedisCulster && propsObject.cfnCacheClusterProps) {
        errorMessages += 'Error - Either provide existingRedisCulster or cfnCacheClusterProps, but not both.\n';
        errorFound = true;
    }
    if (errorFound) {
        throw new Error(errorMessages);
    }
}
exports.CheckRedisClusterProps = CheckRedisClusterProps;
//# sourceMappingURL=data:application/json;base64,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