"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualNode = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const mesh_1 = require("./mesh");
const utils_1 = require("./private/utils");
class VirtualNodeBase extends cdk.Resource {
    /**
     * Grants the given entity `appmesh:StreamAggregatedResources`.
     *
     * @stability stable
     */
    grantStreamAggregatedResources(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: ['appmesh:StreamAggregatedResources'],
            resourceArns: [this.virtualNodeArn],
        });
    }
}
/**
 * VirtualNode represents a newly defined AppMesh VirtualNode.
 *
 * Any inbound traffic that your virtual node expects should be specified as a
 * listener. Any outbound traffic that your virtual node expects to reach
 * should be specified as a backend.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html
 * @stability stable
 */
class VirtualNode extends VirtualNodeBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f;
        super(scope, id, {
            physicalName: props.virtualNodeName || cdk.Lazy.string({ produce: () => cdk.Names.uniqueId(this) }),
        });
        this.backends = new Array();
        this.listeners = new Array();
        jsiiDeprecationWarnings._aws_cdk_aws_appmesh_VirtualNodeProps(props);
        this.mesh = props.mesh;
        this.serviceDiscoveryConfig = (_b = props.serviceDiscovery) === null || _b === void 0 ? void 0 : _b.bind(this);
        (_c = props.backends) === null || _c === void 0 ? void 0 : _c.forEach(backend => this.addBackend(backend));
        (_d = props.listeners) === null || _d === void 0 ? void 0 : _d.forEach(listener => this.addListener(listener));
        const accessLogging = (_e = props.accessLog) === null || _e === void 0 ? void 0 : _e.bind(this);
        const node = new appmesh_generated_1.CfnVirtualNode(this, 'Resource', {
            virtualNodeName: this.physicalName,
            meshName: this.mesh.meshName,
            meshOwner: utils_1.renderMeshOwner(this.env.account, this.mesh.env.account),
            spec: {
                backends: cdk.Lazy.any({ produce: () => this.backends }, { omitEmptyArray: true }),
                listeners: cdk.Lazy.any({ produce: () => this.listeners.map(listener => listener.listener) }, { omitEmptyArray: true }),
                backendDefaults: props.backendDefaults !== undefined
                    ? {
                        clientPolicy: {
                            tls: utils_1.renderTlsClientPolicy(this, (_f = props.backendDefaults) === null || _f === void 0 ? void 0 : _f.tlsClientPolicy),
                        },
                    }
                    : undefined,
                serviceDiscovery: renderServiceDiscovery(this.serviceDiscoveryConfig),
                logging: accessLogging !== undefined ? {
                    accessLog: accessLogging.virtualNodeAccessLog,
                } : undefined,
            },
        });
        this.virtualNodeName = this.getResourceNameAttribute(node.attrVirtualNodeName);
        this.virtualNodeArn = this.getResourceArnAttribute(node.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualNode`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualNode given an ARN.
     *
     * @stability stable
     */
    static fromVirtualNodeArn(scope, id, virtualNodeArn) {
        return new class extends VirtualNodeBase {
            constructor() {
                super(...arguments);
                this.virtualNodeArn = virtualNodeArn;
                this.parsedArn = cdk.Fn.split('/', cdk.Stack.of(scope).splitArn(virtualNodeArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName);
                this.mesh = mesh_1.Mesh.fromMeshName(this, 'Mesh', cdk.Fn.select(0, this.parsedArn));
                this.virtualNodeName = cdk.Fn.select(2, this.parsedArn);
            }
        }(scope, id);
    }
    /**
     * Import an existing VirtualNode given its name.
     *
     * @stability stable
     */
    static fromVirtualNodeAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_appmesh_VirtualNodeAttributes(attrs);
        return new class extends VirtualNodeBase {
            constructor() {
                super(...arguments);
                this.mesh = attrs.mesh;
                this.virtualNodeName = attrs.virtualNodeName;
                this.virtualNodeArn = cdk.Stack.of(this).formatArn({
                    service: 'appmesh',
                    resource: `mesh/${attrs.mesh.meshName}/virtualNode`,
                    resourceName: this.virtualNodeName,
                });
            }
        }(scope, id);
    }
    /**
     * Utility method to add an inbound listener for this VirtualNode.
     *
     * Note: At this time, Virtual Nodes support at most one listener. Adding
     * more than one will result in a failure to deploy the CloudFormation stack.
     * However, the App Mesh team has plans to add support for multiple listeners
     * on Virtual Nodes and Virtual Routers.
     *
     * @see https://github.com/aws/aws-app-mesh-roadmap/issues/120
     * @stability stable
     */
    addListener(listener) {
        jsiiDeprecationWarnings._aws_cdk_aws_appmesh_VirtualNodeListener(listener);
        if (!this.serviceDiscoveryConfig) {
            throw new Error('Service discovery information is required for a VirtualNode with a listener.');
        }
        this.listeners.push(listener.bind(this));
    }
    /**
     * Add a Virtual Services that this node is expected to send outbound traffic to.
     *
     * @stability stable
     */
    addBackend(backend) {
        jsiiDeprecationWarnings._aws_cdk_aws_appmesh_Backend(backend);
        this.backends.push(backend.bind(this).virtualServiceBackend);
    }
}
exports.VirtualNode = VirtualNode;
_a = JSII_RTTI_SYMBOL_1;
VirtualNode[_a] = { fqn: "@aws-cdk/aws-appmesh.VirtualNode", version: "1.144.0" };
function renderServiceDiscovery(config) {
    return config
        ? {
            dns: config === null || config === void 0 ? void 0 : config.dns,
            awsCloudMap: config === null || config === void 0 ? void 0 : config.cloudmap,
        }
        : undefined;
}
//# sourceMappingURL=data:application/json;base64,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