# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['parse']
setup_kwargs = {
    'name': 'parse-with-dot-access',
    'version': '1.17.0',
    'description': 'a fork of parse with dot access',
    'long_description': '# a fork of parse with dot access\n\n\nParse strings using a specification based on the Python format() syntax.\n\n   ``parse()`` is the opposite of ``format()``\n\nThe module is set up to only export ``parse()``, ``search()``, ``findall()``,\nand ``with_pattern()`` when ``import \\*`` is used:\n\n>>> from parse import *\n\nFrom there it\'s a simple thing to parse a string:\n\n.. code-block:: pycon\n\n    >>> parse("It\'s {}, I love it!", "It\'s spam, I love it!")\n    <Result (\'spam\',) {}>\n    >>> _[0]\n    \'spam\'\n\nOr to search a string for some pattern:\n\n.. code-block:: pycon\n\n    >>> search(\'Age: {:d}\\n\', \'Name: Rufus\\nAge: 42\\nColor: red\\n\')\n    <Result (42,) {}>\n\nOr find all the occurrences of some pattern in a string:\n\n.. code-block:: pycon\n\n    >>> \'\'.join(r[0] for r in findall(">{}<", "<p>the <b>bold</b> text</p>"))\n    \'the bold text\'\n\nIf you\'re going to use the same pattern to match lots of strings you can\ncompile it once:\n\n.. code-block:: pycon\n\n    >>> from parse import compile\n    >>> p = compile("It\'s {}, I love it!")\n    >>> print(p)\n    <Parser "It\'s {}, I love it!">\n    >>> p.parse("It\'s spam, I love it!")\n    <Result (\'spam\',) {}>\n\n("compile" is not exported for ``import *`` usage as it would override the\nbuilt-in ``compile()`` function)\n\nThe default behaviour is to match strings case insensitively. You may match with\ncase by specifying `case_sensitive=True`:\n\n.. code-block:: pycon\n\n    >>> parse(\'SPAM\', \'spam\', case_sensitive=True) is None\n    True\n\n\nFormat Syntax\n-------------\n\nA basic version of the `Format String Syntax`_ is supported with anonymous\n(fixed-position), named and formatted fields::\n\n   {[field name]:[format spec]}\n\nField names must be a valid Python identifiers, including dotted names;\nelement indexes imply dictionaries (see below for example).\n\nNumbered fields are also not supported: the result of parsing will include\nthe parsed fields in the order they are parsed.\n\nThe conversion of fields to types other than strings is done based on the\ntype in the format specification, which mirrors the ``format()`` behaviour.\nThere are no "!" field conversions like ``format()`` has.\n\nSome simple parse() format string examples:\n\n.. code-block:: pycon\n\n    >>> parse("Bring me a {}", "Bring me a shrubbery")\n    <Result (\'shrubbery\',) {}>\n    >>> r = parse("The {} who say {}", "The knights who say Ni!")\n    >>> print(r)\n    <Result (\'knights\', \'Ni!\') {}>\n    >>> print(r.fixed)\n    (\'knights\', \'Ni!\')\n    >>> r = parse("Bring out the holy {item}", "Bring out the holy hand grenade")\n    >>> print(r)\n    <Result () {\'item\': \'hand grenade\'}>\n    >>> print(r.named)\n    {\'item\': \'hand grenade\'}\n    >>> print(r[\'item\'])\n    hand grenade\n    >>> \'item\' in r\n    True\n\nNote that `in` only works if you have named fields. Dotted names and indexes\nare possible though the application must make additional sense of the result:\n\n.. code-block:: pycon\n\n    >>> r = parse("Mmm, {food.type}, I love it!", "Mmm, spam, I love it!")\n    >>> print(r)\n    <Result () {\'food.type\': \'spam\'}>\n    >>> print(r.named)\n    {\'food.type\': \'spam\'}\n    >>> print(r[\'food.type\'])\n    spam\n    >>> r = parse("My quest is {quest[name]}", "My quest is to seek the holy grail!")\n    >>> print(r)\n    <Result () {\'quest\': {\'name\': \'to seek the holy grail!\'}}>\n    >>> print(r[\'quest\'])\n    {\'name\': \'to seek the holy grail!\'}\n    >>> print(r[\'quest\'][\'name\'])\n    to seek the holy grail!\n\nIf the text you\'re matching has braces in it you can match those by including\na double-brace ``{{`` or ``}}`` in your format string, just like format() does.\n\n\nFormat Specification\n--------------------\n\nMost often a straight format-less ``{}`` will suffice where a more complex\nformat specification might have been used.\n\nMost of `format()`\'s `Format Specification Mini-Language`_ is supported:\n\n   [[fill]align][0][width][.precision][type]\n\nThe differences between `parse()` and `format()` are:\n\n- The align operators will cause spaces (or specified fill character) to be\n  stripped from the parsed value. The width is not enforced; it just indicates\n  there may be whitespace or "0"s to strip.\n- Numeric parsing will automatically handle a "0b", "0o" or "0x" prefix.\n  That is, the "#" format character is handled automatically by d, b, o\n  and x formats. For "d" any will be accepted, but for the others the correct\n  prefix must be present if at all.\n- Numeric sign is handled automatically.\n- The thousands separator is handled automatically if the "n" type is used.\n- The types supported are a slightly different mix to the format() types.  Some\n  format() types come directly over: "d", "n", "%", "f", "e", "b", "o" and "x".\n  In addition some regular expression character group types "D", "w", "W", "s"\n  and "S" are also available.\n- The "e" and "g" types are case-insensitive so there is not need for\n  the "E" or "G" types. The "e" type handles Fortran formatted numbers (no\n  leading 0 before the decimal point).\n\n===== =========================================== ========\nType  Characters Matched                          Output\n===== =========================================== ========\nl     Letters (ASCII)                             str\nw     Letters, numbers and underscore             str\nW     Not letters, numbers and underscore         str\ns     Whitespace                                  str\nS     Non-whitespace                              str\nd     Digits (effectively integer numbers)        int\nD     Non-digit                                   str\nn     Numbers with thousands separators (, or .)  int\n%     Percentage (converted to value/100.0)       float\nf     Fixed-point numbers                         float\nF     Decimal numbers                             Decimal\ne     Floating-point numbers with exponent        float\n      e.g. 1.1e-10, NAN (all case insensitive)\ng     General number format (either d, f or e)    float\nb     Binary numbers                              int\no     Octal numbers                               int\nx     Hexadecimal numbers (lower and upper case)  int\nti    ISO 8601 format date/time                   datetime\n      e.g. 1972-01-20T10:21:36Z ("T" and "Z"\n      optional)\nte    RFC2822 e-mail format date/time             datetime\n      e.g. Mon, 20 Jan 1972 10:21:36 +1000\ntg    Global (day/month) format date/time         datetime\n      e.g. 20/1/1972 10:21:36 AM +1:00\nta    US (month/day) format date/time             datetime\n      e.g. 1/20/1972 10:21:36 PM +10:30\ntc    ctime() format date/time                    datetime\n      e.g. Sun Sep 16 01:03:52 1973\nth    HTTP log format date/time                   datetime\n      e.g. 21/Nov/2011:00:07:11 +0000\nts    Linux system log format date/time           datetime\n      e.g. Nov  9 03:37:44\ntt    Time                                        time\n      e.g. 10:21:36 PM -5:30\n===== =========================================== ========\n\nSome examples of typed parsing with ``None`` returned if the typing\ndoes not match:\n\n.. code-block:: pycon\n\n    >>> parse(\'Our {:d} {:w} are...\', \'Our 3 weapons are...\')\n    <Result (3, \'weapons\') {}>\n    >>> parse(\'Our {:d} {:w} are...\', \'Our three weapons are...\')\n    >>> parse(\'Meet at {:tg}\', \'Meet at 1/2/2011 11:00 PM\')\n    <Result (datetime.datetime(2011, 2, 1, 23, 0),) {}>\n\nAnd messing about with alignment:\n\n.. code-block:: pycon\n\n    >>> parse(\'with {:>} herring\', \'with     a herring\')\n    <Result (\'a\',) {}>\n    >>> parse(\'spam {:^} spam\', \'spam    lovely     spam\')\n    <Result (\'lovely\',) {}>\n\nNote that the "center" alignment does not test to make sure the value is\ncentered - it just strips leading and trailing whitespace.\n\nWidth and precision may be used to restrict the size of matched text\nfrom the input. Width specifies a minimum size and precision specifies\na maximum. For example:\n\n.. code-block:: pycon\n\n    >>> parse(\'{:.2}{:.2}\', \'look\')           # specifying precision\n    <Result (\'lo\', \'ok\') {}>\n    >>> parse(\'{:4}{:4}\', \'look at that\')     # specifying width\n    <Result (\'look\', \'at that\') {}>\n    >>> parse(\'{:4}{:.4}\', \'look at that\')    # specifying both\n    <Result (\'look at \', \'that\') {}>\n    >>> parse(\'{:2d}{:2d}\', \'0440\')           # parsing two contiguous numbers\n    <Result (4, 40) {}>\n\nSome notes for the date and time types:\n\n- the presence of the time part is optional (including ISO 8601, starting\n  at the "T"). A full datetime object will always be returned; the time\n  will be set to 00:00:00. You may also specify a time without seconds.\n- when a seconds amount is present in the input fractions will be parsed\n  to give microseconds.\n- except in ISO 8601 the day and month digits may be 0-padded.\n- the date separator for the tg and ta formats may be "-" or "/".\n- named months (abbreviations or full names) may be used in the ta and tg\n  formats in place of numeric months.\n- as per RFC 2822 the e-mail format may omit the day (and comma), and the\n  seconds but nothing else.\n- hours greater than 12 will be happily accepted.\n- the AM/PM are optional, and if PM is found then 12 hours will be added\n  to the datetime object\'s hours amount - even if the hour is greater\n  than 12 (for consistency.)\n- in ISO 8601 the "Z" (UTC) timezone part may be a numeric offset\n- timezones are specified as "+HH:MM" or "-HH:MM". The hour may be one or two\n  digits (0-padded is OK.) Also, the ":" is optional.\n- the timezone is optional in all except the e-mail format (it defaults to\n  UTC.)\n- named timezones are not handled yet.\n\nNote: attempting to match too many datetime fields in a single parse() will\ncurrently result in a resource allocation issue. A TooManyFields exception\nwill be raised in this instance. The current limit is about 15. It is hoped\nthat this limit will be removed one day.\n\n.. _`Format String Syntax`:\n  http://docs.python.org/library/string.html#format-string-syntax\n.. _`Format Specification Mini-Language`:\n  http://docs.python.org/library/string.html#format-specification-mini-language\n\n\nResult and Match Objects\n------------------------\n\nThe result of a ``parse()`` and ``search()`` operation is either ``None`` (no match), a\n``Result`` instance or a ``Match`` instance if ``evaluate_result`` is False.\n\nThe ``Result`` instance has three attributes:\n\n``fixed``\n   A tuple of the fixed-position, anonymous fields extracted from the input.\n``named``\n   A dictionary of the named fields extracted from the input.\n``spans``\n   A dictionary mapping the names and fixed position indices matched to a\n   2-tuple slice range of where the match occurred in the input.\n   The span does not include any stripped padding (alignment or width).\n\nThe ``Match`` instance has one method:\n\n``evaluate_result()``\n   Generates and returns a ``Result`` instance for this ``Match`` object.\n\n\n\nCustom Type Conversions\n-----------------------\n\nIf you wish to have matched fields automatically converted to your own type you\nmay pass in a dictionary of type conversion information to ``parse()`` and\n``compile()``.\n\nThe converter will be passed the field string matched. Whatever it returns\nwill be substituted in the ``Result`` instance for that field.\n\nYour custom type conversions may override the builtin types if you supply one\nwith the same identifier:\n\n.. code-block:: pycon\n\n    >>> def shouty(string):\n    ...    return string.upper()\n    ...\n    >>> parse(\'{:shouty} world\', \'hello world\', dict(shouty=shouty))\n    <Result (\'HELLO\',) {}>\n\nIf the type converter has the optional ``pattern`` attribute, it is used as\nregular expression for better pattern matching (instead of the default one):\n\n.. code-block:: pycon\n\n    >>> def parse_number(text):\n    ...    return int(text)\n    >>> parse_number.pattern = r\'\\d+\'\n    >>> parse(\'Answer: {number:Number}\', \'Answer: 42\', dict(Number=parse_number))\n    <Result () {\'number\': 42}>\n    >>> _ = parse(\'Answer: {:Number}\', \'Answer: Alice\', dict(Number=parse_number))\n    >>> assert _ is None, "MISMATCH"\n\nYou can also use the ``with_pattern(pattern)`` decorator to add this\ninformation to a type converter function:\n\n.. code-block:: pycon\n\n    >>> from parse import with_pattern\n    >>> @with_pattern(r\'\\d+\')\n    ... def parse_number(text):\n    ...    return int(text)\n    >>> parse(\'Answer: {number:Number}\', \'Answer: 42\', dict(Number=parse_number))\n    <Result () {\'number\': 42}>\n\nA more complete example of a custom type might be:\n\n.. code-block:: pycon\n\n    >>> yesno_mapping = {\n    ...     "yes":  True,   "no":    False,\n    ...     "on":   True,   "off":   False,\n    ...     "true": True,   "false": False,\n    ... }\n    >>> @with_pattern(r"|".join(yesno_mapping))\n    ... def parse_yesno(text):\n    ...     return yesno_mapping[text.lower()]\n\n\nIf the type converter ``pattern`` uses regex-grouping (with parenthesis),\nyou should indicate this by using the optional ``regex_group_count`` parameter\nin the ``with_pattern()`` decorator:\n\n.. code-block:: pycon\n\n    >>> @with_pattern(r\'((\\d+))\', regex_group_count=2)\n    ... def parse_number2(text):\n    ...    return int(text)\n    >>> parse(\'Answer: {:Number2} {:Number2}\', \'Answer: 42 43\', dict(Number2=parse_number2))\n    <Result (42, 43) {}>\n\nOtherwise, this may cause parsing problems with unnamed/fixed parameters.\n\n\nPotential Gotchas\n-----------------\n\n``parse()`` will always match the shortest text necessary (from left to right)\nto fulfil the parse pattern, so for example:\n\n\n.. code-block:: pycon\n\n    >>> pattern = \'{dir1}/{dir2}\'\n    >>> data = \'root/parent/subdir\'\n    >>> sorted(parse(pattern, data).named.items())\n    [(\'dir1\', \'root\'), (\'dir2\', \'parent/subdir\')]\n\nSo, even though `{\'dir1\': \'root/parent\', \'dir2\': \'subdir\'}` would also fit\nthe pattern, the actual match represents the shortest successful match for\n``dir1``.\n\n----\n\n**Version history (in brief)**:\n\n- 1.17.0 Make left- and center-aligned search consume up to next space\n- 1.16.0 Make compiled parse objects pickleable (thanks @martinResearch)\n- 1.15.0 Several fixes for parsing non-base 10 numbers (thanks @vladikcomper)\n- 1.14.0 More broad acceptance of Fortran number format (thanks @purpleskyfall)\n- 1.13.1 Project metadata correction.\n- 1.13.0 Handle Fortran formatted numbers with no leading 0 before decimal\n  point (thanks @purpleskyfall).\n  Handle comparison of FixedTzOffset with other types of object.\n- 1.12.1 Actually use the `case_sensitive` arg in compile (thanks @jacquev6)\n- 1.12.0 Do not assume closing brace when an opening one is found (thanks @mattsep)\n- 1.11.1 Revert having unicode char in docstring, it breaks Bamboo builds(?!)\n- 1.11.0 Implement `__contains__` for Result instances.\n- 1.10.0 Introduce a "letters" matcher, since "w" matches numbers\n  also.\n- 1.9.1 Fix deprecation warnings around backslashes in regex strings\n  (thanks Mickael Schoentgen). Also fix some documentation formatting\n  issues.\n- 1.9.0 We now honor precision and width specifiers when parsing numbers\n  and strings, allowing parsing of concatenated elements of fixed width\n  (thanks Julia Signell)\n- 1.8.4 Add LICENSE file at request of packagers.\n  Correct handling of AM/PM to follow most common interpretation.\n  Correct parsing of hexadecimal that looks like a binary prefix.\n  Add ability to parse case sensitively.\n  Add parsing of numbers to Decimal with "F" (thanks John Vandenberg)\n- 1.8.3 Add regex_group_count to with_pattern() decorator to support\n  user-defined types that contain brackets/parenthesis (thanks Jens Engel)\n- 1.8.2 add documentation for including braces in format string\n- 1.8.1 ensure bare hexadecimal digits are not matched\n- 1.8.0 support manual control over result evaluation (thanks Timo Furrer)\n- 1.7.0 parse dict fields (thanks Mark Visser) and adapted to allow\n  more than 100 re groups in Python 3.5+ (thanks David King)\n- 1.6.6 parse Linux system log dates (thanks Alex Cowan)\n- 1.6.5 handle precision in float format (thanks Levi Kilcher)\n- 1.6.4 handle pipe "|" characters in parse string (thanks Martijn Pieters)\n- 1.6.3 handle repeated instances of named fields, fix bug in PM time\n  overflow\n- 1.6.2 fix logging to use local, not root logger (thanks Necku)\n- 1.6.1 be more flexible regarding matched ISO datetimes and timezones in\n  general, fix bug in timezones without ":" and improve docs\n- 1.6.0 add support for optional ``pattern`` attribute in user-defined types\n  (thanks Jens Engel)\n- 1.5.3 fix handling of question marks\n- 1.5.2 fix type conversion error with dotted names (thanks Sebastian Thiel)\n- 1.5.1 implement handling of named datetime fields\n- 1.5 add handling of dotted field names (thanks Sebastian Thiel)\n- 1.4.1 fix parsing of "0" in int conversion (thanks James Rowe)\n- 1.4 add __getitem__ convenience access on Result.\n- 1.3.3 fix Python 2.5 setup.py issue.\n- 1.3.2 fix Python 3.2 setup.py issue.\n- 1.3.1 fix a couple of Python 3.2 compatibility issues.\n- 1.3 added search() and findall(); removed compile() from ``import *``\n  export as it overwrites builtin.\n- 1.2 added ability for custom and override type conversions to be\n  provided; some cleanup\n- 1.1.9 to keep things simpler number sign is handled automatically;\n  significant robustification in the face of edge-case input.\n- 1.1.8 allow "d" fields to have number base "0x" etc. prefixes;\n  fix up some field type interactions after stress-testing the parser;\n  implement "%" type.\n- 1.1.7 Python 3 compatibility tweaks (2.5 to 2.7 and 3.2 are supported).\n- 1.1.6 add "e" and "g" field types; removed redundant "h" and "X";\n  removed need for explicit "#".\n- 1.1.5 accept textual dates in more places; Result now holds match span\n  positions.\n- 1.1.4 fixes to some int type conversion; implemented "=" alignment; added\n  date/time parsing with a variety of formats handled.\n- 1.1.3 type conversion is automatic based on specified field types. Also added\n  "f" and "n" types.\n- 1.1.2 refactored, added compile() and limited ``from parse import *``\n- 1.1.1 documentation improvements\n- 1.1.0 implemented more of the `Format Specification Mini-Language`_\n  and removed the restriction on mixing fixed-position and named fields\n- 1.0.0 initial release\n\nThis code is copyright 2012-2020 Richard Jones <richard@python.org>\nSee the end of the source file for the license of use.\n',
    'author': 'Nutchanon Ninyawee',
    'author_email': 'me@nutchanon.org',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/CircleOnCircles/parse',
    'py_modules': modules,
    'python_requires': '>=3.5,<4.0',
}


setup(**setup_kwargs)
