"""
Task Config Validator for hear preprocess tasks
"""

from typing import Dict, Any, List
from schema import Schema, And, Optional, Or, Forbidden
import copy

SPLITS = ["train", "test", "valid"]


def validate_generic_task_config(
    generic_task_config: Dict[str, Any], ignore_extra_keys: bool = True
):
    """
    Validates a task config to be compatible with the hearpreprocess pipeline

    The pipeline runner adds the following keys to the task configuration after
    running this function for validation. Since these keys are added after
    the validation, this function doesnot check these keys -
        * tmp_dir[str] - The location where the luigi intermediate
            work task outputs will be stored
        * mode[str] - One of the different mode names (defined by the
            `modes` key) in which the runner was selected to run the pipeline.
            example - small, 5h, full etc
        * splits[list(str)] - The data splits that will be generated by
            running the pipeline
            These depends on the `split_mode` of the data
            - If the `split_mode` is `presplit_kfold` or `new_split_kfold`,
                the splits will be like -
                    fold01, fold02, ... fold<nfolds>, where nfolds is the
                    number of folds
            - If the `split_mode` is `trainvaltest`, the splits will be -
                train, test, valid

    Some additional Notes on the behaviour of the task configuration -
        * `split_mode` can have three possible values:
            - `trainvaltest` - The pipeline will produce train, test and valid
                data split. The `ExtractMetadata` should explicitly populate
                the `split` column in the metadata with `train`, `test` or
                `valid` values. In case, no rows are marked as `test`
                or `valid`, the train split will be sampled deterministically
                to produce the valid (10 percent of train) and the test
                (10 percent of the test) split.
            - `presplit_kfold` - The pipeline will produce fold00, fold01, ...
                and these should be explicitly populated in the `split`
                columns for each row by the `ExtractMetadata` like
                the `trainvaltest` mode
            - `newsplit_kfold` - The pipeline will produce fold00, fold01, ...
                and these will be produced by deterministically sampling the data
                into folds. In this case, the `ExtractMetadata` should
                populate only one value in the `split` column for all the
                rows.

        * `modes` key in the task configuration helps to override the parameters
            of the task configuration. For example:
                - The 5h, 50h, full, small modes override the
                    max_split_duration_<split/fold> to make different
                    duration for each split/fold of the data
                - The small mode overrides the `download_urls` to download
                    the data from the small version urls
            This key is removed by the runner after overriding the parameters.
            This validator overrides the parameters by each mode defined in the
            configuration and verifies if it is valid

        * If the task configuration is for a tensorflow dataset, additional keys
            needs to be added
            - tfds_task_name (str)
            - tfds_task_version (str)
            - extract_splits (list) - The splits ('train', 'test', 'valid') that
                is present in the tensorflow dataset and should be extracted
                from the tfds source
            This validator checks if the tfds task configuration is correctly
            defined

        * sample_duration can also be set to `None`, rather than integer or
            float, in which case, the audio files in the dataset will not
            be trimmed or padded, rather the original file duration will be
            retained in the output of the pipeline.
            In this case, the max split duration should be set to
            None and no subsampling can be done, as file durations are
            not consistent.
            However, this is only for specific tasks and should not be generally
            used as it is not efficient for downstream pipelines, particularly
            embedding generation in heareval

    Args:
        task_config: Task config to be used with the pipeline
        ignore_extra_keys: Flag for ignoring extra keys in the task configuration.
            Defaults to True

    Raises:
        schema.SchemaError: If the schema doesnot match, this error is raised
            with the information on what didnot match
    """
    assert "split_mode" in generic_task_config, "split_mode key should be defined"
    split_mode: str = generic_task_config["split_mode"]

    assert "modes" in generic_task_config, "modes key should be defined"
    # Value against the `modes` key in the task_config should be a dictionary
    # of keys that needs to be overriden in the task configuration for the mode
    Schema(dict).validate(generic_task_config["modes"])
    task_modes: List = list(generic_task_config["modes"].keys())

    # Validate the generic task config for each mode of the task
    for task_mode in task_modes:
        print(
            f"Validating for '{task_mode}' mode for the "
            f"{generic_task_config['task_name']} task. "
            "If an error occurs in the schema please check the value against "
            "the keys in this task mode."
        )

        task_config: Dict[str, Any] = copy.deepcopy(generic_task_config)
        task_config.update(dict(task_config["modes"][task_mode]))
        del task_config["modes"]

        # GENERIC Taks keys
        # All the keys haveing an `str` method in the `Or` can accept values
        # other than what is specified as well
        schema: Dict[str, Any] = {
            "task_name": str,
            "version": str,
            "embedding_type": Or("scene", "event", str),
            "prediction_type": Or("multiclass", "multilabel", str),
            "split_mode": Or("trainvaltest", "presplit_kfold", "new_split_kfold"),
            # When the sample duration is None, the original audio is retained
            # without any trimming and padding
            "sample_duration": Or(float, int, None),
            "evaluation": Schema([str]),
            "default_mode": Or("5h", "50h", "full", str),
        }
        # DOWNLOAD Source specific keys
        # If the source of data for the task is a tensorflow dataset
        if "tfds_task_name" in task_config:
            schema.update(
                {
                    # The tfds dataset name
                    "tfds_task_name": str,
                    # The tfds dataset name
                    "tfds_task_version": str,
                    # Defines which splits to extract from the tfds dataset source
                    "extract_splits": Schema([Or(*SPLITS)]),
                }
            )
        # If the source of data for the task is source download urls
        else:
            # ignore_extra_keys is true for the download_urls dict
            # as the download source might require a different set of keys
            # and a modified get_download_and_extract_tasks function
            # to be defined for the task in task specific scripts
            schema.update(
                {
                    "download_urls": Schema(
                        [
                            # A set of optional keys are still defined here
                            {
                                # The split name for which the file is being downloaded
                                # This is not restricted to being train, valid or test
                                Optional("split"): str,
                                # If one split is made up of multiple downloads,
                                # each download can have separate names with the same
                                # split name. (Check dcase for example).
                                # Each such download will be in the same split folder
                                Optional("name"): str,
                                # URL for this download
                                Optional("url"): str,
                                # Expected MD5 for the download
                                Optional("md5"): str,
                            }
                        ],
                        ignore_extra_keys=True,
                    )
                }
            )
        # SPLIT Mode specific keys
        if split_mode == "trainvaltest":
            schema.update(
                {
                    # max_task_duration_by_split duration is a dict specifying
                    # the max task duration for each split
                    "max_task_duration_by_split": Schema(
                        {split: Or(int, float, None) for split in SPLITS}
                    ),
                    # nfolds is invalid for this split mode
                    Forbidden(
                        "nfolds",
                        error=(
                            f"nfolds should not be defined for {split_mode} "
                            "split mode"
                        ),
                    ): object,
                    # max_task_duration_by_fold is invalid for this split mode.
                    Forbidden(
                        "max_task_duration_by_fold",
                        error="max_task_duration_by_fold should not be defined for "
                        "{split_mode} split mode",
                    ): object,
                }
            )
            # If the sample duration is set to None, the max_task_duration_by_split
            # should also be None and no subsampling will be done
            if task_config["sample_duration"] is None:
                schema["max_task_duration_by_split"] = Schema(
                    {split: None for split in SPLITS}
                )
        elif split_mode in ["presplit_kfold", "new_split_kfold"]:

            assert (
                "nfolds" in task_config
            ), "nfolds should be defined for {split_mode} split mode."

            schema.update(
                {
                    # nfolds defines the number of folds
                    # If the split_mode is new_split_kfold nfolds is the number of folds
                    # that will be made
                    "nfolds": And(int, lambda nfolds: nfolds >= 3),
                    # max_task_duration_by_fold (in seconds) should be defined
                    # across folds. It should be set to None if the complete fold
                    # is required
                    "max_task_duration_by_fold": Or(int, float, None),
                    # max_task_duration_by_split is invalid for this mode
                    Forbidden(
                        "max_task_duration_by_split",
                        error=(
                            "max_task_duration_by_split should not be defined for "
                            "{split_mode} split_mode"
                        ),
                    ): object,
                }
            )
            # If the sample duration is set to None, the max_task_duration_by_fold
            # should also be None and no subsampling will be done
            if task_config["sample_duration"] is None:
                schema["max_task_duration_by_fold"] = None
        else:
            raise ValueError("Invalid split_mode")

        Schema(schema, ignore_extra_keys=ignore_extra_keys).validate(task_config)
        print("Successfully validated")
