#!/usr/bin/env python3
"""
Pre-processing pipeline for Coughvid Dataset
"""

import logging
from pathlib import Path
from typing import Any, Dict

import luigi
import pandas as pd

import hearpreprocess.pipeline as pipeline

logger = logging.getLogger("luigi-interface")

# Extensions for the audio files in the coughvid dataset
# .wav version is not present in original coughvid dataset,
# but adding it here because the small version will have .wav
# files which are generated by trimming the audio files
VALID_EXTENSIONS = (".wav", ".webm", ".ogg")

generic_task_config = {
    "task_name": "coughvid",
    "version": "v2.0",
    "embedding_type": "scene",
    "prediction_type": "multiclass",
    "split_mode": "new_split_kfold",
    "nfolds": 5,
    # {'mean': 8.57, 'var': 4.66, 'min': 1.86, 'max': 10.19,
    #  '10th': 5.28, '25th': 7.97, '50th': 9.82, '75th': 10.01, '90th': 10.02}
    "sample_duration": 10.19,
    # Original Dataset Paper - https://www.nature.com/articles/s41597-021-00937-4.pdf
    # TODO: Implement AUC or other measure?
    # https://github.com/hearbenchmark/hear2021-secret-tasks/issues/16
    "evaluation": ["top1_acc", "mAP", "d_prime", "aucroc"],
    "download_urls": [
        # test and valid split will be sampled from the train set only
        {
            "split": "train",
            "url": "https://zenodo.org/record/4498364/files/public_dataset.zip",
            "md5": "110d638a292cfa5c3a71a95d60b5b466",
        },
    ],
    "default_mode": "full",
    # Different modes for preprocessing this dataset
    "modes": {
        "full": {
            "max_task_duration_by_fold": None,
        },
        "small": {
            "download_urls": [
                {
                    "split": "train",
                    "url": "https://github.com/kmarufraj/s-task/raw/main/public_dataset-small.zip",  # noqa: E501
                    "md5": "ca3d811f1e533247b531224ca7d12346",
                }
            ],
            "max_task_duration_by_fold": None,
            "sample_duration": 1,
        },
    },
}


class ExtractMetadata(pipeline.ExtractMetadata):
    train = luigi.TaskParameter()

    def requires(self):
        return {"train": self.train}

    def get_requires_metadata(self, split: str) -> pd.DataFrame:

        logger.info(f"Preparing metadata for {split}")

        # Loads and prepares the metadata for a specific split
        split_path = Path(self.requires()[split].workdir).joinpath(split)
        split_path = split_path.joinpath("public_dataset")

        # Read the base metadata to get the label corresponding to the uuid of
        # each file.
        label_map = (
            pd.read_csv(split_path.joinpath("metadata_compiled.csv"))
            # Filter out the data points with null status
            # TODO: https://github.com/hearbenchmark/hear2021-secret-tasks/issues/17
            # Select entries with cough detected probability greater than 0.8
            .loc[lambda df: df["cough_detected"] > 0.8]
            # Select entries with self reported status available
            .loc[lambda df: df["status"].notnull()]
            # The status is the label
            .rename(columns={"status": "label"}).filter(items=["uuid", "label"])
        )

        # Get all the audio files in the split folder
        audio_files = filter(
            lambda p: p.suffix in VALID_EXTENSIONS, list(split_path.glob("*"))
        )
        metadata = (
            pd.DataFrame(audio_files, columns=["relpath"])
            .assign(
                split=split,
                # Get uuid to join with the label map and get the label
                uuid=lambda df: df["relpath"].apply(lambda p: p.stem),
            )
            # Join with the label map and fetch the corresponding
            # label for each file. Drop the uuid as it is no further required
            .merge(label_map, on=["uuid"], how="inner")
            .drop("uuid", axis=1)
        )

        return metadata


def extract_metadata_task(task_config: Dict[str, Any]) -> pipeline.ExtractMetadata:
    # Build the dataset pipeline with the custom metadata configuration task
    download_tasks = pipeline.get_download_and_extract_tasks(task_config)

    return ExtractMetadata(
        outfile="process_metadata.csv", task_config=task_config, **download_tasks
    )
