"use strict";
const cdk = require("@aws-cdk/core");
const appmesh = require("../lib");
let idCounter = 0;
const getNode = (stack) => {
    const mesh = new appmesh.Mesh(stack, `mesh-${++idCounter}`, {
        meshName: 'test-mesh',
    });
    return mesh.addVirtualNode(`virtual-node-${idCounter}`, {
        dnsHostName: 'test-node',
    });
};
module.exports = {
    'interval'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const [min, max] = [5000, 300000];
        // WHEN
        const toThrow = (millis) => getNode(stack).addListeners({
            healthCheck: { interval: cdk.Duration.millis(millis) },
        });
        // THEN
        test.doesNotThrow(() => toThrow(min));
        test.doesNotThrow(() => toThrow(max));
        test.throws(() => toThrow(min - 1), /below the minimum threshold/);
        test.throws(() => toThrow(max + 1), /above the maximum threshold/);
        test.done();
    },
    'timeout'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const [min, max] = [2000, 60000];
        // WHEN
        const toThrow = (millis) => getNode(stack).addListeners({
            healthCheck: { timeout: cdk.Duration.millis(millis) },
        });
        // THEN
        test.doesNotThrow(() => toThrow(min));
        test.doesNotThrow(() => toThrow(max));
        test.throws(() => toThrow(min - 1), /below the minimum threshold/);
        test.throws(() => toThrow(max + 1), /above the maximum threshold/);
        test.done();
    },
    'port'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const [min, max] = [1, 65535];
        // WHEN
        const toThrow = (port) => getNode(stack).addListeners({
            healthCheck: { port },
        });
        // THEN
        test.doesNotThrow(() => toThrow(min));
        test.doesNotThrow(() => toThrow(max));
        // falsy, falls back to portMapping.port
        // test.throws(() => toThrow(min - 1), /below the minimum threshold/);
        test.throws(() => toThrow(max + 1), /above the maximum threshold/);
        test.done();
    },
    'healthyThreshold'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const [min, max] = [2, 10];
        // WHEN
        const toThrow = (healthyThreshold) => getNode(stack).addListeners({
            healthCheck: { healthyThreshold },
        });
        // THEN
        test.doesNotThrow(() => toThrow(min));
        test.doesNotThrow(() => toThrow(max));
        test.throws(() => toThrow(min - 1), /below the minimum threshold/);
        test.throws(() => toThrow(max + 1), /above the maximum threshold/);
        test.done();
    },
    'unhealthyThreshold'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const [min, max] = [2, 10];
        // WHEN
        const toThrow = (unhealthyThreshold) => getNode(stack).addListeners({
            healthCheck: { unhealthyThreshold },
        });
        // THEN
        test.doesNotThrow(() => toThrow(min));
        test.doesNotThrow(() => toThrow(max));
        test.throws(() => toThrow(min - 1), /below the minimum threshold/);
        test.throws(() => toThrow(max + 1), /above the maximum threshold/);
        test.done();
    },
    'throws if path and Protocol.TCP'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const toThrow = (protocol) => getNode(stack).addListeners({
            healthCheck: {
                protocol,
                path: '/',
            },
        });
        // THEN
        test.doesNotThrow(() => toThrow(appmesh.Protocol.HTTP));
        test.throws(() => toThrow(appmesh.Protocol.TCP), /The path property cannot be set with Protocol.TCP/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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