# coding=utf-8
# ======================================
# File:     test_evaluate.py
# Author:   Jackie PENG
# Contact:  jackie.pengzhao@gmail.com
# Created:  2020-02-12
# Desc:
#   Unittest for all back test result
#   evaluation functions.
# ======================================
import unittest

import pandas as pd
import numpy as np

from qteasy.evaluate import eval_alpha, eval_benchmark, eval_beta, eval_fv
from qteasy.evaluate import eval_info_ratio, eval_max_drawdown, eval_sharp
from qteasy.evaluate import eval_volatility


class TestEvaluations(unittest.TestCase):
    """Test all evaluation functions in core.py"""

    # 以下手动计算结果在Excel文件中
    def setUp(self):
        """生成模拟股票价格数据"""
        self.test_data1 = pd.DataFrame([5.34892759, 5.65768696, 5.79227076, 5.56266871, 5.88189632,
                                        6.24795001, 5.92755558, 6.38748165, 6.31331899, 5.86001665,
                                        5.61048472, 5.30696736, 5.40406792, 5.03180571, 5.37886353,
                                        5.78608307, 6.26540339, 6.59348026, 6.90943801, 6.70911677,
                                        6.33015954, 6.06697417, 5.9752499, 6.45786408, 6.95273763,
                                        6.7691991, 6.70355481, 6.28048969, 6.61344541, 6.24620003,
                                        6.47409983, 6.4522311, 6.8773094, 6.99727832, 6.59262674,
                                        6.59014938, 6.63758237, 6.38331869, 6.09902105, 6.35390109,
                                        6.51993567, 6.87244592, 6.83963485, 7.08797815, 6.88003144,
                                        6.83657323, 6.97819483, 7.01600276, 7.12554256, 7.58941523,
                                        7.61014457, 7.21224091, 7.48174399, 7.66490854, 7.51371968,
                                        7.11586198, 6.97147399, 6.67453301, 6.2042138, 6.33967015,
                                        6.22187938, 5.98426993, 6.37096079, 6.55897161, 6.26422645,
                                        6.69363762, 7.12668015, 6.83232926, 7.30524081, 7.4262041,
                                        7.54031383, 7.17545919, 7.20659257, 7.44886016, 7.37094393,
                                        6.88011022, 7.08142491, 6.74992833, 6.5967097, 6.21336693,
                                        6.35565105, 6.82347596, 6.44773408, 6.84538053, 6.47966466,
                                        6.09699528, 5.63927014, 6.01081024, 6.20585303, 6.60528206,
                                        7.01594726, 7.03684251, 6.76574977, 7.08740846, 6.65336462,
                                        7.07126686, 6.80058956, 6.79241977, 6.47843472, 6.39245474],
                                       columns=['value'])

        self.test_data2 = pd.DataFrame([5.09276527, 4.83828592, 4.6000911, 4.63170487, 4.63566451,
                                        4.50546921, 4.96390044, 4.64557907, 4.25787855, 3.76585551,
                                        3.38826334, 3.76243422, 4.06365426, 3.87084726, 3.91400935,
                                        4.13438822, 4.27064542, 4.56776104, 5.03800296, 5.31070529,
                                        5.39902276, 5.21186286, 5.05683114, 4.68842046, 5.11895168,
                                        5.27151571, 5.72294993, 6.09961056, 6.26569635, 6.48806151,
                                        6.16058885, 6.2582459, 6.38934791, 6.57831057, 6.19508831,
                                        5.70155153, 5.20435735, 5.36538825, 5.40450056, 5.2227697,
                                        5.37828693, 5.53058991, 6.02996797, 5.76802181, 5.66166713,
                                        6.07988994, 5.61794367, 5.63218151, 6.10728013, 6.0324168,
                                        6.27164431, 6.27551239, 6.52329665, 7.00470007, 7.34163113,
                                        7.33699083, 7.67661334, 8.09395749, 7.68086668, 7.58341161,
                                        7.46219819, 7.58671899, 7.19348298, 7.40088323, 7.47562005,
                                        7.93342043, 8.2286081, 8.3521632, 8.43590025, 8.34977395,
                                        8.57563095, 8.81586328, 9.08738649, 9.01542031, 8.8653815,
                                        9.21763111, 9.04233017, 8.59533999, 8.47590075, 8.70857222,
                                        8.78890756, 8.92697606, 9.35743773, 9.68280866, 10.15622021,
                                        10.55908549, 10.6337894, 10.55197128, 10.65435176, 10.54611045,
                                        10.19432562, 10.48320884, 10.36176768, 10.03186854, 10.23656092,
                                        10.0062843, 10.13669686, 10.30758958, 9.87904176, 10.05126375],
                                       columns=['value'])

        self.test_data3 = pd.DataFrame([5.02851874, 5.20700348, 5.02410709, 5.49836387, 5.06834371,
                                        5.10956737, 5.15314979, 5.02256472, 5.09746382, 5.23909247,
                                        4.93410336, 4.96316186, 5.40026682, 5.7353255, 5.53438319,
                                        5.79092139, 5.67528173, 5.89840855, 5.75379463, 6.10855386,
                                        5.77322365, 5.84538021, 5.6103973, 5.7518655, 5.49729695,
                                        5.13610628, 5.30524121, 5.68093462, 5.73251319, 6.04420783,
                                        6.26929843, 6.59610234, 6.09872345, 6.25475121, 6.72927396,
                                        6.91395783, 7.00693283, 7.36217783, 7.71516676, 7.67580263,
                                        7.62477511, 7.73600568, 7.53457914, 7.46170277, 7.83658014,
                                        8.11481319, 8.03705544, 7.64948845, 7.52043731, 7.67247943,
                                        7.46511982, 7.43541798, 7.58856517, 7.9392717, 8.25406287,
                                        7.77031632, 8.03223447, 7.86799055, 7.57630999, 7.33230519,
                                        7.22378732, 6.85972264, 7.17548456, 7.5387846, 7.2392632,
                                        6.8455644, 6.59557185, 6.6496796, 6.73685623, 7.18598015,
                                        7.13619128, 6.88060157, 7.1399681, 7.30308077, 6.94942434,
                                        7.0247815, 7.37567798, 7.50080197, 7.59719284, 7.14520561,
                                        7.29913484, 7.79551341, 8.15497781, 8.40456095, 8.86516528,
                                        8.53042688, 8.94268762, 8.52048006, 8.80036284, 8.91602364,
                                        9.19953385, 8.70828953, 8.24613093, 8.18770453, 7.79548389,
                                        7.68627967, 7.23205036, 6.98302636, 7.06515819, 6.95068113],
                                       columns=['value'])

        self.test_data4 = pd.DataFrame([4.97926539, 5.44016005, 5.45122915, 5.74485615, 5.45600553,
                                        5.44858945, 5.2435413, 5.47315161, 5.58464303, 5.36179749,
                                        5.38236326, 5.29614981, 5.76523508, 5.75102892, 6.15316618,
                                        6.03852528, 6.01442228, 5.70510182, 5.22748133, 5.46762379,
                                        5.78926267, 5.8221362, 5.61236849, 5.30615725, 5.24200611,
                                        5.41042642, 5.59940342, 5.28306781, 4.99451932, 5.08799266,
                                        5.38865647, 5.58229139, 5.33492845, 5.48206276, 5.09721379,
                                        5.39190493, 5.29965087, 5.0374415, 5.50798022, 5.43107577,
                                        5.22759507, 4.991809, 5.43153084, 5.39966868, 5.59916352,
                                        5.66412137, 6.00611838, 5.63564902, 5.66723484, 5.29863863,
                                        4.91115153, 5.3749929, 5.75082334, 6.08308148, 6.58091182,
                                        6.77848803, 7.19588758, 7.64862286, 7.99818347, 7.91824794,
                                        8.30341071, 8.45984973, 7.98700002, 8.18924931, 8.60755649,
                                        8.66233396, 8.91018407, 9.0782739, 9.33515448, 8.95870245,
                                        8.98426422, 8.50340317, 8.64916085, 8.93592407, 8.63145745,
                                        8.65322862, 8.39543204, 8.37969997, 8.23394504, 8.04062872,
                                        7.91259763, 7.57252171, 7.72670114, 7.74486117, 8.06908188,
                                        7.99166889, 7.92155906, 8.39956136, 8.80181323, 8.47464091,
                                        8.06557064, 7.87145573, 8.0237959, 8.39481998, 8.68525692,
                                        8.81185461, 8.98632237, 9.0989835, 8.89787405, 8.86508591],
                                       columns=['value'])

        self.test_data5 = pd.DataFrame([4.50258923, 4.35142568, 4.07459514, 3.87791297, 3.73715985,
                                        3.98455684, 4.07587908, 4.00042472, 4.28276612, 4.01362051,
                                        4.13713565, 4.49312372, 4.48633159, 4.4641207, 4.13444605,
                                        3.79107217, 4.22941629, 4.56548511, 4.92472163, 5.27723158,
                                        5.67409193, 6.00176917, 5.88889928, 5.55256103, 5.39308314,
                                        5.2610492, 5.30738908, 5.22222408, 4.90332238, 4.57499908,
                                        4.96097146, 4.81531011, 4.39115442, 4.63200662, 5.04588813,
                                        4.67866025, 5.01705123, 4.83562258, 4.60381702, 4.66187576,
                                        4.41292828, 4.86604507, 4.42280124, 4.07517294, 4.16317319,
                                        4.10316596, 4.42913598, 4.06609666, 3.96725913, 4.15965746,
                                        4.12379564, 4.04054068, 3.84342851, 3.45902867, 3.17649855,
                                        3.09773586, 3.5502119, 3.66396995, 3.66306483, 3.29131401,
                                        2.79558533, 2.88319542, 3.03671098, 3.44645857, 3.88167161,
                                        3.57961874, 3.60180276, 3.96702102, 4.05429995, 4.40056979,
                                        4.05653231, 3.59600456, 3.60792477, 4.09989922, 3.73503663,
                                        4.01892626, 3.94597242, 3.81466605, 3.71417992, 3.93767156,
                                        4.42806557, 4.06988106, 4.03713636, 4.34408673, 4.79810156,
                                        5.18115011, 4.89798406, 5.3960077, 5.72504875, 5.61894017,
                                        5.1958197, 4.85275896, 5.17550207, 4.71548987, 4.62408567,
                                        4.55488535, 4.36532649, 4.26031979, 4.25225607, 4.58627048],
                                       columns=['value'])

        self.test_data6 = pd.DataFrame([5.08639513, 5.05761083, 4.76160923, 4.62166504, 4.62923183,
                                        4.25070173, 4.13447513, 3.90890013, 3.76687608, 3.43342482,
                                        3.67648224, 3.6274775, 3.9385404, 4.39771627, 4.03199346,
                                        3.93265288, 3.50059789, 3.3851961, 3.29743973, 3.2544872,
                                        2.93692949, 2.70893003, 2.55461976, 2.20922332, 2.29054475,
                                        2.2144714, 2.03726827, 2.39007617, 2.29866155, 2.40607111,
                                        2.40440444, 2.79374649, 2.66541922, 2.27018079, 2.08505127,
                                        2.55478864, 2.22415625, 2.58517923, 2.58802256, 2.94870959,
                                        2.69301739, 2.19991535, 2.69473146, 2.64704637, 2.62753542,
                                        2.14240825, 2.38565154, 1.94592117, 2.32243877, 2.69337246,
                                        2.51283854, 2.62484451, 2.15559054, 2.35410875, 2.31219177,
                                        1.96018265, 2.34711266, 2.58083322, 2.40290041, 2.20439791,
                                        2.31472425, 2.16228248, 2.16439749, 2.20080737, 1.73293206,
                                        1.9264407, 2.25089861, 2.69269101, 2.59296687, 2.1420998,
                                        1.67819153, 1.98419023, 2.14479494, 1.89055376, 1.96720648,
                                        1.9916694, 2.37227761, 2.14446036, 2.34573903, 1.86162546,
                                        2.1410721, 2.39204939, 2.52529064, 2.47079939, 2.9299031,
                                        3.09452923, 2.93276708, 3.21731309, 3.06248964, 2.90413406,
                                        2.67844632, 2.45621213, 2.41463398, 2.7373913, 3.14917045,
                                        3.4033949, 3.82283446, 4.02285451, 3.7619638, 4.10346795],
                                       columns=['value'])

        self.test_data7 = pd.DataFrame([4.75233583, 4.47668283, 4.55894263, 4.61765848, 4.622892,
                                        4.58941116, 4.32535872, 3.88112797, 3.47237806, 3.50898953,
                                        3.82530406, 3.6718017, 3.78918195, 4.1800752, 4.01818557,
                                        4.40822582, 4.65474654, 4.89287256, 4.40879274, 4.65505126,
                                        4.36876403, 4.58418934, 4.75687172, 4.3689799, 4.16126498,
                                        4.0203982, 3.77148242, 3.38198096, 3.07261764, 2.9014741,
                                        2.5049543, 2.756105, 2.28779058, 2.16986991, 1.8415962,
                                        1.83319008, 2.20898291, 2.00128981, 1.75747025, 1.26676663,
                                        1.40316876, 1.11126484, 1.60376367, 1.22523829, 1.58816681,
                                        1.49705679, 1.80244138, 1.55128293, 1.35339409, 1.50985759,
                                        1.0808451, 1.05892796, 1.43414812, 1.43039101, 1.73631655,
                                        1.43940867, 1.82864425, 1.71088265, 2.12015154, 2.45417128,
                                        2.84777618, 2.7925612, 2.90975121, 3.25920745, 3.13801182,
                                        3.52733677, 3.65468491, 3.69395211, 3.49862035, 3.24786017,
                                        3.64463138, 4.00331929, 3.62509565, 3.78013949, 3.4174012,
                                        3.76312271, 3.62054004, 3.67206716, 3.60596058, 3.38636199,
                                        3.42580676, 3.32921095, 3.02976759, 3.28258676, 3.45760838,
                                        3.24917528, 2.94618304, 2.86980011, 2.63191259, 2.39566759,
                                        2.53159917, 2.96273967, 3.25626185, 2.97425402, 3.16412191,
                                        3.58280763, 3.23257727, 3.62353556, 3.12806399, 2.92532313],
                                       columns=['value'])

        # 建立一个长度为 500 个数据点的测试数据, 用于测试数据点多于250个的情况下的评价过程
        self.long_data = pd.DataFrame([9.879, 9.916, 10.109, 10.214, 10.361, 10.768, 10.594, 10.288,
                                       10.082, 9.994, 10.125, 10.126, 10.384, 10.734, 10.4, 10.87,
                                       11.338, 11.061, 11.415, 11.724, 12.077, 12.196, 12.064, 12.423,
                                       12.19, 11.729, 11.677, 11.448, 11.485, 10.989, 11.242, 11.239,
                                       11.113, 11.075, 11.471, 11.745, 11.754, 11.782, 12.079, 11.97,
                                       12.178, 11.95, 12.438, 12.612, 12.804, 12.952, 12.612, 12.867,
                                       12.832, 12.832, 13.015, 13.315, 13.249, 12.904, 12.776, 12.64,
                                       12.543, 12.287, 12.225, 11.844, 11.985, 11.945, 11.542, 11.871,
                                       12.245, 12.228, 12.362, 11.899, 11.962, 12.374, 12.816, 12.649,
                                       12.252, 12.579, 12.3, 11.988, 12.177, 12.312, 12.744, 12.599,
                                       12.524, 12.82, 12.67, 12.876, 12.986, 13.271, 13.606, 13.82,
                                       14.161, 13.833, 13.831, 14.137, 13.705, 13.414, 13.037, 12.759,
                                       12.642, 12.948, 13.297, 13.483, 13.836, 14.179, 13.709, 13.655,
                                       13.198, 13.508, 13.953, 14.387, 14.043, 13.987, 13.561, 13.391,
                                       12.923, 12.555, 12.503, 12.292, 11.877, 12.34, 12.141, 11.687,
                                       11.992, 12.458, 12.131, 11.75, 11.739, 11.263, 11.762, 11.976,
                                       11.578, 11.854, 12.136, 12.422, 12.311, 12.56, 12.879, 12.861,
                                       12.973, 13.235, 13.53, 13.531, 13.137, 13.166, 13.31, 13.103,
                                       13.007, 12.643, 12.69, 12.216, 12.385, 12.046, 12.321, 11.9,
                                       11.772, 11.816, 11.871, 11.59, 11.518, 11.94, 11.803, 11.924,
                                       12.183, 12.136, 12.361, 12.406, 11.932, 11.684, 11.292, 11.388,
                                       11.874, 12.184, 12.002, 12.16, 11.741, 11.26, 11.123, 11.534,
                                       11.777, 11.407, 11.275, 11.679, 11.62, 11.218, 11.235, 11.352,
                                       11.366, 11.061, 10.661, 10.582, 10.899, 11.352, 11.792, 11.475,
                                       11.263, 11.538, 11.183, 10.936, 11.399, 11.171, 11.214, 10.89,
                                       10.728, 11.191, 11.646, 11.62, 11.195, 11.178, 11.18, 10.956,
                                       11.205, 10.87, 11.098, 10.639, 10.487, 10.507, 10.92, 10.558,
                                       10.119, 9.882, 9.573, 9.515, 9.845, 9.852, 9.495, 9.726,
                                       10.116, 10.452, 10.77, 11.225, 10.92, 10.824, 11.096, 11.542,
                                       11.06, 10.568, 10.585, 10.884, 10.401, 10.068, 9.964, 10.285,
                                       10.239, 10.036, 10.417, 10.132, 9.839, 9.556, 9.084, 9.239,
                                       9.304, 9.067, 8.587, 8.471, 8.007, 8.321, 8.55, 9.008,
                                       9.138, 9.088, 9.434, 9.156, 9.65, 9.431, 9.654, 10.079,
                                       10.411, 10.865, 10.51, 10.205, 10.519, 10.367, 10.855, 10.642,
                                       10.298, 10.622, 10.173, 9.792, 9.995, 9.904, 9.771, 9.597,
                                       9.506, 9.212, 9.688, 10.032, 9.723, 9.839, 9.918, 10.332,
                                       10.236, 9.989, 10.192, 10.685, 10.908, 11.275, 11.72, 12.158,
                                       12.045, 12.244, 12.333, 12.246, 12.552, 12.958, 13.11, 13.53,
                                       13.123, 13.138, 13.57, 13.389, 13.511, 13.759, 13.698, 13.744,
                                       13.467, 13.795, 13.665, 13.377, 13.423, 13.772, 13.295, 13.073,
                                       12.718, 12.388, 12.399, 12.185, 11.941, 11.818, 11.465, 11.811,
                                       12.163, 11.86, 11.935, 11.809, 12.145, 12.624, 12.768, 12.321,
                                       12.277, 11.889, 12.11, 12.606, 12.943, 12.945, 13.112, 13.199,
                                       13.664, 14.051, 14.189, 14.339, 14.611, 14.656, 15.112, 15.086,
                                       15.263, 15.021, 15.346, 15.572, 15.607, 15.983, 16.151, 16.215,
                                       16.096, 16.089, 16.32, 16.59, 16.657, 16.752, 16.583, 16.743,
                                       16.373, 16.662, 16.243, 16.163, 16.491, 16.958, 16.977, 17.225,
                                       17.637, 17.344, 17.684, 17.892, 18.036, 18.182, 17.803, 17.588,
                                       17.101, 17.538, 17.124, 16.787, 17.167, 17.138, 16.955, 17.148,
                                       17.135, 17.635, 17.718, 17.675, 17.622, 17.358, 17.754, 17.729,
                                       17.576, 17.772, 18.239, 18.441, 18.729, 18.319, 18.608, 18.493,
                                       18.069, 18.122, 18.314, 18.423, 18.709, 18.548, 18.384, 18.391,
                                       17.988, 17.986, 17.653, 17.249, 17.298, 17.06, 17.36, 17.108,
                                       17.348, 17.596, 17.46, 17.635, 17.275, 17.291, 16.933, 17.337,
                                       17.231, 17.146, 17.148, 16.751, 16.891, 17.038, 16.735, 16.64,
                                       16.231, 15.957, 15.977, 16.077, 16.054, 15.797, 15.67, 15.911,
                                       16.077, 16.17, 15.722, 15.258, 14.877, 15.138, 15., 14.811,
                                       14.698, 14.407, 14.583, 14.704, 15.153, 15.436, 15.634, 15.453,
                                       15.877, 15.696, 15.563, 15.927, 16.255, 16.696, 16.266, 16.698,
                                       16.365, 16.493, 16.973, 16.71, 16.327, 16.605, 16.486, 16.846,
                                       16.935, 17.21, 17.389, 17.546, 17.773, 17.641, 17.485, 17.794,
                                       17.354, 16.904, 16.675, 16.43, 16.898, 16.819, 16.921, 17.201,
                                       17.617, 17.368, 17.864, 17.484],
                                      columns=['value'])

        self.long_bench = pd.DataFrame([9.7, 10.179, 10.321, 9.855, 9.936, 10.096, 10.331, 10.662,
                                        10.59, 11.031, 11.154, 10.945, 10.625, 10.233, 10.284, 10.252,
                                        10.221, 10.352, 10.444, 10.773, 10.904, 11.104, 10.797, 10.55,
                                        10.943, 11.352, 11.641, 11.983, 11.696, 12.138, 12.365, 12.379,
                                        11.969, 12.454, 12.947, 13.119, 13.013, 12.763, 12.632, 13.034,
                                        12.681, 12.561, 12.938, 12.867, 13.202, 13.132, 13.539, 13.91,
                                        13.456, 13.692, 13.771, 13.904, 14.069, 13.728, 13.97, 14.228,
                                        13.84, 14.041, 13.963, 13.689, 13.543, 13.858, 14.118, 13.987,
                                        13.611, 14.028, 14.229, 14.41, 14.74, 15.03, 14.915, 15.207,
                                        15.354, 15.665, 15.877, 15.682, 15.625, 15.175, 15.105, 14.893,
                                        14.86, 15.097, 15.178, 15.293, 15.238, 15., 15.283, 14.994,
                                        14.907, 14.664, 14.888, 15.297, 15.313, 15.368, 14.956, 14.802,
                                        14.506, 14.257, 14.619, 15.019, 15.049, 14.625, 14.894, 14.978,
                                        15.434, 15.578, 16.038, 16.107, 16.277, 16.365, 16.204, 16.465,
                                        16.401, 16.895, 17.057, 16.621, 16.225, 16.075, 15.863, 16.292,
                                        16.551, 16.724, 16.817, 16.81, 17.192, 16.86, 16.745, 16.707,
                                        16.552, 16.133, 16.301, 16.08, 15.81, 15.75, 15.909, 16.127,
                                        16.457, 16.204, 16.329, 16.748, 16.624, 17.011, 16.548, 16.831,
                                        16.653, 16.791, 16.57, 16.778, 16.928, 16.932, 17.22, 16.876,
                                        17.301, 17.422, 17.689, 17.316, 17.547, 17.534, 17.409, 17.669,
                                        17.416, 17.859, 17.477, 17.307, 17.245, 17.352, 17.851, 17.412,
                                        17.144, 17.138, 17.085, 16.926, 16.674, 16.854, 17.064, 16.95,
                                        16.609, 16.957, 16.498, 16.552, 16.175, 15.858, 15.697, 15.781,
                                        15.583, 15.36, 15.558, 16.046, 15.968, 15.905, 16.358, 16.783,
                                        17.048, 16.762, 17.224, 17.363, 17.246, 16.79, 16.608, 16.423,
                                        15.991, 15.527, 15.147, 14.759, 14.792, 15.206, 15.148, 15.046,
                                        15.429, 14.999, 15.407, 15.124, 14.72, 14.713, 15.022, 15.092,
                                        14.982, 15.001, 14.734, 14.713, 14.841, 14.562, 15.005, 15.483,
                                        15.472, 15.277, 15.503, 15.116, 15.12, 15.442, 15.476, 15.789,
                                        15.36, 15.764, 16.218, 16.493, 16.642, 17.088, 16.816, 16.645,
                                        16.336, 16.511, 16.2, 15.994, 15.86, 15.929, 16.316, 16.416,
                                        16.746, 17.173, 17.531, 17.627, 17.407, 17.49, 17.768, 17.509,
                                        17.795, 18.147, 18.63, 18.945, 19.021, 19.518, 19.6, 19.744,
                                        19.63, 19.32, 18.933, 19.297, 19.598, 19.446, 19.236, 19.198,
                                        19.144, 19.159, 19.065, 19.032, 18.586, 18.272, 18.119, 18.3,
                                        17.894, 17.744, 17.5, 17.083, 17.092, 16.864, 16.453, 16.31,
                                        16.681, 16.342, 16.447, 16.715, 17.068, 17.067, 16.822, 16.673,
                                        16.675, 16.592, 16.686, 16.397, 15.902, 15.597, 15.357, 15.162,
                                        15.348, 15.603, 15.283, 15.257, 15.082, 14.621, 14.366, 14.039,
                                        13.957, 14.141, 13.854, 14.243, 14.414, 14.033, 13.93, 14.104,
                                        14.461, 14.249, 14.053, 14.165, 14.035, 14.408, 14.501, 14.019,
                                        14.265, 14.67, 14.797, 14.42, 14.681, 15.16, 14.715, 14.292,
                                        14.411, 14.656, 15.094, 15.366, 15.055, 15.198, 14.762, 14.294,
                                        13.854, 13.811, 13.549, 13.927, 13.897, 13.421, 13.037, 13.32,
                                        13.721, 13.511, 13.999, 13.529, 13.418, 13.881, 14.326, 14.362,
                                        13.987, 14.015, 13.599, 13.343, 13.307, 13.689, 13.851, 13.404,
                                        13.577, 13.395, 13.619, 13.195, 12.904, 12.553, 12.294, 12.649,
                                        12.425, 11.967, 12.062, 11.71, 11.645, 12.058, 12.136, 11.749,
                                        11.953, 12.401, 12.044, 11.901, 11.631, 11.396, 11.036, 11.244,
                                        10.864, 11.207, 11.135, 11.39, 11.723, 12.084, 11.8, 11.471,
                                        11.33, 11.504, 11.295, 11.3, 10.901, 10.494, 10.825, 11.054,
                                        10.866, 10.713, 10.875, 10.846, 10.947, 11.422, 11.158, 10.94,
                                        10.521, 10.36, 10.411, 10.792, 10.472, 10.305, 10.525, 10.853,
                                        10.556, 10.72, 10.54, 10.583, 10.299, 10.061, 10.004, 9.903,
                                        9.796, 9.472, 9.246, 9.54, 9.456, 9.177, 9.484, 9.557,
                                        9.493, 9.968, 9.536, 9.39, 8.922, 8.423, 8.518, 8.686,
                                        8.771, 9.098, 9.281, 8.858, 9.027, 8.553, 8.784, 8.996,
                                        9.379, 9.846, 9.855, 9.502, 9.608, 9.761, 9.409, 9.4,
                                        9.332, 9.34, 9.284, 8.844, 8.722, 8.376, 8.775, 8.293,
                                        8.144, 8.63, 8.831, 8.957, 9.18, 9.601, 9.695, 10.018,
                                        9.841, 9.743, 9.292, 8.85, 9.316, 9.288, 9.519, 9.738,
                                        9.289, 9.785, 9.804, 10.06, 10.188, 10.095, 9.739, 9.881,
                                        9.7, 9.991, 10.391, 10.002],
                                       columns=['value'])

    def test_performance_stats(self):
        """test the function performance_statistics()
        """
        # TODO: implement this test
        pass

    # noinspection PyTypeChecker
    def test_fv(self):
        print(f'test with test arr and empty DataFrame')
        self.assertAlmostEqual(eval_fv(self.test_data1), 6.39245474)
        self.assertAlmostEqual(eval_fv(self.test_data2), 10.05126375)
        self.assertAlmostEqual(eval_fv(self.test_data3), 6.95068113)
        self.assertAlmostEqual(eval_fv(self.test_data4), 8.86508591)
        self.assertAlmostEqual(eval_fv(self.test_data5), 4.58627048)
        self.assertAlmostEqual(eval_fv(self.test_data6), 4.10346795)
        self.assertAlmostEqual(eval_fv(self.test_data7), 2.92532313)
        self.assertAlmostEqual(eval_fv(pd.DataFrame()), -np.inf)
        print(f'Error testing')
        self.assertRaises(TypeError, eval_fv, 15)
        self.assertRaises(KeyError,
                          eval_fv,
                          pd.DataFrame([1, 2, 3], columns=['non_value']))

    # noinspection PyTypeChecker
    def test_max_drawdown(self):
        print(f'test with test arr and empty DataFrame')
        self.assertAlmostEqual(eval_max_drawdown(self.test_data1)[0], 0.264274308)
        self.assertEqual(eval_max_drawdown(self.test_data1)[1], 53)
        self.assertEqual(eval_max_drawdown(self.test_data1)[2], 86)
        self.assertTrue(np.isnan(eval_max_drawdown(self.test_data1)[3]))
        self.assertAlmostEqual(eval_max_drawdown(self.test_data2)[0], 0.334690849)
        self.assertEqual(eval_max_drawdown(self.test_data2)[1], 0)
        self.assertEqual(eval_max_drawdown(self.test_data2)[2], 10)
        self.assertEqual(eval_max_drawdown(self.test_data2)[3], 19)
        self.assertAlmostEqual(eval_max_drawdown(self.test_data3)[0], 0.244452899)
        self.assertEqual(eval_max_drawdown(self.test_data3)[1], 90)
        self.assertEqual(eval_max_drawdown(self.test_data3)[2], 99)
        self.assertTrue(np.isnan(eval_max_drawdown(self.test_data3)[3]))
        self.assertAlmostEqual(eval_max_drawdown(self.test_data4)[0], 0.201849684)
        self.assertEqual(eval_max_drawdown(self.test_data4)[1], 14)
        self.assertEqual(eval_max_drawdown(self.test_data4)[2], 50)
        self.assertEqual(eval_max_drawdown(self.test_data4)[3], 54)
        self.assertAlmostEqual(eval_max_drawdown(self.test_data5)[0], 0.534206456)
        self.assertEqual(eval_max_drawdown(self.test_data5)[1], 21)
        self.assertEqual(eval_max_drawdown(self.test_data5)[2], 60)
        self.assertTrue(np.isnan(eval_max_drawdown(self.test_data5)[3]))
        self.assertAlmostEqual(eval_max_drawdown(self.test_data6)[0], 0.670062689)
        self.assertEqual(eval_max_drawdown(self.test_data6)[1], 0)
        self.assertEqual(eval_max_drawdown(self.test_data6)[2], 70)
        self.assertTrue(np.isnan(eval_max_drawdown(self.test_data6)[3]))
        self.assertAlmostEqual(eval_max_drawdown(self.test_data7)[0], 0.783577449)
        self.assertEqual(eval_max_drawdown(self.test_data7)[1], 17)
        self.assertEqual(eval_max_drawdown(self.test_data7)[2], 51)
        self.assertTrue(np.isnan(eval_max_drawdown(self.test_data7)[3]))
        self.assertEqual(eval_max_drawdown(pd.DataFrame()), -np.inf)
        print(f'Error testing')
        self.assertRaises(TypeError, eval_fv, 15)
        self.assertRaises(KeyError,
                          eval_fv,
                          pd.DataFrame([1, 2, 3], columns=['non_value']))
        # test max drawdown == 0:
        # TODO: investigate: how does divide by zero change?
        self.assertAlmostEqual(eval_max_drawdown(self.test_data4 - 5)[0], 1.0770474121951792)
        self.assertEqual(eval_max_drawdown(self.test_data4 - 5)[1], 14)
        self.assertEqual(eval_max_drawdown(self.test_data4 - 5)[2], 50)

    # noinspection PyTypeChecker
    def test_info_ratio(self):
        reference = self.test_data1
        self.assertAlmostEqual(eval_info_ratio(self.test_data2, reference, 'value'), 0.075553316)
        self.assertAlmostEqual(eval_info_ratio(self.test_data3, reference, 'value'), 0.018949457)
        self.assertAlmostEqual(eval_info_ratio(self.test_data4, reference, 'value'), 0.056328143)
        self.assertAlmostEqual(eval_info_ratio(self.test_data5, reference, 'value'), -0.004270068)
        self.assertAlmostEqual(eval_info_ratio(self.test_data6, reference, 'value'), 0.009198027)
        self.assertAlmostEqual(eval_info_ratio(self.test_data7, reference, 'value'), -0.000890283)

    def test_volatility(self):
        self.assertAlmostEqual(eval_volatility(self.test_data1), 0.748646166)
        self.assertAlmostEqual(eval_volatility(self.test_data2), 0.75527442)
        self.assertAlmostEqual(eval_volatility(self.test_data3), 0.654188853)
        self.assertAlmostEqual(eval_volatility(self.test_data4), 0.688375814)
        self.assertAlmostEqual(eval_volatility(self.test_data5), 1.089989522)
        self.assertAlmostEqual(eval_volatility(self.test_data6), 1.775419308)
        self.assertAlmostEqual(eval_volatility(self.test_data7), 1.962758406)
        self.assertAlmostEqual(eval_volatility(self.test_data1, logarithm=False), 0.750993311)
        self.assertAlmostEqual(eval_volatility(self.test_data2, logarithm=False), 0.75571473)
        self.assertAlmostEqual(eval_volatility(self.test_data3, logarithm=False), 0.655331424)
        self.assertAlmostEqual(eval_volatility(self.test_data4, logarithm=False), 0.692683021)
        self.assertAlmostEqual(eval_volatility(self.test_data5, logarithm=False), 1.09602969)
        self.assertAlmostEqual(eval_volatility(self.test_data6, logarithm=False), 1.774789504)
        self.assertAlmostEqual(eval_volatility(self.test_data7, logarithm=False), 2.003329156)

        self.assertEqual(eval_volatility(pd.DataFrame()), -np.inf)
        self.assertRaises(AssertionError, eval_volatility, [1, 2, 3])

        # 测试长数据的Volatility计算
        expected_volatility = np.array([np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        np.nan, np.nan, np.nan, np.nan, np.nan,
                                        0.39955371, 0.39974258, 0.40309866, 0.40486593, 0.4055514,
                                        0.40710639, 0.40708157, 0.40609006, 0.4073625, 0.40835305,
                                        0.41155304, 0.41218193, 0.41207489, 0.41300276, 0.41308415,
                                        0.41292392, 0.41207645, 0.41238397, 0.41229291, 0.41164056,
                                        0.41316317, 0.41348842, 0.41462249, 0.41474574, 0.41652625,
                                        0.41649176, 0.41701556, 0.4166593, 0.41684221, 0.41491689,
                                        0.41435209, 0.41549087, 0.41849338, 0.41998049, 0.41959106,
                                        0.41907311, 0.41916103, 0.42120773, 0.42052391, 0.42111225,
                                        0.42124589, 0.42356445, 0.42214672, 0.42324022, 0.42476639,
                                        0.42621689, 0.42549439, 0.42533678, 0.42539414, 0.42545038,
                                        0.42593637, 0.42652095, 0.42665489, 0.42699563, 0.42798159,
                                        0.42784512, 0.42898006, 0.42868781, 0.42874188, 0.42789631,
                                        0.4277768, 0.42776827, 0.42685216, 0.42660989, 0.42563155,
                                        0.42618281, 0.42606281, 0.42505222, 0.42653242, 0.42555378,
                                        0.42500842, 0.42561939, 0.42442059, 0.42395414, 0.42384356,
                                        0.42319135, 0.42397497, 0.42488579, 0.42449729, 0.42508766,
                                        0.42509878, 0.42456616, 0.42535577, 0.42681884, 0.42688552,
                                        0.42779918, 0.42706058, 0.42792887, 0.42762114, 0.42894045,
                                        0.42977398, 0.42919859, 0.42829041, 0.42780946, 0.42825318,
                                        0.42858952, 0.42858315, 0.42805601, 0.42764751, 0.42744107,
                                        0.42775518, 0.42707283, 0.4258592, 0.42615335, 0.42526286,
                                        0.4248906, 0.42368986, 0.4232565, 0.42265079, 0.42263954,
                                        0.42153046, 0.42132051, 0.41995353, 0.41916605, 0.41914271,
                                        0.41876945, 0.41740175, 0.41583884, 0.41614026, 0.41457908,
                                        0.41472411, 0.41310876, 0.41261041, 0.41212369, 0.41211677,
                                        0.4100645, 0.40852504, 0.40860297, 0.40745338, 0.40698661,
                                        0.40644546, 0.40591375, 0.40640744, 0.40620663, 0.40656649,
                                        0.40727154, 0.40797605, 0.40807137, 0.40808913, 0.40809676,
                                        0.40711767, 0.40724628, 0.40713077, 0.40772698, 0.40765157,
                                        0.40658297, 0.4065991, 0.405011, 0.40537645, 0.40432626,
                                        0.40390177, 0.40237701, 0.40291623, 0.40301797, 0.40324145,
                                        0.40312864, 0.40328316, 0.40190955, 0.40246506, 0.40237663,
                                        0.40198407, 0.401969, 0.40185623, 0.40198313, 0.40005643,
                                        0.39940743, 0.39850438, 0.39845398, 0.39695093, 0.39697295,
                                        0.39663201, 0.39675444, 0.39538699, 0.39331959, 0.39326074,
                                        0.39193287, 0.39157266, 0.39021327, 0.39062591, 0.38917591,
                                        0.38976991, 0.38864187, 0.38872158, 0.38868096, 0.38868377,
                                        0.38842057, 0.38654784, 0.38649517, 0.38600464, 0.38408115,
                                        0.38323049, 0.38260215, 0.38207663, 0.38142669, 0.38003262,
                                        0.37969367, 0.37768092, 0.37732108, 0.37741991, 0.37617779,
                                        0.37698504, 0.37606784, 0.37499276, 0.37533731, 0.37350437,
                                        0.37375172, 0.37385382, 0.37384003, 0.37338938, 0.37212288,
                                        0.37273075, 0.370559, 0.37038506, 0.37062153, 0.36964661,
                                        0.36818564, 0.3656634, 0.36539259, 0.36428672, 0.36502487,
                                        0.3647148, 0.36551435, 0.36409919, 0.36348181, 0.36254383,
                                        0.36166601, 0.36142665, 0.35954942, 0.35846915, 0.35886759,
                                        0.35813867, 0.35642888, 0.35375231, 0.35061783, 0.35078463,
                                        0.34995508, 0.34688918, 0.34548257, 0.34633158, 0.34622833,
                                        0.34652111, 0.34622774, 0.34540951, 0.34418809, 0.34276593,
                                        0.34160916, 0.33811193, 0.33822709, 0.3391685, 0.33883381])
        test_volatility = eval_volatility(self.long_data)
        test_volatility_roll = self.long_data['volatility'].values
        self.assertAlmostEqual(test_volatility, np.nanmean(expected_volatility))
        self.assertTrue(np.allclose(expected_volatility, test_volatility_roll, equal_nan=True))

    # noinspection PyCallingNonCallable
    def test_sharp(self):
        self.assertAlmostEqual(eval_sharp(self.test_data1, 0), 0.970116743)
        self.assertAlmostEqual(eval_sharp(self.test_data2, 0), 2.654078559)
        self.assertAlmostEqual(eval_sharp(self.test_data3, 0), 1.573319618)
        self.assertAlmostEqual(eval_sharp(self.test_data4, 0), 2.449630585)
        self.assertAlmostEqual(eval_sharp(self.test_data5, 0.002), 0.578781892)
        self.assertAlmostEqual(eval_sharp(self.test_data6, 0.002), 0.570048419)
        self.assertAlmostEqual(eval_sharp(self.test_data7, 0.002), 0.347565045)

        # 测试长数据的sharp率计算
        expected_sharp = np.array([np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   -0.18950439, -0.23259702, -0.41509768, -0.34196712, -0.30898022,
                                   -0.27261473, -0.19745172, -0.14024279, 0.00288945, -0.04793207,
                                   0.05172158, -0.00366449, -0.00811567, 0.01688611, 0.17194485,
                                   0.16817806, -0.01524985, -0.02636865, -0.02937101, -0.13030031,
                                   -0.08873878, -0.15994946, -0.21137991, -0.20720521, -0.26281744,
                                   -0.26195973, -0.20128055, -0.17623172, -0.21612580, -0.15592511,
                                   -0.23457913, -0.30773631, -0.15522904, -0.06201306, -0.22066334,
                                   -0.24946104, -0.23208820, -0.13751779, -0.21974845, -0.25530392,
                                   -0.24823211, -0.08850979, -0.13609506, -0.08931705, -0.03194321,
                                   0.02869142, 0.06856389, 0.05990231, 0.08336173, 0.06678007,
                                   0.09186937, 0.11351727, 0.15256927, 0.28835942, 0.24062825,
                                   0.26824379, 0.36200869, 0.37873880, 0.41170183, 0.52816943,
                                   0.49013443, 0.50579283, 0.53864639, 0.52924168, 0.43436575,
                                   0.38763786, 0.37010600, 0.52012123, 0.42479623, 0.30569598,
                                   0.15826813, 0.12771609, 0.20406185, 0.10069257, 0.10579959,
                                   0.14153344, 0.03371507, 0.07854204, 0.06617579, 0.03427064,
                                   0.06312959, -0.01726723, 0.07709579, 0.13109071, 0.13777385,
                                   0.00445136, -0.06295816, -0.17335114, -0.18769825, -0.03774939,
                                   0.02480958, -0.02630054, 0.07502533, 0.14019807, 0.28780404,
                                   0.40331636, 0.44760993, 0.41634062, 0.39808925, 0.37277711,
                                   0.38401653, 0.32260324, 0.42880681, 0.40056919, 0.53084963,
                                   0.51071726, 0.43979661, 0.42370520, 0.50569430, 0.52451085,
                                   0.58066379, 0.60965996, 0.72892039, 0.83760065, 0.85716005,
                                   0.91181985, 0.97140360, 0.90466106, 0.88967470, 1.02561163,
                                   0.90179900, 0.79948780, 0.91316604, 1.05900004, 1.06400808,
                                   1.20357827, 1.15844283, 1.07306721, 1.20560917, 1.17735454,
                                   1.14023069, 1.10364582, 1.07303471, 0.99407802, 0.86275438,
                                   0.92738518, 0.84663503, 0.74878700, 0.74960380, 0.74527083,
                                   0.79228011, 0.81455010, 0.78606745, 0.89452664, 0.92420097,
                                   0.98940387, 0.97285085, 1.03171820, 1.05290710, 1.11975838,
                                   1.04310747, 1.15949439, 1.24976844, 1.26765194, 1.29411974,
                                   1.29888363, 1.35288816, 1.25115877, 1.22102476, 1.20311050,
                                   1.17664024, 1.20108320, 1.19392486, 1.16309306, 1.24184337,
                                   1.29679486, 1.32879912, 1.30735953, 1.15850183, 1.03516892,
                                   1.08084983, 1.01270232, 1.14759296, 1.22055657, 1.28729822,
                                   1.23404983, 1.16170851, 1.27175350, 1.24782004, 1.16297745,
                                   1.12110696, 1.27445830, 1.25458118, 1.21525262, 1.21237266,
                                   1.22263836, 1.34342989, 1.38527506, 1.26355887, 1.14721881,
                                   0.98483625, 1.01251995, 1.06548737, 1.01977026, 1.10053487,
                                   1.11748630, 0.99038426, 1.08500268, 1.10215164, 1.19781664,
                                   1.16132067, 0.97136575, 0.79944211, 0.85136924, 0.92886974,
                                   0.89862484, 0.87750353, 0.87816903, 0.85111696, 0.95651186,
                                   0.98053294, 1.14788929, 1.22157931, 1.18444097, 1.15561992,
                                   1.21908058, 1.31759397, 1.44647541, 1.59291883, 1.68048169,
                                   1.51660187, 1.58388524, 1.63449268, 1.59194395, 1.56595444,
                                   1.43533620, 1.28896458, 1.22588302, 1.28524777, 1.36908723,
                                   1.31654526, 1.25593829, 1.41240036, 1.57695971, 1.60847895,
                                   1.51052846, 1.62557901, 1.77557972, 1.72982094, 1.56287433,
                                   1.53541187, 1.55148893, 1.52791909, 1.59902474, 1.70742124,
                                   1.84548298, 2.08064663, 1.98788139, 2.04559367, 2.05991876])
        test_sharp = eval_sharp(self.long_data, 0.015)
        expected = float(np.nanmean(expected_sharp))
        self.assertAlmostEqual(expected, test_sharp)
        self.assertTrue(np.allclose(self.long_data['sharp'].values, expected_sharp, equal_nan=True))

    def test_beta(self):
        reference = self.test_data1
        self.assertAlmostEqual(eval_beta(self.test_data2, reference, 'value'), -0.017148939)
        self.assertAlmostEqual(eval_beta(self.test_data3, reference, 'value'), -0.042204233)
        self.assertAlmostEqual(eval_beta(self.test_data4, reference, 'value'), -0.15652986)
        self.assertAlmostEqual(eval_beta(self.test_data5, reference, 'value'), -0.049195532)
        self.assertAlmostEqual(eval_beta(self.test_data6, reference, 'value'), -0.026995082)
        self.assertAlmostEqual(eval_beta(self.test_data7, reference, 'value'), -0.01147809)

        self.assertRaises(TypeError, eval_beta, [1, 2, 3], reference, 'value')
        self.assertRaises(TypeError, eval_beta, self.test_data3, [1, 2, 3], 'value')
        self.assertRaises(KeyError, eval_beta, self.test_data3, reference, 'not_found_value')

        # 测试长数据的beta计算
        expected_beta = np.array([np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  np.nan, np.nan, np.nan, np.nan, np.nan,
                                  -0.04988841, -0.05127618, -0.04692104, -0.04272652, -0.04080598,
                                  -0.0493347, -0.0460858, -0.0416761, -0.03691527, -0.03724924,
                                  -0.03678865, -0.03987324, -0.03488321, -0.02567672, -0.02690303,
                                  -0.03010128, -0.02437967, -0.02571932, -0.02455681, -0.02839811,
                                  -0.03358653, -0.03396697, -0.03466321, -0.03050966, -0.0247583,
                                  -0.01629325, -0.01880895, -0.01480403, -0.01348783, -0.00544294,
                                  -0.00648176, -0.00467036, -0.01135331, -0.0156841, -0.02340763,
                                  -0.02615705, -0.02730771, -0.02906174, -0.02860664, -0.02412914,
                                  -0.02066416, -0.01744816, -0.02185133, -0.02145285, -0.02681765,
                                  -0.02827694, -0.02394581, -0.02744096, -0.02778825, -0.02703065,
                                  -0.03160023, -0.03615371, -0.03681072, -0.04265126, -0.04344738,
                                  -0.04232421, -0.04705272, -0.04533344, -0.04605934, -0.05272737,
                                  -0.05156463, -0.05134196, -0.04730733, -0.04425352, -0.03869831,
                                  -0.04159571, -0.04223998, -0.04346747, -0.04229844, -0.04740093,
                                  -0.04992507, -0.04621232, -0.04477644, -0.0486915, -0.04598224,
                                  -0.04943463, -0.05006391, -0.05362256, -0.04994067, -0.05464769,
                                  -0.05443275, -0.05513493, -0.05173594, -0.04500994, -0.04662891,
                                  -0.03903505, -0.0419592, -0.04307773, -0.03925718, -0.03711574,
                                  -0.03992631, -0.0433058, -0.04533641, -0.0461183, -0.05600344,
                                  -0.05758377, -0.05959874, -0.05605942, -0.06002859, -0.06253002,
                                  -0.06747014, -0.06427915, -0.05931947, -0.05769974, -0.04791515,
                                  -0.05175088, -0.05748039, -0.05385232, -0.05072975, -0.05052637,
                                  -0.05125567, -0.05005785, -0.05325104, -0.04977727, -0.04947867,
                                  -0.05148544, -0.05739156, -0.05742069, -0.06047279, -0.0558414,
                                  -0.06086126, -0.06265151, -0.06411129, -0.06828052, -0.06781762,
                                  -0.07083409, -0.07211207, -0.06799162, -0.06913295, -0.06775162,
                                  -0.0696265, -0.06678248, -0.06867502, -0.06581961, -0.07055823,
                                  -0.06448184, -0.06097973, -0.05795587, -0.0618383, -0.06130145,
                                  -0.06050652, -0.05936661, -0.05749424, -0.0499, -0.05050495,
                                  -0.04962687, -0.05033439, -0.05070116, -0.05422009, -0.05369759,
                                  -0.05548943, -0.05907353, -0.05933035, -0.05927918, -0.06227663,
                                  -0.06011455, -0.05650432, -0.05828134, -0.05620949, -0.05715323,
                                  -0.05482478, -0.05387113, -0.05095559, -0.05377999, -0.05334267,
                                  -0.05220438, -0.04001521, -0.03892434, -0.03660782, -0.04282708,
                                  -0.04324623, -0.04127048, -0.04227559, -0.04275226, -0.04347049,
                                  -0.04125853, -0.03806295, -0.0330632, -0.03155531, -0.03277152,
                                  -0.03304518, -0.03878731, -0.03830672, -0.03727434, -0.0370571,
                                  -0.04509224, -0.04207632, -0.04116198, -0.04545179, -0.04584584,
                                  -0.05287341, -0.05417433, -0.05175836, -0.05005509, -0.04268674,
                                  -0.03442321, -0.03457309, -0.03613426, -0.03524391, -0.03629479,
                                  -0.04361312, -0.02626705, -0.02406115, -0.03046384, -0.03181044,
                                  -0.03375164, -0.03661673, -0.04520779, -0.04926951, -0.05726738,
                                  -0.0584486, -0.06220608, -0.06800563, -0.06797431, -0.07562211,
                                  -0.07481996, -0.07731229, -0.08413381, -0.09031826, -0.09691925,
                                  -0.11018071, -0.11952675, -0.10826026, -0.11173895, -0.10756359,
                                  -0.10775916, -0.11664559, -0.10505051, -0.10606547, -0.09855355,
                                  -0.10004159, -0.10857084, -0.12209301, -0.11605758, -0.11105113,
                                  -0.1155195, -0.11569505, -0.10513348, -0.09611072, -0.10719791,
                                  -0.10843965, -0.11025856, -0.10247839, -0.10554044, -0.10927647,
                                  -0.10645088, -0.09982498, -0.10542734, -0.09631372, -0.08229695])
        test_beta_mean = eval_beta(self.long_data, self.long_bench, 'value')
        test_beta_roll = self.long_data['beta'].values
        expected = float(np.nanmean(expected_beta))
        self.assertAlmostEqual(test_beta_mean, expected)
        self.assertTrue(np.allclose(test_beta_roll, expected_beta, equal_nan=True))

    def test_alpha(self):
        reference = self.test_data1
        self.assertAlmostEqual(eval_alpha(self.test_data2, 5, reference, 'value', 0.5), 11.63072977)
        self.assertAlmostEqual(eval_alpha(self.test_data3, 5, reference, 'value', 0.5), 1.886590071)
        self.assertAlmostEqual(eval_alpha(self.test_data4, 5, reference, 'value', 0.5), 6.827021872)
        self.assertAlmostEqual(eval_alpha(self.test_data5, 5, reference, 'value', 0.92), -1.192265168)
        self.assertAlmostEqual(eval_alpha(self.test_data6, 5, reference, 'value', 0.92), -1.437142359)
        self.assertAlmostEqual(eval_alpha(self.test_data7, 5, reference, 'value', 0.92), -1.781311545)

        # 测试长数据的alpha计算
        expected_alpha = np.array([np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   np.nan, np.nan, np.nan, np.nan, np.nan,
                                   -0.09418119, -0.11188463, -0.17938358, -0.15588172, -0.1462678,
                                   -0.13089586, -0.10780125, -0.09102891, -0.03987585, -0.06075686,
                                   -0.02459503, -0.04104284, -0.0444565, -0.04074585, 0.02191275,
                                   0.02255955, -0.05583375, -0.05875539, -0.06055551, -0.09648245,
                                   -0.07913737, -0.10627829, -0.12320965, -0.12368335, -0.1506743,
                                   -0.15768033, -0.13638829, -0.13065298, -0.14537834, -0.127428,
                                   -0.15504529, -0.18184636, -0.12652146, -0.09190138, -0.14847221,
                                   -0.15840648, -0.1525789, -0.11859418, -0.14700954, -0.16295761,
                                   -0.16051645, -0.10364859, -0.11961134, -0.10258267, -0.08090148,
                                   -0.05727746, -0.0429945, -0.04672356, -0.03581408, -0.0439215,
                                   -0.03429495, -0.0260362, -0.01075022, 0.04931808, 0.02779388,
                                   0.03984083, 0.08311951, 0.08995566, 0.10522428, 0.16159058,
                                   0.14238174, 0.14759783, 0.16257712, 0.158908, 0.11302115,
                                   0.0909566, 0.08272888, 0.15261884, 0.10546376, 0.04990313,
                                   -0.01284111, -0.02720704, 0.00454725, -0.03965491, -0.03818265,
                                   -0.02186992, -0.06574751, -0.04846454, -0.05204211, -0.06316498,
                                   -0.05095099, -0.08502656, -0.04681162, -0.02362027, -0.02205091,
                                   -0.07706374, -0.10371841, -0.14434688, -0.14797935, -0.09055402,
                                   -0.06739549, -0.08824959, -0.04855888, -0.02291244, 0.04027138,
                                   0.09370505, 0.11472939, 0.10243593, 0.0921445, 0.07662648,
                                   0.07946651, 0.05450718, 0.10497677, 0.09068334, 0.15462924,
                                   0.14231034, 0.10544952, 0.09980256, 0.14035223, 0.14942974,
                                   0.17624102, 0.19035477, 0.2500807, 0.30724652, 0.31768915,
                                   0.35007521, 0.38412975, 0.34356521, 0.33614463, 0.41206165,
                                   0.33999177, 0.28045963, 0.34076789, 0.42220356, 0.42314636,
                                   0.50790423, 0.47713348, 0.42520169, 0.50488411, 0.48705211,
                                   0.46252601, 0.44325578, 0.42640573, 0.37986783, 0.30652822,
                                   0.34503393, 0.2999069, 0.24928617, 0.24730218, 0.24326897,
                                   0.26657905, 0.27861168, 0.26392824, 0.32552649, 0.34177792,
                                   0.37837011, 0.37025267, 0.4030612, 0.41339361, 0.45076809,
                                   0.40383354, 0.47093422, 0.52505036, 0.53614256, 0.5500943,
                                   0.55319293, 0.59021451, 0.52358459, 0.50605947, 0.49359168,
                                   0.47895956, 0.49320243, 0.4908336, 0.47310767, 0.51821564,
                                   0.55105932, 0.57291504, 0.5599809, 0.46868842, 0.39620087,
                                   0.42086934, 0.38317217, 0.45934108, 0.50048866, 0.53941991,
                                   0.50676751, 0.46500915, 0.52993663, 0.51668366, 0.46405428,
                                   0.44100603, 0.52726147, 0.51565458, 0.49186248, 0.49001081,
                                   0.49367648, 0.56422294, 0.58882785, 0.51334664, 0.44386256,
                                   0.35056709, 0.36490029, 0.39205071, 0.3677061, 0.41134736,
                                   0.42315067, 0.35356394, 0.40324562, 0.41340007, 0.46503322,
                                   0.44355762, 0.34854314, 0.26412842, 0.28633753, 0.32335224,
                                   0.30761141, 0.29709569, 0.29570487, 0.28000063, 0.32802547,
                                   0.33967726, 0.42511212, 0.46252357, 0.44244974, 0.42152907,
                                   0.45436727, 0.50482359, 0.57339198, 0.6573356, 0.70912003,
                                   0.60328917, 0.6395092, 0.67015805, 0.64241557, 0.62779142,
                                   0.55028063, 0.46448736, 0.43709245, 0.46777983, 0.51789439,
                                   0.48594916, 0.4456216, 0.52008189, 0.60548684, 0.62792473,
                                   0.56645031, 0.62766439, 0.71829315, 0.69481356, 0.59550329,
                                   0.58133754, 0.59014148, 0.58026655, 0.61719273, 0.67373203,
                                   0.75573056, 0.89501633, 0.8347253, 0.87964685, 0.89015835])
        test_alpha_mean = eval_alpha(self.long_data, 100, self.long_bench, 'value')
        test_alpha_roll = self.long_data['alpha'].values
        expected = float(np.nanmean(expected_alpha))
        self.assertAlmostEqual(test_alpha_mean, expected)
        self.assertTrue(np.allclose(test_alpha_roll, expected_alpha, equal_nan=True))

    def test_calmar(self):
        """test evaluate function eval_calmar()"""
        # TODO: implement this test
        pass

    def test_benchmark(self):
        reference = self.test_data1
        tr, yr = eval_benchmark(self.test_data2, reference, 'value')
        self.assertAlmostEqual(tr, 0.19509091)
        self.assertAlmostEqual(yr, 0.929154957)
        tr, yr = eval_benchmark(self.test_data3, reference, 'value')
        self.assertAlmostEqual(tr, 0.19509091)
        self.assertAlmostEqual(yr, 0.929154957)
        tr, yr = eval_benchmark(self.test_data4, reference, 'value')
        self.assertAlmostEqual(tr, 0.19509091)
        self.assertAlmostEqual(yr, 0.929154957)
        tr, yr = eval_benchmark(self.test_data5, reference, 'value')
        self.assertAlmostEqual(tr, 0.19509091)
        self.assertAlmostEqual(yr, 0.929154957)
        tr, yr = eval_benchmark(self.test_data6, reference, 'value')
        self.assertAlmostEqual(tr, 0.19509091)
        self.assertAlmostEqual(yr, 0.929154957)
        tr, yr = eval_benchmark(self.test_data7, reference, 'value')
        self.assertAlmostEqual(tr, 0.19509091)
        self.assertAlmostEqual(yr, 0.929154957)

    def test_evaluate(self):
        """ 测试回测结果评价函数"""
        # TODO: implement this test
        pass


if __name__ == '__main__':
    unittest.main()