"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_sdk_1 = require("aws-sdk"); // eslint-disable-line import/no-extraneous-dependencies
const got_1 = require("got");
const secretsmanager = new aws_sdk_1.SecretsManager({ apiVersion: '2017-10-17' });
const slackClient = got_1.default.extend({
    prefixUrl: 'https://slack.com/api',
});
async function handler(event) {
    var _a, _b;
    console.log('Event: %j', event);
    const data = await secretsmanager.getSecretValue({
        SecretId: event.ResourceProperties.configurationTokenSecretArn,
    }).promise();
    if (!data.SecretString) {
        throw new Error('No secret string found in configuration token secret');
    }
    const secret = JSON.parse(data.SecretString);
    let accessToken = secret.accessToken;
    if (!accessToken || isExpired((_a = secret.exp) !== null && _a !== void 0 ? _a : 0)) {
        if (!secret.refreshToken) {
            throw new Error('No refresh token found in configuration token secret');
        }
        console.log('Refreshing access token');
        const rotate = await slackClient.get('tooling.tokens.rotate', {
            searchParams: { refresh_token: secret.refreshToken },
        }).json();
        if (!rotate.ok) {
            throw new Error(`Failed to refresh access token: ${rotate.error}`);
        }
        console.log('Access token refreshed');
        accessToken = rotate.token;
        console.log('Saving access token');
        const putSecretValue = await secretsmanager.putSecretValue({
            SecretId: event.ResourceProperties.configurationTokenSecretArn,
            SecretString: JSON.stringify({
                accessToken,
                refreshToken: rotate.refresh_token,
                exp: rotate.exp,
            }),
        }).promise();
        console.log(`Successfully saved access token in secret ${putSecretValue.ARN}`);
    }
    const operation = event.RequestType.toLowerCase();
    const request = getManifestRequest(event);
    console.log(`Calling ${operation} manifest API: %j`, request);
    const response = await slackClient.post(`apps.manifest.${operation}`, {
        headers: { Authorization: `Bearer ${accessToken}` },
        json: request,
    }).json();
    if (!response.ok) {
        const errors = response.errors
            ? response.errors.map((err) => `${err.message} at ${err.pointer}`).join('\n')
            : '';
        throw new Error(`Failed to ${operation} manifest: ${response.error}.${errors ? `\n${errors}}` : ''}`);
    }
    console.log(`Successfully ${operation}d Slack app ${(_b = event.PhysicalResourceId) !== null && _b !== void 0 ? _b : response.app_id}`);
    if (event.RequestType === 'Create' && response.credentials) {
        console.log('Saving app credentials');
        const putSecretValue = await secretsmanager.putSecretValue({
            SecretId: event.ResourceProperties.credentialsSecretArn,
            SecretString: JSON.stringify({
                appId: response.app_id,
                clientId: response.credentials.client_id,
                clientSecret: response.credentials.client_secret,
                verificationToken: response.credentials.verification_token,
                signingSecret: response.credentials.signing_secret,
            }),
        }).promise();
        console.log(`Successfully saved app credentials in secret ${putSecretValue.ARN}`);
    }
    return {
        PhysicalResourceId: response.app_id,
        Data: {
            appId: response.app_id,
        },
    };
}
exports.handler = handler;
function isExpired(iat) {
    return (iat - (Date.now() / 1000)) < 0;
}
function getManifestRequest(event) {
    switch (event.RequestType) {
        case 'Create':
            return {
                manifest: event.ResourceProperties.manifest,
            };
        case 'Update':
            return {
                app_id: event.PhysicalResourceId,
                manifest: event.ResourceProperties.manifest,
            };
        case 'Delete':
            return {
                app_id: event.PhysicalResourceId,
            };
    }
}
//# sourceMappingURL=data:application/json;base64,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