import os
import sys
from pathlib import Path

import click

from .root import Root
from .utils import argument, group

CFG_FILE = {"bash": Path("~/.bashrc"), "zsh": Path("~/.zshrc")}
SOURCE_CMD = {"bash": "bash_source", "zsh": "zsh_source"}

ACTIVATION_TEMPLATE = 'eval "$(_NEURO_COMPLETE={cmd} {exe})"'


@group()
def completion() -> None:
    """
    Output shell completion code.
    """


@completion.command()
@argument("shell", type=click.Choice(["bash", "zsh"]))
async def generate(root: Root, shell: str) -> None:
    """
    Provide an instruction for shell completion generation.
    """
    root.print(f"Push the following line into your {CFG_FILE[shell]}")
    root.print(
        ACTIVATION_TEMPLATE.format(cmd=SOURCE_CMD[shell], exe=sys.argv[0]),
        crop=False,
        overflow="ignore",
    )


@completion.command()
@argument("shell", type=click.Choice(["bash", "zsh"]))
async def patch(root: Root, shell: str) -> None:
    """
    Automatically patch shell configuration profile to enable completion
    """
    GENERATED_START = (
        b"# Start of generated by neuro. Please do not edit this comment.\n"
    )
    GENERATED_END = b"\n# End of generated by neuro. Please do not edit this comment.\n"

    profile_file = CFG_FILE[shell].expanduser()

    code = (
        GENERATED_START
        + os.fsencode(
            ACTIVATION_TEMPLATE.format(cmd=SOURCE_CMD[shell], exe=sys.argv[0])
        )
        + GENERATED_END
    )

    try:
        with profile_file.open("rb+") as profile:
            content = profile.read()
    except FileNotFoundError:
        content = b""

    start = content.find(GENERATED_START)
    if start != -1:
        end = content.find(GENERATED_END)
        if end == -1:
            raise click.ClickException(
                f"Malformed guarding comments. Please modify {profile_file} manually"
            )
        content = content[:start] + code + content[end + len(GENERATED_END) :]
    else:
        if content != b"":
            content += b"\n" + code
        else:
            content = code

    with profile_file.open("wb+") as profile:
        profile.write(content)
    root.print(f"Added completion configuration into '{profile_file}'")
