"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const AWS = require("aws-sdk-mock");
const sinon = require("sinon");
const replica_handler_1 = require("../lib/replica-handler");
AWS.setSDK(require.resolve('aws-sdk'));
const createEvent = {
    RequestType: 'Create',
    ResourceProperties: {
        TableName: 'my-table',
        Region: 'eu-west-2',
        ServiceToken: 'token'
    },
    ServiceToken: 'token',
    ResponseURL: 'url',
    LogicalResourceId: 'logical-id',
    RequestId: 'request-id',
    StackId: 'stack-id',
    ResourceType: 'resource-type'
};
beforeEach(done => {
    process.env.USE_NORMAL_SDK = 'true';
    done();
});
afterEach(done => {
    delete process.env.USE_NORMAL_SDK;
    AWS.restore();
    done();
});
test('on event', async () => {
    const updateTableMock = sinon.fake.resolves({});
    AWS.mock('DynamoDB', 'updateTable', updateTableMock);
    const data = await replica_handler_1.onEventHandler(createEvent);
    sinon.assert.calledWith(updateTableMock, {
        TableName: 'my-table',
        ReplicaUpdates: [
            {
                Create: {
                    RegionName: 'eu-west-2'
                }
            },
        ]
    });
    expect(data).toEqual({
        PhysicalResourceId: 'eu-west-2'
    });
});
test('on event does not call updateTable for Update requests', async () => {
    const updateTableMock = sinon.fake.resolves({});
    AWS.mock('DynamoDB', 'updateTable', updateTableMock);
    const data = await replica_handler_1.onEventHandler({
        ...createEvent,
        RequestType: 'Update',
    });
    sinon.assert.notCalled(updateTableMock);
    expect(data).toEqual({
        PhysicalResourceId: 'eu-west-2'
    });
});
test('is complete for create returns false without replicas', async () => {
    const describeTableMock = sinon.fake.resolves({
        Table: {}
    });
    AWS.mock('DynamoDB', 'describeTable', describeTableMock);
    const data = await replica_handler_1.isCompleteHandler(createEvent);
    expect(data).toEqual({ IsComplete: false });
});
test('is complete for create returns false when replica is not active', async () => {
    const describeTableMock = sinon.fake.resolves({
        Table: {
            Replicas: [
                {
                    RegionName: 'eu-west-2',
                    ReplicaStatus: 'CREATING'
                }
            ]
        }
    });
    AWS.mock('DynamoDB', 'describeTable', describeTableMock);
    const data = await replica_handler_1.isCompleteHandler(createEvent);
    expect(data).toEqual({ IsComplete: false });
});
test('is complete for create returns false when table is not active', async () => {
    const describeTableMock = sinon.fake.resolves({
        Table: {
            Replicas: [
                {
                    RegionName: 'eu-west-2',
                    ReplicaStatus: 'ACTIVE'
                }
            ],
            TableStatus: 'UPDATING',
        }
    });
    AWS.mock('DynamoDB', 'describeTable', describeTableMock);
    const data = await replica_handler_1.isCompleteHandler(createEvent);
    expect(data).toEqual({ IsComplete: false });
});
test('is complete for create returns true when replica is active', async () => {
    const describeTableMock = sinon.fake.resolves({
        Table: {
            Replicas: [
                {
                    RegionName: 'eu-west-2',
                    ReplicaStatus: 'ACTIVE'
                }
            ],
            TableStatus: 'ACTIVE',
        }
    });
    AWS.mock('DynamoDB', 'describeTable', describeTableMock);
    const data = await replica_handler_1.isCompleteHandler(createEvent);
    expect(data).toEqual({ IsComplete: true });
});
test('is complete for delete returns true when replica is gone', async () => {
    const describeTableMock = sinon.fake.resolves({
        Table: {
            Replicas: [
                {
                    RegionName: 'eu-west-1',
                    ReplicaStatus: 'ACTIVE'
                }
            ],
            TableStatus: 'ACTIVE',
        }
    });
    AWS.mock('DynamoDB', 'describeTable', describeTableMock);
    const data = await replica_handler_1.isCompleteHandler({
        ...createEvent,
        RequestType: 'Delete'
    });
    expect(data).toEqual({ IsComplete: true });
});
//# sourceMappingURL=data:application/json;base64,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