"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_sdk_1 = require("aws-sdk"); // eslint-disable-line import/no-extraneous-dependencies
const child_process_1 = require("child_process");
let latestSdkInstalled = false;
function installLatestSdk() {
    console.log('Installing latest AWS SDK v2');
    // Both HOME and --prefix are needed here because /tmp is the only writable location
    child_process_1.execSync('HOME=/tmp npm install aws-sdk@2 --production --no-package-lock --no-save --prefix /tmp');
    latestSdkInstalled = true;
}
async function onEventHandler(event) {
    console.log('Event: %j', event);
    /**
     * Process only Create and Delete requests. We shouldn't receive any
     * update request and in case we do there is nothing to update.
     */
    if (event.RequestType === 'Create' || event.RequestType === 'Delete') {
        // ReplicaUpdates has been introduced in v2.577.0
        // Node.js 12.x currently uses v2.536.0
        if (!latestSdkInstalled && !process.env.USE_NORMAL_SDK) {
            installLatestSdk();
        }
        let AWS;
        if (process.env.USE_NORMAL_SDK) { // For tests only
            AWS = require('aws-sdk'); // eslint-disable-line @typescript-eslint/no-require-imports, import/no-extraneous-dependencies
        }
        else {
            AWS = require('/tmp/node_modules/aws-sdk'); // eslint-disable-line @typescript-eslint/no-require-imports
        }
        const dynamodb = new AWS.DynamoDB();
        const data = await dynamodb.updateTable({
            TableName: event.ResourceProperties.TableName,
            ReplicaUpdates: [
                {
                    [event.RequestType]: {
                        RegionName: event.ResourceProperties.Region
                    }
                },
            ]
        }).promise();
        console.log('Update table: %j', data);
    }
    return { PhysicalResourceId: event.ResourceProperties.Region };
}
exports.onEventHandler = onEventHandler;
async function isCompleteHandler(event) {
    var _a, _b, _c;
    console.log('Event: %j', event);
    const dynamodb = new aws_sdk_1.DynamoDB();
    const data = await dynamodb.describeTable({
        TableName: event.ResourceProperties.TableName
    }).promise();
    console.log('Describe table: %j', data);
    const tableActive = !!(((_a = data.Table) === null || _a === void 0 ? void 0 : _a.TableStatus) === 'ACTIVE');
    const replicas = (_c = (_b = data.Table) === null || _b === void 0 ? void 0 : _b.Replicas) !== null && _c !== void 0 ? _c : [];
    const regionReplica = replicas.find(r => r.RegionName === event.ResourceProperties.Region);
    const replicaActive = !!((regionReplica === null || regionReplica === void 0 ? void 0 : regionReplica.ReplicaStatus) === 'ACTIVE');
    switch (event.RequestType) {
        case 'Create':
        case 'Update':
            // Complete when replica is reported as ACTIVE
            return { IsComplete: tableActive && replicaActive };
        case 'Delete':
            // Complete when replica is gone
            return { IsComplete: tableActive && regionReplica === undefined };
    }
}
exports.isCompleteHandler = isCompleteHandler;
//# sourceMappingURL=data:application/json;base64,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