# This file is part of the Pattern and Anomaly Detection Library (openclean_pattern).
#
# Copyright (C) 2021 New York University.
#
# openclean_pattern is released under the Revised BSD License. See file LICENSE for
# full license details.

"""classes of base OpencleanPattern objects"""

from abc import abstractmethod, ABCMeta
from collections import defaultdict, Counter
from typing import List, Optional, Union, Iterable, Dict

import numpy as np

from openclean.function.token.base import Tokenizer
from openclean_pattern.datatypes.base import SupportedDataTypes
from openclean_pattern.function.value import IsMatch
from openclean.function.token.base import Token
from openclean_pattern.utils.utils import StringComparator

from openclean.profiling.pattern.base import Pattern


class OpencleanPattern(Pattern, metaclass=ABCMeta):
    """Contains PatternElements used inside the Patterns class to store a single row / column pattern"""

    def __init__(self, container: Iterable):
        """Initialize the OpencleanPattern class

        Parameters
        ----------
        container: Iterable
            the base class to store pattern elements in
        """
        self.container = container
        self.freq = 0
        self.idx = set()

    @abstractmethod
    def compare(self, value: str, tokenizer: Tokenizer) -> bool:
        """Evaluat the given value against the pattern.

        Returns True if the value matches the pattern and Flase otherwise.

        Parameters
        ----------
        value : str
            The value to match with the pattern
        tokenizer: openclean.function.token.base.Tokenizer
            Tokenizer that was used when generating the pattern.
        """
        raise NotImplementedError()

    def compile(self, negate: Optional[bool] = False, tokenizer: Optional[Tokenizer] = None) -> IsMatch:
        """Get an instance of a value function that is predicate which can be
        used to test whether an given value is accepted by the pattern or not.

        Parameters
        ----------
        negate: bool, default=False
            If the negate flag is True, the returned predicate should return
            True for values that are not accepeted by the pattern and False for
            those that are accepeted.
        tokenizer: Tokenizer (optional)
            The Tokenizer used to generate the original pattern. required
            to recreate the tokenization and type detection on the new values

        Returns
        -------
        openclean.function.value.base.ValueFunction
        """
        return IsMatch(func=self.compare, negated=negate, tokenizer=tokenizer)

    def metadata(self) -> Dict:
        """Return a dictionary containing optional metadata associated with the
        pattern. This can for example be statistics generated by the pattern
        discovery algorithm providing additional information or evidence for
        the confidence that the algorithm has in the pattern or the relevance
        of the pattern.

        The structure of the dictionary is implementation-dependent. If no
        additional metadata was generated an empty dictionary should be
        returned.

        Returns
        -------
        dict
        """
        this = dict()
        for v, i in vars(self).items():
            if v != 'container':
                this[v] = i
        return this

    def pattern(self) -> str:
        """Get a string representation of the pattern for display purposes.

        Returns
        -------
        string
        """
        return str(self.container)

    def to_dict(self) -> Dict:
        """Returns a dictionary serialization of the pattern. This is an
        external representation that is used when the results of a pattern
        finder are included in the result generated by a data profiler.

        Returns
        -------
        dict
        """
        this = dict()
        for v, i in vars(self).items():
            this[v] = i
        return this

    @abstractmethod
    def update(self, tokens: Iterable[Token]) -> None:
        """update using the tokens, the list (of OpencleanPattern Elements

        Parameters
        ----------
         tokens: Token
            the tokens to use create the OpencleanPattern
        """
        raise NotImplementedError()

    def __iter__(self):
        return iter(self.container)

    def __repr__(self):
        return str(self.container)

    def __len__(self):
        return len(self.container)

    def __getitem__(self, item):
        return self.container[item]


class SingularRowPattern(OpencleanPattern):
    """Class to create / store a singular pattern created from a row / condensed into a row form"""

    def __init__(self):
        """initializes the class"""
        container = list()
        super(SingularRowPattern, self).__init__(container)

    def __eq__(self, other):
        for s, o in zip(self, other):
            if s.element_type != o.element_type:
                return False
        return len(self) == len(other)

    def __hash__(self):
        return hash(str(self))

    def append(self, value) -> None:
        """append to the container without referencing it explicitly

        Parameters
        ----------
        value: Token
            The Token to append
        """
        self.container.append(value)

    def compare(self, value: Union[str, List[Token]], tokenizer: Optional[Tokenizer] = None) -> bool:
        """Parses through a single value's tokens and returns False if any Token
        value.regex_type doesn't match the respective pattern.element_type.

        Parameters
        ----------
        value : str or list of openclean.function.token.base.Token
            The value to match with the pattern
        tokenizer: openclean.function.token.base.Tokenizer, default=None
            Tokenizer that was used when generating the pattern.

        Returns
        -------
            bool

        Raises
        ------
        ValueError
        """
        if isinstance(value, str):
            value = [value]

        if isinstance(value, list) or isinstance(value, tuple):
            if len(value) == 1 and isinstance(value[0], str) and not isinstance(value[0], Token):
                value = tokenizer.encode(value)[0]
            else:
                for v in value:
                    if not isinstance(v, Token):
                        raise ValueError("Invalid Token: {}".format(v))

        if len(self) != len(value):
            return False

        for p, v in zip(self, value):
            if not isinstance(p, PatternElement):
                raise ValueError("Invalid PatternElement")

            if p.element_type != v.regex_type:
                if p.element_type == SupportedDataTypes.ALPHANUM and \
                        (v.regex_type in [SupportedDataTypes.ALPHA, SupportedDataTypes.DIGIT]):
                    continue
                elif p.element_type == SupportedDataTypes.GAP and v.regex_type == SupportedDataTypes.GAP:
                    continue
                return False

            if p.len_min <= v.size <= p.len_max:
                continue
            else:
                return False

        return True

    def update(self, tokens: Iterable[Token]) -> None:
        """update using the tokens, the list (of OpencleanPattern Elements

        Parameters
        ----------
         tokens: tuple(Token)
            the tokens to use create the OpencleanPattern
        """
        for r, e in zip(tokens, self):
            e.update(r)

        self.idx.add(r.rowidx)
        self.freq += 1


class SingularColumnPattern(OpencleanPattern):
    """Class to create / store a singular patterns created from a row"""

    def __init__(self, size_coverage: float = 1):
        """initializes the class

        Parameters
        ----------
        size_coverage: float
            size coverage passed used by the size monitor as threshold
        """
        container = dict()
        super(SingularColumnPattern, self).__init__(container)
        self.column_min = np.inf
        self.column_max = -np.inf
        self.column_freq = 0
        self.size_coverage = size_coverage

    def update(self, tokens: Token):
        """update the column pattern using the tokens, the list (of OpencleanPattern Elements

        Parameters
        ----------
         tokens: Token
            the tokens to use create the OpencleanPattern
        """
        if isinstance(tokens, Token):
            tokens = [tokens]
        for token in tokens:
            if not isinstance(token, Token):
                raise TypeError("expected: openclean.function.token.base.Token, got: {}".format(token.__class__))

            if token.regex_type not in self.container:
                self[token.regex_type] = PatternElementSizeMonitor(threshold=self.size_coverage).update(token)
            else:
                self[token.regex_type].update(token)

        # Profile of all the tokens that went into creating this pattern,
        # not just the ones that created the final pattern element
        self.idx.add(token.rowidx)
        self.column_freq += 1
        self.column_min = min(self.column_min, token.size)
        self.column_max = max(self.column_max, token.size)

    def top(self):
        """returns the top PatternElement in the column

        Returns
        -------
            PatternElement
        """
        max = -np.inf
        top = None
        for i, c in self.items():
            if c.freq > max:
                top = c
                max = top.freq
        # loads the size anomalies removed Pattern Element from the PatternElementSizeMonitor
        return top.load()

    def compare(self, value: Union[str, List[Token]], tokenizer: Optional[Tokenizer] = None) -> bool:
        """Evaluates the given value against the pattern and returns a boolean

        Parameters
        ----------
        value: list[Tokens]
            value / tokens to compare with the pattern
        tokenizer: openclean.function.token.base.Tokenizer
            Tokenizer that was used when generating the pattern.
        """
        raise NotImplementedError()

    def __setitem__(self, key, value):
        self.container[key] = value

    def items(self) -> Iterable:
        return self.container.items()


# -- Pattern Dicts ------------------------------------------------------------

class Patterns(defaultdict, metaclass=ABCMeta):
    """Class to create patterns from tokens"""

    def __init__(self, size_coverage: float = 1):
        super(Patterns, self).__init__()
        self.global_freq = 0
        self.anoms = set()
        self.size_coverage = size_coverage

    @staticmethod
    def keygen(row: Iterable) -> str:
        """converts the row types into the key for the Patterns class"""
        key = ''
        for r in row:
            key += ' ' + str(r)
        return key.strip()

    @abstractmethod
    def insert(self, row):
        """insert the row into the respective method"""
        raise NotImplementedError()

    @abstractmethod
    def condense(self):
        """converts the patterns class to the default representation incase they were compiled using a different method

        Returns
        -------
            RowPatterns
        """
        raise NotImplementedError()

    def stats(self) -> Dict:
        """calculates shares of each OpencleanPattern. Ensure this is computed on the Patterns.condensed() object

        Returns
        -------
            dict
        """
        shares = defaultdict(float)
        for p in self:
            shares[p] = self[p].freq / self.global_freq
        return shares

    def top(self, n: int = 1, pattern: bool = False) -> Union[str, Pattern]:
        """gets the element type with the n ranked share. Ensure this is computed on the Patterns.condensed() object

        Parameters
        ----------
        n: int
            ranking
        pattern: bool
            returns either the string representation or the Pattern Object
        Returns
        -------
            str / Pattern
        """
        if n < 1:
            raise ValueError("rank should be greater than zero")

        n -= 1  # change rank to index

        shares = self.stats()
        sorted_shares = sorted(shares.items(), key=lambda kv: kv[1], reverse=True)
        return sorted_shares[n][0] if not pattern else self[sorted_shares[n][0]]

    def anomalies(self, n: int = 1):
        """gets the indices of rows that didnt match the nth pattern. Ensure this is computed on the
         Patterns.condensed() object

        Parameters
        ----------
        n: int
            rank of pattern to pick

        Returns
        -------
            list
        """
        top = self.top(n=n)
        anomalies = list()
        for p in self:
            if p != top:
                [anomalies.append(id) for id in self[p].idx]
            elif p == top:
                pattern = self[p]
                all_idxs = set()
                for pat in pattern:
                    all_idxs.update(pat.idx)

                [anomalies.append(ix) for ix in all_idxs if ix not in pattern.idx]

        return anomalies

    def __hash__(self):
        return hash(str(self))

    def __eq__(self, other):
        eq = len(self) == len(other)
        for s, o in zip(self, other):
            eq = eq and s == o
        return eq and self.global_freq == other.global_freq

    def __repr__(self):
        vals = ''
        for i in self:
            vals += str(i)
        return '{}({})'.format(self.__class__.__name__, vals)


class RowPatterns(Patterns):
    """Token type tracker for each distinct pattern that appears in the rows
    """

    def __init__(self, size_coverage: float = 1):
        """initializes the row object

        Parameters
        ----------
        size_coverage :  float
            the threshold value for the PatternElementSizeMonitor
        """
        super(RowPatterns, self).__init__(size_coverage)

    def insert(self, row: Iterable[Token]):
        """Inserts a row into the discovered patterns or updates the PatternRow object

        Parameters
        ----------
        row : tuple of Tokens
        """

        self.global_freq += 1
        types = list()
        [types.append(i.regex_type) for i in row]

        key = Patterns.keygen(types)

        if key not in self:
            self[key] = SingularRowPattern()
            for r in row:
                self[key].append(PatternElementSizeMonitor(threshold=self.size_coverage).update(r))
            self[key].freq += 1
            self[key].idx.add(r.rowidx)
        else:
            self[key].update(row)

    def condense(self):
        """executes the pattern element size monitors and creates final pattern elements that have anomalous values
        excluded.

        Returns
        -------
            RowPatterns
        """
        for pats in self.values():
            for i, s in enumerate(pats):
                pats.container[i] = s.load()  # load PatternElement from size monitor
        return self

    def distribution(self):
        """returns each pattern and it's frequency

        Returns
        -------
            dict
        """
        freqs = defaultdict(float)
        for p in self:
            freqs[p] = self[p].freq
        return freqs


class ColumnPatterns(Patterns):
    """
    Token type tracker for each supported datatype that appears in the same column token position
    e.g.
    123 BARCLAY AVE, NY === NUM(3) SPACE ALPHA(7) SPACE STREET PUNC(,) SPACE STATE
    23 NEWTON ST, OH ====== NUM(2) SPACE ALPHA(6) SPACE STREET PUNC(,) SPACE STATE
    ABRA, KADABRA AVE, MN = ALPHA(4) PUNC(,) SPACE ALPHA(7) SPACE STREET PUNC(,) SPACE STATE
    for column token position 0, the OpencleanPattern components would be:
        {
            NUM:
                PatternElementSizeMonitor
                    PatternElementSet{
                        self.regex_type = NUM
                        self.size = 3
                        self.freq = 1
                        self.values = {123}
                        self.idx = {0}
                    },
                    PatternElementSet{
                        self.regex_type = NUM
                        self.size = 2
                        self.freq = 1
                        self.values = {23}
                        self.idx = {1}
                    }
                }
            ALPHA:
                PatternElementSizeMonitor{
                    PatternElementSet{
                        self.regex_type = ALPHA
                        self.size = 4
                        self.freq = 1
                        self.values = {'abra'}
                        self.idx = {2}
                    }
                }
        }
        self.global_min = 2
        self.global_max = 4
        self.global_freq = 3

    The insert method adds tokens to the respective PatternElementSizeMonitor. The final PatternElements are
    loaded from the SizeMonitors and condensed into a Regex Expression as a RowPatterns
    """

    def __init__(self, size_coverage: float = 1):
        """initializes the PatternColumn object

        Parameters
        ----------
        size_coverage :  float
            the threshold value for the PatternElementSizeMonitor
        """
        super(ColumnPatterns, self).__init__(size_coverage)

    def insert(self, row: Iterable[Token]):
        """insert the row into the respective method

        Parameters
        ----------
        row : list of Tokens
            the tokens to insert/ use to update the respective PatternColumnElement
        """
        self.global_freq += 1
        for key, token in enumerate(row):
            if not isinstance(token, Token):
                raise TypeError("expected: openclean.function.token.base.Token, got: {}".format(token.__class__))

            if key not in self:
                self[key] = SingularColumnPattern(self.size_coverage)

            self[key].update(token)

    def condense(self):
        """finds the top element in each column and returns the derived pattern

        Returns
        -------
            RowPatterns
        """
        pattern = list()
        for ind, col in self.items():
            pattern.append(col.top())

        types = list()
        [types.append(i.element_type) for i in pattern]

        key = Patterns.keygen(pattern)

        patterns = RowPatterns()
        patterns[key] = SingularRowPattern()

        # add patterns and idx and freq of pattern
        for pat in pattern:
            patterns[key].append(pat)

            # calculate intersection of all indices as the indices of interest
            if len(patterns[key].idx) == 0:
                patterns[key].idx = pat.idx
            else:
                patterns[key].idx = patterns[key].idx.intersection(pat.idx)

            patterns[key].freq = len(patterns[key].idx)

        # add global freq (total rows)
        patterns.global_freq = self.global_freq

        return patterns


# -- Pattern element / building blocks / Monitor ----------------------------------------

class PatternElementSet(object):
    """An individual set object to store values, idx and freq of the same type and size

            PatternElementSet{
                self.regex_type = NUM
                self.size = 2
                self.freq = 1
                self.values = list()
                self.idx = set()
            },
    """

    def __init__(self):
        """init the set"""
        self.regex_type = None
        self.size = 0
        self.values = set()
        self.idx = set()
        self.freq = 0

    def add(self, token: Token):
        if self.regex_type is None:
            self.regex_type = token.regex_type
            self.size = token.size
        elif token.regex_type is not self.regex_type:
            raise Exception("Incompatible Token used to update PatternElementSet")
        self.values.add(token.value)
        self.idx.add(token.rowidx)
        self.freq += 1

        return self

    def __hash__(self):
        return hash(str(self.regex_type) + str(self.size))

    def __iter__(self):
        return self


class PatternElementSizeMonitor(defaultdict):
    """Keeps track of all the pattern element sets. A set is one with the same
    regex type tokens of the same size. e.g. 4 digit numbers will be one set
    and 5 digit numbers another. Only the biggest sets are compiled into a
    PatternElemenet. This is done to identify and prevent anomalous values from
    being introduced during pattern compilation.
    """

    def __init__(self, threshold: float = 1):
        """init the monitor

        Parameters
        ----------
        threshold: int (default: 100%)
            the proportion of values that is considered non-anomalous. By default,
            values of all size will be included in the pattern.
        """
        self.default_factory = PatternElementSet
        self.freq = 0
        self.threshold = threshold

    def update(self, token: Token):
        """update the elements in the tracker

        Parameters
        ----------
        token: Token
            the token object to insert into the monitor
        """
        self[token.size].add(token)
        self.freq += 1

        return self

    def load(self):
        """On the tracked sets, perform this pseudocode:

        1. get the frequency for each set and sort them
        2. starting from the largest, keep adding to the pattern to evolve it
        3. stop when 97.5% of the values have been added

        Note: if there are multiple sets with the same frequency, to be fair
        add them all before stopping even if you're not able to stop at 90%
        """
        # create a counter and sort the frequencies
        minmax = Counter()
        for i in self.values():
            minmax[i.size] += i.freq
        common = minmax.most_common()

        freq_sets = defaultdict(list)
        for c in common:
            freq_sets[c[1]].append(c[0])

        # select sizes to be able to combine 97.5% of the data
        covered = 0
        sorted_freqs = sorted(freq_sets.keys(), reverse=True)
        pe = None
        for k in sorted_freqs:
            v = freq_sets[k]
            for set_id in v:
                if pe is None:
                    pe = PatternElement(self[set_id])
                else:
                    pe.update(self[set_id])
                covered += k

            if covered / self.freq > self.threshold:
                break

        return pe


class PatternElement(object):
    """
    Element tracker for a single supported datatype that appear in the same column token position
    e.g.
    231 BARCLAY AVE, NY === NUM(3) SPACE ALPHA(7) SPACE STREET PUNC(,) SPACE STATE
    23 NEWTON ST, OH ====== NUM(2) SPACE ALPHA(6) SPACE STREET PUNC(,) SPACE STATE

    for column token position 0, the PatternColumnElement object  would be:

        self.element_type = DIGIT #type of regex element
        self.regex = NUMERIC #regex representation
        self.len_min = 2 #min len
        self.len_max = 3 #max len

        self.idx = [0, 1] #list of indices that went into this element. useful to trace mismatches back to rows

        self.punc_list = [] #list of punc tokens if this is a PUNCTUATION element

        self.partial_regex = 23X #partial regex value
        self.partial_ambiguous = False #is partial regex value too ambiguous to be used

        self.freq = 2

    The update_pattern method keeps adding similar type tokens to PatternColumnElement object and once all the tokens have been
    exhausted, the final PatternColumnElement object  is condensed by PatternColumn into a Regex Expression for the position
    """

    def __init__(self, token: Token = None):
        """initializes the PatternElement and keeps track of numerous stats incrementally as it builds the regexp

        Parameters
        ----------
        token : Token
            the token used to create this PatternElement object
        """
        self.element_type = None  # type of regex element
        self.len_min = np.inf  # min len
        self.len_max = -np.inf  # max len
        self.values = set()

        self.idx = set()  # list of indices that went into this element. useful to trace mismatches back to rows
        self.punc_list = list()  # list of punc tokens if this is a PUNCTUATION elemenet
        self.partial_regex = None  # partial regex value
        self.partial_ambiguous = False  # is partial regex value too ambiguous to be used
        self.freq = 0  # total frequency of tokens seen (useful for element proportions to identify anomalous patterns)

        if isinstance(token, Token):
            token = PatternElementSet().add(token)
        if token is not None:
            self.from_set(token)  # init PatternElement from PatternElementSet

    def update(self, next_input: Union[PatternElementSet, Token]):
        """updates the PatternElement object

        Parameters
        ----------
        next_input : Token or PatternElementSet
            the token or set to update this PatternElement object

        """
        if isinstance(next_input, Token):
            next_input = PatternElementSet().add(next_input)

        if isinstance(next_input, PatternElementSet):
            if next_input.regex_type == SupportedDataTypes.PUNCTUATION:
                self.punc_list += next_input.values
            elif not self.partial_ambiguous:
                unknown_threshold = 0.8
                # todo: because partial regex is built incrementally too, the order
                # of token.values can have a huge impact on the results. Is there
                # another way? what about updating with a tree like clusterW for alignment?
                for val in next_input.values:
                    if self.partial_regex is None:
                        self.partial_regex = val
                    new_partial_regex, ambiguity_ratio = StringComparator.compare_strings(self.partial_regex, val)
                    if ambiguity_ratio > unknown_threshold:
                        self.partial_ambiguous = True
                    self.partial_regex = new_partial_regex
            self.freq += next_input.freq
            self.idx = self.idx.union(next_input.idx)
            self.values = self.values.union(next_input.values)
            self.len_min = min(self.len_min, next_input.size)
            self.len_max = max(self.len_max, next_input.size)
        else:
            raise TypeError("expected Token or PatternElementSet")

    def from_set(self, s: PatternElementSet):
        """create a Pattern Element object from input set

        Parameters
        ----------
        s : PatternElementSet
            the set to use as the init
        """
        if self.element_type is not None:
            raise Exception("use from_set to create new PatternElements. To update an existing one, use 'update'")

        self.element_type = s.regex_type  # type of regex element
        self.update(s)

    def __str__(self):
        """String representation of the PatternElement object

        Returns
        -------
            str
        """
        if self.element_type in [SupportedDataTypes.PUNCTUATION, SupportedDataTypes.SPACE_REP]:
            return '{}({})'.format(self.element_type, ''.join(self.punc_list))

        pattern = self.element_type if self.partial_ambiguous else self.partial_regex
        return '{}({}-{})'.format(pattern, self.len_min, self.len_max)

    def __repr__(self):
        return '{}({}-{})'.format(self.element_type, self.len_min, self.len_max)

    def __eq__(self, other):
        return self.element_type == other.element_type and \
               self.len_min == other.len_min and \
               self.len_max == other.len_max and \
               self.idx == other.idx and \
               self.punc_list == other.punc_list and \
               self.partial_regex == other.partial_regex and \
               self.partial_ambiguous == other.partial_ambiguous and \
               self.freq == other.freq
