/**
 * plugin.js
 *
 * Released under LGPL License.
 * Copyright (c) 1999-2015 Ephox Corp. All rights reserved
 *
 * License: http://www.tinymce.com/license
 * Contributing: http://www.tinymce.com/contributing
 */

/*global tinymce:true */

tinymce.PluginManager.add('cmsimage', function(editor) {
  function getImageSize(url, callback) {
    var img = document.createElement('img');

    function done(width, height) {
      if (img.parentNode) {
        img.parentNode.removeChild(img);
      }

      callback({width: width, height: height});
    }

    img.onload = function() {
      done(Math.max(img.width, img.clientWidth), Math.max(img.height, img.clientHeight));
    };

    img.onerror = function() {
      done();
    };

    var style = img.style;
    style.visibility = 'hidden';
    style.position = 'fixed';
    style.bottom = style.left = 0;
    style.width = style.height = 'auto';

    document.body.appendChild(img);
    img.src = url;
  }

  function buildListItems(inputList, itemCallback, startItems) {
    function appendItems(values, output) {
      output = output || [];

      tinymce.each(values, function(item) {
        var menuItem = {text: item.text || item.title};

        if (item.menu) {
          menuItem.menu = appendItems(item.menu);
        } else {
          menuItem.value = item.value;
          itemCallback(menuItem);
        }

        output.push(menuItem);
      });

      return output;
    }

    return appendItems(inputList, startItems || []);
  }

  function createImageList(callback) {
    return function() {
      var imageList = editor.settings.image_list;

      if (typeof imageList == "string") {
        tinymce.util.XHR.send({
          url: imageList,
          success: function(text) {
            callback(tinymce.util.JSON.parse(text));
          }
        });
      } else if (typeof imageList == "function") {
        imageList(callback);
      } else {
        callback(imageList);
      }
    };
  }

  function showDialog(imageList) {
    var win, data = {}, dom = editor.dom, imgElm, figureElm;
    var width, height, imageDimensions = editor.settings.image_dimensions !== false;

    function recalcSize() {
      var widthCtrl, heightCtrl, newWidth, newHeight;

      widthCtrl = win.find('#width')[0];
      heightCtrl = win.find('#height')[0];

      if (!widthCtrl || !heightCtrl) {
        return;
      }

      newWidth = widthCtrl.value();
      newHeight = heightCtrl.value();

      if (win.find('#constrain')[0].checked() && width && height && newWidth && newHeight) {
        if (width != newWidth) {
          newHeight = Math.round((newWidth / width) * newHeight);

          if (!isNaN(newHeight)) {
            heightCtrl.value(newHeight);
          }
        } else {
          newWidth = Math.round((newHeight / height) * newWidth);

          if (!isNaN(newWidth)) {
            widthCtrl.value(newWidth);
          }
        }
      }

      width = newWidth;
      height = newHeight;
    }

    /*
     * Stuff in here handles the resizing of the images in the wysiwyg
     */
    function onSubmitForm() {
      var figureElm, oldImg;

      function waitLoad(imgElm) {
        function selectImage() {
          imgElm.onload = imgElm.onerror = null;

          if (editor.selection) {
            editor.selection.select(imgElm);
            editor.nodeChanged();
          }
        }

        imgElm.onload = function() {
          if (!data.width && !data.height && imageDimensions) {
            dom.setAttribs(imgElm, {
              width: imgElm.clientWidth,
              height: imgElm.clientHeight
            });
          }

          selectImage();
        };

        imgElm.onerror = selectImage;
      }

      updateStyle();
      recalcSize();

      data = tinymce.extend(data, win.toJSON());

      // Setup new data excluding style properties
      data = {
        src: data.src,
        alt: data.alt || '',
        title: data.title || '',
        width: data.width === '' ? null : data.width,
        height: data.height === '' ? null : data.height,
        style: data.style || null,
        "class": data["class"]
      };

      editor.undoManager.transact(function() {
        if (!data.src) {
          if (imgElm) {
            dom.remove(imgElm);
            editor.focus();
            editor.nodeChanged();
          }

          return;
        }

        if (data.title === "") {
          data.title = null;
        }

        if (!imgElm) {
          data.id = '__mcenew';
          editor.focus();
          editor.selection.setContent(dom.createHTML('img', data));
          imgElm = dom.get('__mcenew');
          dom.setAttrib(imgElm, 'id', null);
        } else {
          dom.setAttribs(imgElm, data);
        }

        editor.editorUpload.uploadImagesAuto();

        if (dom.is(imgElm.parentNode, 'figure.image')) {
          figureElm = imgElm.parentNode;
          dom.insertAfter(imgElm, figureElm);
          dom.remove(figureElm);
        }

        waitLoad(imgElm);
      });
    }

    imgElm = editor.selection.getNode();
    figureElm = dom.getParent(imgElm, 'figure.image');

    if (figureElm) {
      imgElm = dom.select('img', figureElm)[0];
    }

    if (imgElm && (imgElm.nodeName != 'IMG' || imgElm.getAttribute('data-mce-object') || imgElm.getAttribute('data-mce-placeholder'))) {
      imgElm = null;
    }

    if (imgElm) {
      width = dom.getAttrib(imgElm, 'width');
      height = dom.getAttrib(imgElm, 'height');

      data = {
        src: dom.getAttrib(imgElm, 'src'),
        alt: dom.getAttrib(imgElm, 'alt'),
        title: dom.getAttrib(imgElm, 'title'),
        "class": dom.getAttrib(imgElm, 'class'),
        width: width,
        height: height
      };
    }

    function getFormItems() {
      // General settings shared between simple and advanced dialogs
      var generalFormItems = [];

      if (editor.settings.image_description !== false) {
        generalFormItems.push({
          name: 'alt',
          type: 'textbox',
          label: 'Image description'
        });
      }

      if (editor.settings.image_title) {
        generalFormItems.push({
          name: 'title',
          type: 'textbox',
          label: 'Image Title'
        });
      }

      if (imageDimensions) {
        generalFormItems.push({
          type: 'container',
          label: 'Dimensions',
          layout: 'flex',
          direction: 'row',
          align: 'center',
          spacing: 5,
          items: [
            {
              name: 'width',
              type: 'textbox',
              maxLength: 5,
              size: 3,
              onchange: recalcSize,
              ariaLabel: 'Width'
            },
            {type: 'label', text: 'x'},
            {
              name: 'height',
              type: 'textbox',
              maxLength: 5,
              size: 3,
              onchange: recalcSize,
              ariaLabel: 'Height'
            },
            {
              name: 'constrain',
              type: 'checkbox',
              checked: true,
              text: 'Constrain proportions'
            }
          ]
        });
      }

      return generalFormItems;
    }

    function updateStyle() {
      function addPixelSuffix(value) {
        if (value.length > 0 && /^[0-9]+$/.test(value)) {
          value += 'px';
        }

        return value;
      }

      if (!editor.settings.image_advtab) {
        return;
      }

      var data = win.toJSON(),
        css = dom.parseStyle(data.style);

      win.find('#style').value(dom.serializeStyle(dom.parseStyle(dom.serializeStyle(css))));
    }

    function createMediaListIframe() {
      var iframe = document.createElement('iframe')
      iframe.src = window.cmsUrls.imageLibraryIframe + '?&file__iregex=.(png|gif|jpg|jpeg)$'
      iframe.frameborder = '0'
      iframe.width = '100%'
      return iframe.outerHTML
    }

    // Set active editor
    window.active_tinymce = editor

    // Generate the media list table HTML
    var iframeHtml = createMediaListIframe();

    // Advanced dialog shows general+advanced tabs
    win = editor.windowManager.open({
      title: 'Insert/edit image',
      data: data,
      bodyType: 'tabpanel',
      body: [
        {
          title: 'Media gallery',
          type: 'panel',
          minWidth: 750,
          minHeight: 500,
          spacing: 5,
          items: [
            {
              type: 'container',
              html: iframeHtml
            }
          ]
        },
        {
          title: 'Advanced',
          type: 'form',
          items: getFormItems()
        },
      ],
      onSubmit: onSubmitForm
    });

    window.active_tinymce_window = win
  }

  editor.addButton('cmsimage', {
    icon: 'image',
    tooltip: 'Insert/edit image',
    onclick: createImageList(showDialog),
    stateSelector: 'img:not([data-mce-object],[data-mce-placeholder]),figure.image'
  });

  editor.addMenuItem('cmsimage', {
    icon: 'image',
    text: 'Insert/edit image',
    onclick: createImageList(showDialog),
    context: 'insert',
    prependToContext: true
  });

  editor.addCommand('mceImage', createImageList(showDialog));
});
