"use strict";
// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
Object.defineProperty(exports, "__esModule", { value: true });
const default_1 = require("./default");
/**
 * A namespace for session interfaces and factory functions.
 */
var Session;
(function (Session) {
    /**
     * List the running sessions.
     *
     * @param settings - The server settings to use for the request.
     *
     * @returns A promise that resolves with the list of session models.
     *
     * #### Notes
     * Uses the [Jupyter Notebook API](http://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter/notebook/master/notebook/services/api/api.yaml#!/sessions), and validates the response.
     *
     * All client-side sessions are updated with current information.
     *
     * The promise is fulfilled on a valid response and rejected otherwise.
     */
    function listRunning(settings) {
        return default_1.DefaultSession.listRunning(settings);
    }
    Session.listRunning = listRunning;
    /**
     * Start a new session.
     *
     * @param options - The options used to start the session.
     *
     * @returns A promise that resolves with the session instance.
     *
     * #### Notes
     * Uses the [Jupyter Notebook API](http://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter/notebook/master/notebook/services/api/api.yaml#!/sessions), and validates the response.
     *
     * A path must be provided.  If a kernel id is given, it will
     * connect to an existing kernel.  If no kernel id or name is given,
     * the server will start the default kernel type.
     *
     * The promise is fulfilled on a valid response and rejected otherwise.
     *
     * Wrap the result in an Session object. The promise is fulfilled
     * when the session is created on the server, otherwise the promise is
     * rejected.
     */
    function startNew(options) {
        return default_1.DefaultSession.startNew(options);
    }
    Session.startNew = startNew;
    /**
     * Find a session by id.
     *
     * @param id - The id of the target session.
     *
     * @param settings - The server settings.
     *
     * @returns A promise that resolves with the session model.
     *
     * #### Notes
     * If the session was already started via `startNew`, the existing
     * Session object's information is used in the fulfillment value.
     *
     * Otherwise, we attempt to find to the existing session.
     * The promise is fulfilled when the session is found,
     * otherwise the promise is rejected.
     */
    function findById(id, settings) {
        return default_1.DefaultSession.findById(id, settings);
    }
    Session.findById = findById;
    /**
     * Find a session by path.
     *
     * @param path - The path of the target session.
     *
     * @param settings: The server settings.
     *
     * @returns A promise that resolves with the session model.
     *
     * #### Notes
     * If the session was already started via `startNewSession`, the existing
     * Session object's info is used in the fulfillment value.
     *
     * Otherwise, we attempt to find to the existing
     * session using [listRunningSessions].
     * The promise is fulfilled when the session is found,
     * otherwise the promise is rejected.
     *
     * If the session was not already started and no `options` are given,
     * the promise is rejected.
     */
    function findByPath(path, settings) {
        return default_1.DefaultSession.findByPath(path, settings);
    }
    Session.findByPath = findByPath;
    /**
     * Connect to a running session.
     *
     * @param model - The model of the target session.
     *
     * @param settigns - The server settings.
     *
     * @returns The session instance.
     *
     * #### Notes
     * If the session was already started via `startNew`, the existing
     * Session object is used as the fulfillment value.
     *
     * Otherwise, we attempt to connect to the existing session.
     */
    function connectTo(model, settings) {
        return default_1.DefaultSession.connectTo(model, settings);
    }
    Session.connectTo = connectTo;
    /**
     * Shut down a session by id.
     *
     * @param id - The id of the target session.
     *
     * @param settings - The server settings.
     *
     * @returns A promise that resolves when the session is shut down.
     *
     */
    function shutdown(id, settings) {
        return default_1.DefaultSession.shutdown(id, settings);
    }
    Session.shutdown = shutdown;
    /**
     * Shut down all sessions.
     *
     * @returns A promise that resolves when all of the sessions are shut down.
     */
    function shutdownAll(settings) {
        return default_1.DefaultSession.shutdownAll(settings);
    }
    Session.shutdownAll = shutdownAll;
})(Session = exports.Session || (exports.Session = {}));
//# sourceMappingURL=session.js.map