"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrowApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const lambda = require("aws-cdk-lib/aws-lambda");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const logs = require("aws-cdk-lib/aws-logs");
/**
 * For copying shared code to all paths
 */
const fse = require("fs-extra");
const DEFAULT_LAMBDA_CODE = `
exports.handler = async function (event, context, callback) {
  try {
    const data = {
      statusCode: 201,
    };
    return data;
  } catch (uncaughtError) {
    console.error(uncaughtError);
    throw uncaughtError;
  }
}
`;
/**
 * @experimental
 */
class CrowApi extends constructs_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        const { sourceDirectory = 'src', sharedDirectory = 'shared', useAuthorizerLambda = false, authorizerDirectory = 'authorizer', authorizerConfiguration = {}, createApiKey = false, logRetention = logs.RetentionDays.ONE_WEEK, databaseTables: databaseTablesUsed = {}, apiGatewayConfiguration = {}, apiGatewayName = 'crow-api', lambdaConfigurations = {}, } = props;
        const specialDirectories = [
            sharedDirectory,
            authorizerDirectory,
        ];
        const defaultLambda = new lambda.Function(this, 'default-crow-lambda', {
            runtime: lambda.Runtime.NODEJS_12_X,
            code: new lambda.InlineCode(DEFAULT_LAMBDA_CODE),
            handler: 'index.handler',
            logRetention,
        });
        // API Gateway log group
        const gatewayLogGroup = new logs.LogGroup(this, 'api-access-logs', {
            retention: logs.RetentionDays.ONE_WEEK,
        });
        // The API Gateway itself
        const gateway = new apigateway.LambdaRestApi(this, apiGatewayName, {
            handler: defaultLambda,
            proxy: false,
            deploy: true,
            deployOptions: {
                loggingLevel: apigateway.MethodLoggingLevel.ERROR,
                accessLogDestination: new apigateway.LogGroupLogDestination(gatewayLogGroup),
            },
            apiKeySourceType: createApiKey ? apigateway.ApiKeySourceType.HEADER : undefined,
            ...apiGatewayConfiguration,
        });
        if (createApiKey) {
            // API Key
            const apiKey = gateway.addApiKey('api-key');
            const usagePlan = new apigateway.UsagePlan(this, 'usage-plan', {
                throttle: {
                    burstLimit: 5000,
                    rateLimit: 10000,
                },
                apiStages: [
                    {
                        api: gateway,
                        stage: gateway.deploymentStage,
                    },
                ],
            });
            usagePlan.addApiKey(apiKey);
        }
        let tokenAuthorizer;
        if (useAuthorizerLambda) {
            const fullAuthorizerDirectory = `${sourceDirectory}/${authorizerDirectory}`;
            copySharedDirectory(fullAuthorizerDirectory);
            const authorizerLambda = new lambda.Function(this, 'authorizer-lambda', {
                runtime: lambda.Runtime.NODEJS_14_X,
                code: lambda.Code.fromAsset(fullAuthorizerDirectory),
                handler: 'index.handler',
                logRetention,
            });
            this.authorizerLambda = authorizerLambda;
            const tokenAuthorizerConfiguration = {
                handler: authorizerLambda,
                resultsCacheTtl: cdk.Duration.seconds(3600),
                ...authorizerConfiguration,
            };
            tokenAuthorizer = new apigateway.TokenAuthorizer(this, 'token-authorizer', tokenAuthorizerConfiguration);
        }
        // Returns child directories given the path of a parent
        function getDirectoryChildren(parentDirectory) {
            const directories = fse.readdirSync(parentDirectory, { withFileTypes: true })
                .filter((dirent) => dirent.isDirectory())
                .map((dirent) => dirent.name);
            return directories;
        }
        // Copies shared directory to a given target directory
        function copySharedDirectory(targetPath) {
            const sourceSharedDirectory = `${sourceDirectory}/${sharedDirectory}`;
            const targetSharedDirectory = `${targetPath}/${sharedDirectory}`;
            // NOTE this is file I/O so it takes a while
            fse.emptyDirSync(targetSharedDirectory);
            // Copy shared code to target path
            try {
                fse.copySync(sourceSharedDirectory, targetSharedDirectory);
            }
            catch (err) {
                // console.error(err);
                console.error(`There was an error copying the shared directory to ${targetSharedDirectory}`);
                // Let this pass and not disrupt the entire application
            }
        }
        // Given a path, look for crow.json and return configuration
        function getConfiguration(path) {
            const configurationFile = `${path}/crow.json`;
            try {
                if (fse.existsSync(configurationFile)) {
                    const configuration = fse.readJsonSync(configurationFile);
                    return configuration;
                }
            }
            catch (err) {
                console.error(err);
                // Don't want to crash synth if config file isn't present
            }
            return {};
        }
        function bundleLambdaProps(userConfiguration, codePath, apiPath) {
            const { lambdaConfiguration, databaseTables, } = userConfiguration;
            const propConfiguration = lambdaConfigurations[apiPath] || {};
            const lambdaProps = {
                runtime: lambda.Runtime.NODEJS_14_X,
                code: lambda.Code.fromAsset(codePath),
                handler: 'index.handler',
                logRetention,
                environment: {},
                ...lambdaConfiguration,
                ...propConfiguration,
            };
            if (databaseTables) {
                const tableEnvironmentVariables = {};
                Object.entries(databaseTables).forEach(([table, environmentVariableName]) => {
                    var _b;
                    tableEnvironmentVariables[environmentVariableName] = (_b = databaseTablesUsed[table]) === null || _b === void 0 ? void 0 : _b.tableName;
                });
                const environmentWithTables = {
                    // Let any manual environment variables take precedence over
                    //   automated ones
                    ...tableEnvironmentVariables,
                    ...lambdaProps.environment,
                };
                lambdaProps.environment = environmentWithTables;
            }
            return lambdaProps;
        }
        function grantTablePermissions(newLambda, userConfiguration) {
            const { databaseTables, } = userConfiguration;
            if (!databaseTables) {
                return;
            }
            Object.keys(databaseTables).forEach((table) => {
                var _b;
                (_b = databaseTablesUsed[table]) === null || _b === void 0 ? void 0 : _b.grantFullAccess(newLambda);
            });
        }
        const root = sourceDirectory;
        const verbs = ['get', 'post', 'put', 'delete'];
        const graph = {};
        const lambdasByPath = {};
        // Initialize with root
        graph['/'] = {
            resource: gateway.root,
            path: root,
            paths: [],
            verbs: [],
        };
        // First is directory path, second is API path
        const nodes = [[root, '/']];
        // BFS that creates API Gateway structure and copies shared code
        while (nodes.length) {
            const [directoryPath, apiPath] = nodes.shift() || ['i hate', 'typescript'];
            const children = getDirectoryChildren(directoryPath);
            // console.log(`${apiPath}'s children are: ${children}`);
            // Don't have to worry about previously visited nodes since this is a file structure...unless there are symlinks?
            children.forEach((child) => {
                const newDirectoryPath = `${directoryPath}/${child}`;
                // If we're on the root path, don't separate with a slash (/)
                //   because it ends up looking like //child-path
                const newApiPath = apiPath === '/' ? `/${child}` : `${apiPath}/${child}`;
                if (verbs.includes(child)) {
                    // If directory is a verb, we don't traverse it anymore
                    //   and need to create an API Gateway method and Lambda
                    const configuration = getConfiguration(newDirectoryPath);
                    const lambdaProps = bundleLambdaProps(configuration, newDirectoryPath, newApiPath);
                    const { useAuthorizerLambda: authorizerLambdaConfigured } = configuration;
                    copySharedDirectory(newDirectoryPath);
                    const newLambda = new lambda.Function(this, newDirectoryPath, lambdaProps);
                    grantTablePermissions(newLambda, configuration);
                    const methodConfiguration = {};
                    if (authorizerLambdaConfigured && useAuthorizerLambda) {
                        methodConfiguration.authorizationType = apigateway.AuthorizationType.CUSTOM;
                        methodConfiguration.authorizer = tokenAuthorizer;
                    }
                    graph[apiPath].resource.addMethod(child.toUpperCase(), new apigateway.LambdaIntegration(newLambda), methodConfiguration);
                    graph[apiPath].verbs.push(child);
                    lambdasByPath[newApiPath] = newLambda;
                }
                else if (specialDirectories.includes(child)) {
                    // The special directories should not result in an API path
                    // This means the API also cannot have a resource with the
                    //   same name
                }
                else {
                    // If directory is not a verb, create new API Gateway resource
                    //   for use by verb directory later
                    const newResource = graph[apiPath].resource
                        .resourceForPath(child);
                    nodes.push([newDirectoryPath, newApiPath]);
                    // Add child to parent's paths
                    graph[apiPath].paths.push(child);
                    // Initialize graph to include child
                    graph[newApiPath] = {
                        resource: newResource,
                        path: newDirectoryPath,
                        paths: [],
                        verbs: [],
                    };
                }
            });
        }
        // console.log(graph);
        // Expose API Gateway
        this.gateway = gateway;
        this.lambdaFunctions = lambdasByPath;
    }
}
exports.CrowApi = CrowApi;
_a = JSII_RTTI_SYMBOL_1;
CrowApi[_a] = { fqn: "crow-api.CrowApi", version: "1.0.2" };
//# sourceMappingURL=data:application/json;base64,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