import os
import sys

import Orange.data
from AnyQt.QtWidgets import QApplication
from Orange.widgets import widget
from Orange.widgets.utils.signals import Input, Output

if "site-packages/Orange/widgets" in os.path.dirname(os.path.abspath(__file__)).replace("\\", "/"):
    from Orange.widgets.orangecontrib.AAIT.llm import lemmes
    from Orange.widgets.orangecontrib.AAIT.utils import thread_management
    from Orange.widgets.orangecontrib.AAIT.utils.import_uic import uic
    from Orange.widgets.orangecontrib.AAIT.utils.initialize_from_ini import apply_modification_from_python_file
else:
    from orangecontrib.AAIT.llm import lemmes
    from orangecontrib.AAIT.utils import thread_management
    from orangecontrib.AAIT.utils.import_uic import uic
    from orangecontrib.AAIT.utils.initialize_from_ini import apply_modification_from_python_file


@apply_modification_from_python_file(filepath_original_widget=__file__)
class OWLemmatizer(widget.OWWidget):
    name = "Lemmes & Tags"
    description = "Generate the lemmes and the tags on the column 'content' of a Table"
    icon = "icons/owlemmatizer.svg"
    if "site-packages/Orange/widgets" in os.path.dirname(os.path.abspath(__file__)).replace("\\", "/"):
        icon = "icons_dev/owlemmatizer.svg"
    gui = os.path.join(os.path.dirname(os.path.abspath(__file__)), "designer/owlemmatizer.ui")
    want_control_area = False
    priority = 1070

    class Inputs:
        request = Input("Request", Orange.data.Table)
        data = Input("Data", Orange.data.Table)
        model_path = Input("Model", str)

    class Outputs:
        data = Output("Data", Orange.data.Table)

    @Inputs.request
    def set_request(self, in_request):
        self.data = in_request
        self.request = True
        if self.autorun == True:
            self.run()

    @Inputs.data
    def set_data(self, in_data):
        self.data = in_data
        self.request = False
        if self.autorun==True:
            self.run()

    @Inputs.model_path
    def set_model(self, in_model_path):
        self.model_path = in_model_path
        if self.autorun == True:
            self.run()

    def __init__(self):
        super().__init__()
        # Qt Management
        self.setFixedWidth(470)
        self.setFixedHeight(300)
        uic.loadUi(self.gui, self)

        # Data Management
        self.request = False
        self.data = None
        self.model_path = None
        self.thread = None
        self.autorun = True
        self.result = None
        self.post_initialized()

    def run(self):
        # if thread is running quit
        if self.thread is not None:
            self.thread.safe_quit()

        if self.data is None:
            self.Outputs.data.send(None)
            return

        if self.model_path is None:
            self.Outputs.data.send(None)
            return

        # Verification of in_data
        self.error("")
        if not "content" in self.data.domain:
            self.error('You need a "content" column in input data')
            return

        if type(self.data.domain["content"]).__name__ != 'StringVariable':
            self.error('"content" column needs to be a Text')
            return

        # Start progress bar
        self.progressBarInit()

        # Connect and start thread : main function, progress, result and finish
        # --> progress is used in the main function to track progress (with a callback)
        # --> result is used to collect the result from main function
        # --> finish is just an empty signal to indicate that the thread is finished
        self.thread = thread_management.Thread(lemmes.create_lemmes_and_tags, self.data, self.model_path, self.request)
        self.thread.progress.connect(self.handle_progress)
        self.thread.result.connect(self.handle_result)
        self.thread.finish.connect(self.handle_finish)
        self.thread.start()

    def handle_progress(self, value: float) -> None:
        self.progressBarSet(value)

    def handle_result(self, result):
        try:
            self.result = result
            self.Outputs.data.send(result)
        except Exception as e:
            print("An error occurred when sending out_data:", e)
            self.Outputs.data.send(None)
            return

    def handle_finish(self):
        self.progressBarFinished()

    def post_initialized(self):
        pass

if __name__ == "__main__":
    app = QApplication(sys.argv)
    my_widget = OWLemmatizer()
    my_widget.show()
    if hasattr(app, "exec"):
        app.exec()
    else:
        app.exec_()
