# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['np_validator', 'np_validator.dataclasses']

package_data = \
{'': ['*']}

install_requires = \
['PyYAML>=6.0,<7.0',
 'h5py>=3.7.0,<4.0.0',
 'marshmallow>=3.17.0,<4.0.0',
 'rich>=12.5.1,<13.0.0',
 'typer>=0.6.1,<0.7.0']

extras_require = \
{':python_version < "3.8"': ['importlib_metadata>=4.5.0,<5.0.0'],
 ':python_version >= "3.8" and python_version < "4.0"': ['numpy>=1.23.2,<2.0.0']}

entry_points = \
{'console_scripts': ['np-validator = np_validator.__main__:app']}

setup_kwargs = {
    'name': 'np-validator',
    'version': '0.4.9',
    'description': 'Awesome `np-validator` is a Python cli/package created with https://github.com/TezRomacH/python-package-template',
    'long_description': '# np-validator\n\n<div align="center">\n\n[![Build status](https://github.com/np_validator/np-validator/workflows/build/badge.svg?branch=master&event=push)](https://github.com/np_validator/np-validator/actions?query=workflow%3Abuild)\n[![Python Version](https://img.shields.io/pypi/pyversions/np-validator.svg)](https://pypi.org/project/np-validator/)\n[![Dependencies Status](https://img.shields.io/badge/dependencies-up%20to%20date-brightgreen.svg)](https://github.com/np_validator/np-validator/pulls?utf8=%E2%9C%93&q=is%3Apr%20author%3Aapp%2Fdependabot)\n\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n[![Security: bandit](https://img.shields.io/badge/security-bandit-green.svg)](https://github.com/PyCQA/bandit)\n[![Pre-commit](https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white)](https://github.com/np_validator/np-validator/blob/master/.pre-commit-config.yaml)\n[![Semantic Versions](https://img.shields.io/badge/%20%20%F0%9F%93%A6%F0%9F%9A%80-semantic--versions-e10079.svg)](https://github.com/np_validator/np-validator/releases)\n[![License](https://img.shields.io/github/license/np_validator/np-validator)](https://github.com/np_validator/np-validator/blob/master/LICENSE)\n![Coverage Report](assets/images/coverage.svg)\n\n`np-validator` is just a simple Python cli/package that validates data sources for the neuropixel pipline using templated workflows.\n\n</div>\n\n## Quick overview\n\n### API\n\nGenerating validation steps\n\n```python\nfrom np_validator import Processor, Validator, ValidationStep, run_validation\n\n# make a basic filesize validator\nfs_validator = Validator(\n  name="meets_filesize_threshold",\n  args={\n    "threshold": 10,\n  },\n)\n\n# add validator to a validation step\nvalidation_step_0 = ValidationStep(\n  path_suffix=".mapping.pkl",\n  validators=[fs_validator, ],\n)\n\n# make a validation step with a processor\n# processors convert the data source from it\'s basic state, a filepath, to an easier to use object\n\n# unpickle, unpickles an arbitrary filepath into a python object\nunpickler = Processor(\n  name="unpickle",\n)\n\n# has_dict_key, checks if a dict-like interface has a key at path\nsession_uuid_validator = Validator(\n  name="has_dict_key",\n  args={\n    "path": ["session_uuid", ],\n  }\n)\n\n# assembling it all together as a validation step\n\nvalidation_step_1 = ValidationStep(\n  path_suffix=".behavior.pkl",\n  processor=unpickler,\n  validators=[session_uuid_validator, ],\n)\n\n# running a validation\nfilepaths = [\n  "some/sort/of/path/prefix/uuid-maybe.mapping.pkl",\n  "some/sort/of/path/prefix/uuid-maybe.behavior.pkl",\n  "this/will/be/ignored/uuid-maybe.replay.pkl",\n]\nresults = run_validation(\n  filepaths,\n  [validation_step_0, validation_step_1, ],\n)\n```\n\n## Command-line interface\n\n### Run a validation\n\n```\n$ np-validator validate file_list.json validation_steps.yml output.json\n```\n\n### Get version\n\n```\n$ np-validator main --version\n```\n\n## Documentation\n\nFor more detailed documentation on using this package please refer to the [docs](https://alleninstitute.github.io/np_validator/).\n\n## Contributing\n\n### Processor\n\nTo add a new processor, add the function to `np_validator/processors.py`. Ideally create a test for it in `tests`. Run the tests to help ensure that no regressions have been introduced. Each processor is expected to have one required argument which is expected to be a string filepath.\n\n### Validator\n\nTo add a new validator, add the function to `np_validator/validators.py`. Ideally create a test for it in `tests`. Run the tests to help ensure that no regressions have been introduced.\n\n## Very first steps\n\n### Initialize your code\n\n1. Initialize `git` inside your repo:\n\n```bash\ncd np-validator && git init\n```\n\n2. If you don\'t have `Poetry` installed run:\n\n```bash\nmake poetry-download\n```\n\n3. Initialize poetry and install `pre-commit` hooks:\n\n```bash\nmake install\nmake pre-commit-install\n```\n\n4. Run the codestyle:\n\n```bash\nmake codestyle\n```\n\n5. Upload initial code to GitHub:\n\n```bash\ngit add .\ngit commit -m ":tada: Initial commit"\ngit branch -M main\ngit remote add origin https://github.com/np_validator/np-validator.git\ngit push -u origin main\n```\n\n### Set up bots\n\n- Set up [Dependabot](https://docs.github.com/en/github/administering-a-repository/enabling-and-disabling-version-updates#enabling-github-dependabot-version-updates) to ensure you have the latest dependencies.\n- Set up [Stale bot](https://github.com/apps/stale) for automatic issue closing.\n\n### Poetry\n\nWant to know more about Poetry? Check [its documentation](https://python-poetry.org/docs/).\n\n<details>\n<summary>Details about Poetry</summary>\n<p>\n\nPoetry\'s [commands](https://python-poetry.org/docs/cli/#commands) are very intuitive and easy to learn, like:\n\n- `poetry add numpy@latest`\n- `poetry run pytest`\n- `poetry publish --build`\n\netc\n\n</p>\n</details>\n\n### Building and releasing your package\n\nBuilding a new version of the application contains steps:\n\n- Bump the version of your package `poetry version <version>`. You can pass the new version explicitly, or a rule such as `major`, `minor`, or `patch`. For more details, refer to the [Semantic Versions](https://semver.org/) standard.\n- Make a commit to `GitHub`.\n- Create a `GitHub release`.\n- And... publish 🙂 `poetry publish --build`\n\n## 🎯 What\'s next\n\nWell, that\'s up to you 💪🏻. I can only recommend the packages and articles that helped me.\n\n- [`Typer`](https://github.com/tiangolo/typer) is great for creating CLI applications.\n- [`Rich`](https://github.com/willmcgugan/rich) makes it easy to add beautiful formatting in the terminal.\n- [`Pydantic`](https://github.com/samuelcolvin/pydantic/) – data validation and settings management using Python type hinting.\n- [`Loguru`](https://github.com/Delgan/loguru) makes logging (stupidly) simple.\n- [`tqdm`](https://github.com/tqdm/tqdm) – fast, extensible progress bar for Python and CLI.\n- [`IceCream`](https://github.com/gruns/icecream) is a little library for sweet and creamy debugging.\n- [`orjson`](https://github.com/ijl/orjson) – ultra fast JSON parsing library.\n- [`Returns`](https://github.com/dry-python/returns) makes you function\'s output meaningful, typed, and safe!\n- [`Hydra`](https://github.com/facebookresearch/hydra) is a framework for elegantly configuring complex applications.\n- [`FastAPI`](https://github.com/tiangolo/fastapi) is a type-driven asynchronous web framework.\n\nArticles:\n\n- [Open Source Guides](https://opensource.guide/).\n- [A handy guide to financial support for open source](https://github.com/nayafia/lemonade-stand)\n- [GitHub Actions Documentation](https://help.github.com/en/actions).\n- Maybe you would like to add [gitmoji](https://gitmoji.carloscuesta.me/) to commit names. This is really funny. 😄\n\n## 🚀 Features\n\n### Development features\n\n- Supports for `Python 3.7` and higher.\n- [`Poetry`](https://python-poetry.org/) as the dependencies manager. See configuration in [`pyproject.toml`](https://github.com/np_validator/np-validator/blob/master/pyproject.toml) and [`setup.cfg`](https://github.com/np_validator/np-validator/blob/master/setup.cfg).\n- Automatic codestyle with [`black`](https://github.com/psf/black), [`isort`](https://github.com/timothycrosley/isort) and [`pyupgrade`](https://github.com/asottile/pyupgrade).\n- Ready-to-use [`pre-commit`](https://pre-commit.com/) hooks with code-formatting.\n- Type checks with [`mypy`](https://mypy.readthedocs.io); docstring checks with [`darglint`](https://github.com/terrencepreilly/darglint); security checks with [`safety`](https://github.com/pyupio/safety) and [`bandit`](https://github.com/PyCQA/bandit)\n- Testing with [`pytest`](https://docs.pytest.org/en/latest/).\n- Ready-to-use [`.editorconfig`](https://github.com/np_validator/np-validator/blob/master/.editorconfig), [`.dockerignore`](https://github.com/np_validator/np-validator/blob/master/.dockerignore), and [`.gitignore`](https://github.com/np_validator/np-validator/blob/master/.gitignore). You don\'t have to worry about those things.\n\n### Deployment features\n\n- `GitHub` integration: issue and pr templates.\n- `Github Actions` with predefined [build workflow](https://github.com/np_validator/np-validator/blob/master/.github/workflows/build.yml) as the default CI/CD.\n- Everything is already set up for security checks, codestyle checks, code formatting, testing, linting, docker builds, etc with [`Makefile`](https://github.com/np_validator/np-validator/blob/master/Makefile#L89). More details in [makefile-usage](#makefile-usage).\n- [Dockerfile](https://github.com/np_validator/np-validator/blob/master/docker/Dockerfile) for your package.\n- Always up-to-date dependencies with [`@dependabot`](https://dependabot.com/). You will only [enable it](https://docs.github.com/en/github/administering-a-repository/enabling-and-disabling-version-updates#enabling-github-dependabot-version-updates).\n- Automatic drafts of new releases with [`Release Drafter`](https://github.com/marketplace/actions/release-drafter). You may see the list of labels in [`release-drafter.yml`](https://github.com/np_validator/np-validator/blob/master/.github/release-drafter.yml). Works perfectly with [Semantic Versions](https://semver.org/) specification.\n\n### Open source community features\n\n- Ready-to-use [Pull Requests templates](https://github.com/np_validator/np-validator/blob/master/.github/PULL_REQUEST_TEMPLATE.md) and several [Issue templates](https://github.com/np_validator/np-validator/tree/master/.github/ISSUE_TEMPLATE).\n- Files such as: `LICENSE`, `CONTRIBUTING.md`, `CODE_OF_CONDUCT.md`, and `SECURITY.md` are generated automatically.\n- [`Stale bot`](https://github.com/apps/stale) that closes abandoned issues after a period of inactivity. (You will only [need to setup free plan](https://github.com/marketplace/stale)). Configuration is [here](https://github.com/np_validator/np-validator/blob/master/.github/.stale.yml).\n- [Semantic Versions](https://semver.org/) specification with [`Release Drafter`](https://github.com/marketplace/actions/release-drafter).\n\n## Installation\n\n```bash\npip install -U np-validator\n```\n\nor install with `Poetry`\n\n```bash\npoetry add np-validator\n```\n\n### Makefile usage\n\n[`Makefile`](https://github.com/np_validator/np-validator/blob/master/Makefile) contains a lot of functions for faster development.\n\n<details>\n<summary>1. Download and remove Poetry</summary>\n<p>\n\nTo download and install Poetry run:\n\n```bash\nmake poetry-download\n```\n\nTo uninstall\n\n```bash\nmake poetry-remove\n```\n\n</p>\n</details>\n\n<details>\n<summary>2. Install all dependencies and pre-commit hooks</summary>\n<p>\n\nInstall requirements:\n\n```bash\nmake install\n```\n\nPre-commit hooks coulb be installed after `git init` via\n\n```bash\nmake pre-commit-install\n```\n\n</p>\n</details>\n\n<details>\n<summary>3. Codestyle</summary>\n<p>\n\nAutomatic formatting uses `pyupgrade`, `isort` and `black`.\n\n```bash\nmake codestyle\n\n# or use synonym\nmake formatting\n```\n\nCodestyle checks only, without rewriting files:\n\n```bash\nmake check-codestyle\n```\n\n> Note: `check-codestyle` uses `isort`, `black` and `darglint` library\n\nUpdate all dev libraries to the latest version using one comand\n\n```bash\nmake update-dev-deps\n```\n\n</p>\n</details>\n\n<details>\n<summary>4. Code security</summary>\n<p>\n\n```bash\nmake check-safety\n```\n\nThis command launches `Poetry` integrity checks as well as identifies security issues with `Safety` and `Bandit`.\n\n```bash\nmake check-safety\n```\n\n</p>\n</details>\n\n<details>\n<summary>5. Type checks</summary>\n<p>\n\nRun `mypy` static type checker\n\n```bash\nmake mypy\n```\n\n</p>\n</details>\n\n<details>\n<summary>6. Tests with coverage badges</summary>\n<p>\n\nRun `pytest`\n\n```bash\nmake test\n```\n\n</p>\n</details>\n\n<details>\n<summary>7. All linters</summary>\n<p>\n\nOf course there is a command to ~~rule~~ run all linters in one:\n\n```bash\nmake lint\n```\n\nthe same as:\n\n```bash\nmake test && make check-codestyle && make mypy && make check-safety\n```\n\n</p>\n</details>\n\n<details>\n<summary>8. Docker</summary>\n<p>\n\n```bash\nmake docker-build\n```\n\nwhich is equivalent to:\n\n```bash\nmake docker-build VERSION=latest\n```\n\nRemove docker image with\n\n```bash\nmake docker-remove\n```\n\nMore information [about docker](https://github.com/np_validator/np-validator/tree/master/docker).\n\n</p>\n</details>\n\n<details>\n<summary>9. Cleanup</summary>\n<p>\nDelete pycache files\n\n```bash\nmake pycache-remove\n```\n\nRemove package build\n\n```bash\nmake build-remove\n```\n\nDelete .DS_STORE files\n\n```bash\nmake dsstore-remove\n```\n\nRemove .mypycache\n\n```bash\nmake mypycache-remove\n```\n\nOr to remove all above run:\n\n```bash\nmake cleanup\n```\n\n</p>\n</details>\n\n## 📈 Releases\n\nYou can see the list of available releases on the [GitHub Releases](https://github.com/np_validator/np-validator/releases) page.\n\nWe follow [Semantic Versions](https://semver.org/) specification.\n\nWe use [`Release Drafter`](https://github.com/marketplace/actions/release-drafter). As pull requests are merged, a draft release is kept up-to-date listing the changes, ready to publish when you’re ready. With the categories option, you can categorize pull requests in release notes using labels.\n\n### List of labels and corresponding titles\n\n|               **Label**               |  **Title in Releases**  |\n| :-----------------------------------: | :---------------------: |\n|       `enhancement`, `feature`        |       🚀 Features       |\n| `bug`, `refactoring`, `bugfix`, `fix` | 🔧 Fixes & Refactoring  |\n|       `build`, `ci`, `testing`        | 📦 Build System & CI/CD |\n|              `breaking`               |   💥 Breaking Changes   |\n|            `documentation`            |    📝 Documentation     |\n|            `dependencies`             | ⬆️ Dependencies updates |\n\nYou can update it in [`release-drafter.yml`](https://github.com/np_validator/np-validator/blob/master/.github/release-drafter.yml).\n\nGitHub creates the `bug`, `enhancement`, and `documentation` labels for you. Dependabot creates the `dependencies` label. Create the remaining labels on the Issues tab of your GitHub repository, when you need them.\n\n## 🛡 License\n\n[![License](https://img.shields.io/github/license/np_validator/np-validator)](https://github.com/np_validator/np-validator/blob/master/LICENSE)\n\nThis project is licensed under the terms of the `MIT` license. See [LICENSE](https://github.com/np_validator/np-validator/blob/master/LICENSE) for more details.\n\n## 📃 Citation\n\n```bibtex\n@misc{np-validator,\n  author = {np-validator},\n  title = {Awesome `np-validator` is a Python cli/package created with https://github.com/TezRomacH/python-package-template},\n  year = {2022},\n  publisher = {GitHub},\n  journal = {GitHub repository},\n  howpublished = {\\url{https://github.com/np_validator/np-validator}}\n}\n```\n\n## Credits [![🚀 Your next Python package needs a bleeding-edge project structure.](https://img.shields.io/badge/python--package--template-%F0%9F%9A%80-brightgreen)](https://github.com/TezRomacH/python-package-template)\n\nThis project was generated with [`python-package-template`](https://github.com/TezRomacH/python-package-template)\n',
    'author': 'np-validator',
    'author_email': 'chrism@alleninstitute.org',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/np_validator/np-validator',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
