"""
    Oy Provider Modules
    ____________________
    Full reference:
    https://api-docs.oyindonesia.com/
"""
import uuid

from oy.core.provider import BaseProvider
from oy.urls import URLS

class OyProvider(BaseProvider):
    """
        Oy Provider
    """

    def _api_name_to_full_url(self, api_name):
        """ using base url that generated by base
        provider we build right url """
        return self.base_url + URLS[api_name]

    def prepare_request(self, **kwargs):
        self.request_contract.url = self._api_name_to_full_url(
            kwargs["api_name"]
        )
        self.request_contract.method = kwargs["method"]
        if "payload" in kwargs:
            self.request_contract.payload = kwargs["payload"]

    def inquiry_account(self, recipient_bank, recipient_account):
        """
            Bank Account Inquiry
            This API is mandatory to hit before remit API.
            Use this API to get beneficiary account details.

            Parameters:
                recipient_bank : string
                recipient_account: string

            Returns:
                recipient_bank: string
                recipient_account: string
                recipient_name: string
                timestamp: string
        """
        payload = {
            "api_name": "INQUIRY_ACCOUNT",
            "method": "POST",
            "payload": {
                "recipient_bank": recipient_bank,
                "recipient_account": recipient_account
            }
        }
        response = self.execute(**payload)
        # trim status code
        return response

    def disburse(self, recipient_bank, recipient_account, amount, note=None,
                 partner_trx_id=str(uuid.uuid4())):
        """
            Use this API to start disbursing money to a
            specific beneficiary account.

            Parameters:
                recipient_bank : string
                recipient_account: string
                amount: decimal
                note: string (optional)
                partner_trx_id: string (optional)

            Returns:
                recipient_bank: string
                recipient_account: string
                amount: int
                trx_id: string
                partner_trx_id: string
                timestamp: string
        """
        if note is None:
            note = "Transfer {}".format(partner_trx_id)

        payload = {
            "api_name": "DISBURSE",
            "method": "POST",
            "payload": {
                "recipient_bank": recipient_bank,
                "recipient_account": recipient_account,
                "amount": amount,
                "note": note,
                "partner_trx_id": partner_trx_id
            }
        }
        response = self.execute(**payload)
        return response

    def disburse_status(self, partner_trx_id):
        """
            To get status of a disbursement request, you can call this API.
            You may need to call this API few times
            until getting a final status (success / failed)

            Parameters:
                partner_trx_id : string

            Returns:
                bank_code: string
                name: string
                account_no: string
                trx_id: string
                trx_reference: string
                amount: decimal
                timestamp: string
                created_date: string
                last_updated_at: string
        """
        payload = {
            "api_name": "DISBURSE_STATUS",
            "method": "POST",
            "payload": {
                "partner_trx_id": partner_trx_id
            }
        }
        response = self.execute(**payload)
        return response

    def get_balance(self):
        """
            Use this API to get partner balance.

            Returns:
                balance: decimal
        """
        payload = {
            "api_name": "GET_BALANCE",
            "method": "GET",
        }
        response = self.execute(**payload)
        return response

    def generate_va(self, bank_code, amount, partner_user_id=None):
        """
            Use this API to generate VA number

            Parameters:
                bank_code : string
                amount : string
                partner_user_id : string (optional)

            Returns:
                vaNumber:  string
                amount: decimal
        """
        payload = {
            "api_name": "GENERATE_VA",
            "method": "POST",
            "payload": {
                "partner_user_id": partner_user_id,
                "bank_code": bank_code,
                "amount": amount
            }
        }
        response = self.execute(**payload)
        return response
