# -*- coding: utf-8 -*-
'''Chemical Engineering Design Library (ChEDL). Utilities for process modeling.
Copyright (C) 2021 Caleb Bell <Caleb.Andrew.Bell@gmail.com>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.'''

import pytest
from thermo.eos import *
from thermo.eos_volume import *
from fluids.constants import R
from math import log, exp, sqrt, log10
from fluids.numerics import linspace, derivative, logspace, assert_close, assert_close1d, assert_close2d, assert_close3d


def test_volume_solutions_sympy():
    Vs = volume_solutions_sympy(0.01, 1e-05, 2.5405184201558786e-05, 5.081036840311757e-05, -6.454233843151321e-10, 0.3872747173781095)
    Vs = [complex(v) for v in Vs]
    assert_close(Vs[0].real, 2.540546e-05, rtol=1e-5)
    assert_close(Vs[1].real, 4.660380256021831, rtol=1e-5)
    assert_close(Vs[2].real, 8309.802187086572, rtol=1e-5)
    
    
def validate_volume(args, solver, rtol=1e-15):
    b = args[2]
    if args in hard_solutions:
        Vs_good = hard_solutions[args]
    else:
        Vs_good = volume_solutions_mpmath_float(*args)
    Vs_test = solver(*args)
#     print(Vs_good, Vs_test, 'unfiltered')


    Vs_filtered = [i.real for i in Vs_good if (i.real ==0 or abs(i.imag/i.real) < 1E-20) and i.real > b]
    Vs_test_filtered = [i.real for i in Vs_test if (i.real ==0 or abs(i.imag/i.real) < 1E-20) and i.real > b]

#     print(Vs_filtered, Vs_test_filtered, 'filtered')

    if len(Vs_filtered) in (2, 3):
        two_roots_mpmath = True
        Vs_mpmath = min(Vs_filtered), max(Vs_filtered)
    elif len(Vs_filtered) == 0:
        raise ValueError("No volumes found in mpmath")
    else:
        two_roots_mpmath = False
        V_mpmath = Vs_filtered[0]

    if len(Vs_test_filtered) in (2, 3):
        two_roots = True
        Vs = min(Vs_test_filtered), max(Vs_test_filtered)
    elif len(Vs_test_filtered) == 1:
        V = Vs_test_filtered[0]
        two_roots = False
    else:
        raise ValueError("No volumes found")

    if two_roots != two_roots_mpmath:
        raise ValueError("mpmath and volume do not agree")
    if two_roots:
        assert_close(Vs_mpmath[0], Vs[0], rtol=rtol)
        assert_close(Vs_mpmath[1], Vs[1], rtol=rtol)
    else:
        assert_close(V_mpmath, V, rtol=rtol)

hard_parameters = [(0.01, 1e-05, 2.5405184201558786e-05, 5.081036840311757e-05, -6.454233843151321e-10, 0.3872747173781095),
    (0.0001, 154141458.17537114, 2.590839755349289e-05, 2.590839755349289e-05, 0.0, 348530.6151663297),
# (1e-50, 154141458.17537114, 2.590839755349289e-05, 2.590839755349289e-05, 0.0, 348530.6151663297),
# (1e-50, 1e-10, 2.590839755349289e-05, 2.590839755349289e-05, 0.0, 348530.6151663297),
   (0.0001, 0.01, 2.590839755349289e-05, 2.590839755349289e-05, 0.0, 348530.6151663297),
  (3274.5491628777654, 464158883.3612621, 2.581307744572419e-05, 4.4568245538197885e-05, -6.600882066641172e-10, 1.1067350681279088e-16),
(3944.2060594377003, 2.1544346900318865, 2.581307744572419e-05, 4.4568245538197885e-05, -6.600882066641172e-10, 1.9534962802746123e-22),
(0.0001, 1e-10, 2.590839755349289e-05, 2.590839755349289e-05, 0.0, 348530.6151663297),
(25.950242113997547, 4.534878508129528e-05, 1.6557828000856244e-05, 3.311565600171249e-05, -2.741616681059391e-10, 0.027002116417835224),
(25.950242113997547, 1.26185688306607e-56, 1.6557828000856244e-05, 3.311565600171249e-05, -2.741616681059391e-10, 0.027002116417835224),

(613.5907273413233, 9.326033468834164e-10, 0.00018752012786874033, 0.00037504025573748066, -3.5163798355908724e-08, 5.586919514409722),
(613.5907273413233, 3.59381366380539e-09, 0.00018752012786874033, 0.00037504025573748066, -3.5163798355908724e-08, 5.586919514409722),
(1072.267222010334, 3.59381366380539e-09, 0.00018752012786874033, 0.00037504025573748066, -3.5163798355908724e-08, 2.4032479793276456),
(138.48863713938846, 1.7886495290578157e-21, 1.982717843669203e-05, 3.965435687338406e-05, -3.9311700476042536e-10, 0.15641356549520483),
(114.97569953977451, 1.4849682622544817e-59, 1.982717843669203e-05, 3.965435687338406e-05, -3.9311700476042536e-10, 0.16714767025919666),
(4750.810162102852, 2.0565123083490847e-07, 1.844021093437284e-05, 1.844021093437284e-05, 0.0, -0.38954532427473954),
(138.48863713938846, 2.0565123083490847e-07, 2.2081238710001263e-05, 2.2081238710001263e-05, 0.0, 0.14781347198203576),
(138.48863713938846, 2.4770763559923018e-27, 1.982717843669203e-05, 3.965435687338406e-05, -3.9311700476042536e-10, 0.15636129503164162),
(166.8100537200073, 1.8738174228604786e-55, 2.649436442871357e-05, 3.6558693331591566e-05, 1.1139171395338494e-10, 0.24858665978918032),
(138.48863713938846, 5.336699231207437e-08, 2.6802130405895265e-05, 5.360426081179053e-05, -7.183541942946155e-10, 0.2788974230306992),
(15.406476869568507, 1e-20, 4.3064722069084415e-05, 0.0, 0.0,  0.23028755623367986),
(0.0001, 1e-200, 2.8293382168187418e-05,  2.8293382168187418e-5, 0.0, 0.8899971641507631),
(138.48863713938846, 3.0538555088340556e-06, 2.2081238710001263e-05, 2.2081238710001263e-05, 0.0, 0.14823101466889332),
(26.592852725396106, 3.4793909343154458e-152, 3.859918856291673e-05, 8.683016370068596e-05, -1.478300943637663e-09, 5.381957466908362),
(242.01282647944032, 599484250.31892, 4.51442411333263e-05, 4.51442411333263e-05, 0.0, 0.6466302896368097),
(69.6, 9.730954086862206e-158, 0.0005170491461435971, 0.0005170491461435971, 0.0, 164.4978116018701),
(138.48863713938846, 3.0538555088340556e-06, 2.2081238710001263e-05, 2.2081238710001263e-05, 0.0, 0.14823101466889332),
(242.01282647944032, 6.892612104351147e-21, 2.208123871000126e-05, 2.208123871000126e-05, 0.0, 0.09892412606156903),
(4714.866363457429, 0.01, 2.8986924337681703e-05, 4.923467294821763e-05, -8.306951335097421e-10, 1.810368573677903e-13),
]

hard_solutions = {}
hard_solutions[(0.01, 1e-05, 2.5405184201558786e-05, 5.081036840311757e-05, -6.454233843151321e-10, 0.3872747173781095)] = ((2.540546134155487e-05+0j), (4.660380256021552+0j), (8309.802187086572+0j))
hard_solutions[(0.0001, 154141458.17537114, 2.590839755349289e-05, 2.590839755349289e-05, 0.0, 348530.6151663297)] = ((-1.2954196079724573e-05-0.04755111651434644j), (-1.2954196079724573e-05+0.04755111651434644j), (2.590839755349609e-05+0j))
hard_solutions[(0.0001, 0.01, 2.590839755349289e-05, 2.590839755349289e-05, 0.0, 348530.6151663297)] = ((2.590839755349609e-05+0j), (0.04155935889198945-5903.648153040752j), (0.04155935889198945+5903.648153040752j))
hard_solutions[(3274.5491628777654, 464158883.3612621, 2.581307744572419e-05, 4.4568245538197885e-05, -6.600882066641172e-10, 1.1067350681279088e-16)] = ((-5.6293979197087446e-05+0j), (1.1725733658889563e-05+0j), (8.44699675366581e-05+0j))
hard_solutions[(3944.2060594377003, 2.1544346900318865, 2.581307744572419e-05, 4.4568245538197885e-05, -6.600882066641172e-10, 1.9534962802746123e-22)] = ((-5.6293979197087446e-05+0j), (1.1725733658889564e-05+0j), (15221.605020951218+0j))
hard_solutions[(0.0001, 1e-10, 2.590839755349289e-05, 2.590839755349289e-05, 0.0, 348530.6151663297)] = ((2.590839755349609e-05+0j), (4157231.3090636656-58889927.6575019j), (4157231.3090636656+58889927.6575019j))
hard_solutions[(25.950242113997547, 4.534878508129528e-05, 1.6557828000856244e-05, 3.311565600171249e-05, -2.741616681059391e-10, 0.027002116417835224)] = ((2.8142588271282462e-05+0j), (6.388926373423698e-05+0j), (4757841.198990095+0j))
hard_solutions[(25.950242113997547, 1.26185688306607e-56, 1.6557828000856244e-05, 3.311565600171249e-05, -2.741616681059391e-10, 0.027002116417835224)] = ((2.8142588271381216e-05+0j), (6.388926373258853e-05+0j), (1.7098794711536342e+58+0j))
hard_solutions[(613.5907273413233, 9.326033468834164e-10, 0.00018752012786874033, 0.00037504025573748066, -3.5163798355908724e-08, 5.586919514409722)] = ((0.00036003701200507717-0.00020140884141394387j), (0.00036003701200507717+0.00020140884141394387j), (5470361201655.266+0j))
hard_solutions[(613.5907273413233, 3.59381366380539e-09, 0.00018752012786874033, 0.00037504025573748066, -3.5163798355908724e-08, 5.586919514409722)] = ((0.0003600370120050773-0.00020140884141394387j), (0.0003600370120050773+0.00020140884141394387j), (1419571976339.7139+0j))
hard_solutions[(1072.267222010334, 3.59381366380539e-09, 0.00018752012786874033, 0.00037504025573748066, -3.5163798355908724e-08, 2.4032479793276456)] = ((-5.273828977268465e-05-0.00011226526383639805j), (-5.273828977268465e-05+0.00011226526383639805j), (2480742344508.6885+0j))
hard_solutions[(138.48863713938846, 1.7886495290578157e-21, 1.982717843669203e-05, 3.965435687338406e-05, -3.9311700476042536e-10, 0.15641356549520483)] = ((4.452906816711374e-05+0j), (5.165607971109782e-05+0j), (6.437586446244575e+23+0j))
hard_solutions[(114.97569953977451, 1.4849682622544817e-59, 1.982717843669203e-05, 3.965435687338406e-05, -3.9311700476042536e-10, 0.16714767025919666)] = ((2.8922506501396595e-05+0j), (0.00010627090030439677+0j), (6.437586446245873e+61+0j))
hard_solutions[(4750.810162102852, 2.0565123083490847e-07, 1.844021093437284e-05, 1.844021093437284e-05, 0.0, -0.38954532427473954)] = ((-3.3698500469541746e-05+0j), (5.3964908213117695e-06+0j), (192074870344.23166+0j))
hard_solutions[(138.48863713938846, 2.0565123083490847e-07, 2.2081238710001263e-05, 2.2081238710001263e-05, 0.0, 0.14781347198203576)] = ((5.314469847879485e-05-3.197447908673642e-06j), (5.314469847879485e-05+3.197447908673642e-06j), (5599084390.886886+0j))
hard_solutions[(138.48863713938846, 2.4770763559923018e-27, 1.982717843669203e-05, 3.965435687338406e-05, -3.9311700476042536e-10, 0.15636129503164162)] = ((4.4691122094334564e-05+0j), (5.14486307833789e-05+0j), (4.648458226767767e+29+0j))
hard_solutions[(166.8100537200073, 1.8738174228604786e-55, 2.649436442871357e-05, 3.6558693331591566e-05, 1.1139171395338494e-10, 0.24858665978918032)] = ((5.620521817029162e-05+0j), (8.647050715752194e-05+0j), (7.401660050048555e+57+0j))
hard_solutions[(138.48863713938846, 5.336699231207437e-08, 2.6802130405895265e-05, 5.360426081179053e-05, -7.183541942946155e-10, 0.2788974230306992)] = ((3.8448880760645455e-05+0j), (0.00015015914801409685+0j), (21576231798.8811+0j))
hard_solutions[(15.406476869568507, 1e-20, 4.3064722069084415e-05, 0.0, 0.0, 0.23028755623367986)] = ((4.4148916440236454e-05+0j), (0.001753616202719976+0j), (1.280965760094699e+22+0j))
hard_solutions[(0.0001, 1e-200, 2.8293382168187418e-05, 2.8293382168187418e-05, 0.0, 0.8899971641507631)] = ((2.8293383663890278e-05+0j), (1070.4204925508334+0j), (8.31446261815324e+196+0j))
hard_solutions[(138.48863713938846, 3.0538555088340556e-06, 2.2081238710001263e-05, 2.2081238710001263e-05, 0.0, 0.14823101466889332)] = ((5.228993310247619e-05+0j), (5.436208452435612e-05+0j), (377050778.3368337+0j))
hard_solutions[(26.592852725396106, 3.4793909343154458e-152, 3.859918856291673e-05, 8.683016370068596e-05, -1.478300943637663e-09, 5.381957466908362)] = ((3.8738294833368536e-05+0j), (0.024215584631321646+0j), (6.354712191571019e+153+0j))
hard_solutions[(242.01282647944032, 599484250.31892, 4.51442411333263e-05, 4.51442411333263e-05, 0.0, 0.6466302896368097)] = ((-2.2244638012439608e-05-2.286733817769648e-05j), (-2.2244638012439608e-05+2.286733817769648e-05j), (4.784583893342342e-05+0j))
hard_solutions[(69.6, 9.730954086862206e-158, 0.0005170491461435971, 0.0005170491461435971, 0.0, 164.4978116018701)] = ((0.0005189404277120311+0j), (0.2832246311263108+0j), (5.946863925755771e+159+0j))
hard_solutions[(138.48863713938846, 3.0538555088340556e-06, 2.2081238710001263e-05, 2.2081238710001263e-05, 0.0, 0.14823101466889332)] = ((5.228993310247619e-05+0j), (5.436208452435612e-05+0j), (377050778.3368337+0j))
hard_solutions[(242.01282647944032, 6.892612104351147e-21, 2.208123871000126e-05, 2.208123871000126e-05, 0.0, 0.09892412606156903)] = ((1.3540386918605232e-05-3.003691156915976e-05j), (1.3540386918605232e-05+3.003691156915976e-05j), (2.919367241929447e+23+0j))
hard_solutions[(4714.866363457429, 0.01, 2.8986924337681703e-05, 4.923467294821763e-05, -8.306951335097421e-10, 1.810368573677903e-13)] = ((-6.252127279461262e-05+0j), (1.328659984639961e-05+0j), (3920158.012884476+0j))


# TODO important make mpmath cache answers here

@pytest.mark.parametrize("params", hard_parameters)
@pytest.mark.parametrize("solver", [volume_solutions_halley, GCEOS.volume_solutions])
def test_hard_default_solver_volumes(solver, params):
    validate_volume(params, solver, rtol=1e-14)