"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const function_1 = require("./function");
const function_base_1 = require("./function-base");
/**
 * A Lambda that will only ever be added to a stack once.
 *
 * This construct is a way to guarantee that the lambda function will be guaranteed to be part of the stack,
 * once and only once, irrespective of how many times the construct is declared to be part of the stack.
 * This is guaranteed as long as the `uuid` property and the optional `lambdaPurpose` property stay the same
 * whenever they're declared into the stack.
 *
 * @resource AWS::Lambda::Function
 */
class SingletonFunction extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaFunction = this.ensureLambda(props);
        this.permissionsNode = this.lambdaFunction.node;
        this.functionArn = this.lambdaFunction.functionArn;
        this.functionName = this.lambdaFunction.functionName;
        this.role = this.lambdaFunction.role;
        this.grantPrincipal = this.lambdaFunction.grantPrincipal;
        this.canCreatePermissions = true; // Doesn't matter, addPermission is overriden anyway
    }
    addPermission(name, permission) {
        return this.lambdaFunction.addPermission(name, permission);
    }
    /**
     * Using node.addDependency() does not work on this method as the underlying lambda function is modeled
     * as a singleton across the stack. Use this method instead to declare dependencies.
     */
    addDependency(...up) {
        this.lambdaFunction.node.addDependency(...up);
    }
    /**
     * The SingletonFunction construct cannot be added as a dependency of another construct using
     * node.addDependency(). Use this method instead to declare this as a dependency of another construct.
     */
    dependOn(down) {
        down.node.addDependency(this.lambdaFunction);
    }
    /**
     * Returns the construct tree node that corresponds to the lambda function.
     * @internal
     */
    _functionNode() {
        return this.lambdaFunction.node;
    }
    ensureLambda(props) {
        const constructName = (props.lambdaPurpose || 'SingletonLambda') + slugify(props.uuid);
        const existing = cdk.Stack.of(this).node.tryFindChild(constructName);
        if (existing) {
            // Just assume this is true
            return existing;
        }
        return new function_1.Function(cdk.Stack.of(this), constructName, props);
    }
}
exports.SingletonFunction = SingletonFunction;
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
//# sourceMappingURL=data:application/json;base64,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