"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
var RuntimeFamily;
(function (RuntimeFamily) {
    RuntimeFamily[RuntimeFamily["NODEJS"] = 0] = "NODEJS";
    RuntimeFamily[RuntimeFamily["JAVA"] = 1] = "JAVA";
    RuntimeFamily[RuntimeFamily["PYTHON"] = 2] = "PYTHON";
    RuntimeFamily[RuntimeFamily["DOTNET_CORE"] = 3] = "DOTNET_CORE";
    RuntimeFamily[RuntimeFamily["GO"] = 4] = "GO";
    RuntimeFamily[RuntimeFamily["RUBY"] = 5] = "RUBY";
    RuntimeFamily[RuntimeFamily["OTHER"] = 6] = "OTHER";
})(RuntimeFamily = exports.RuntimeFamily || (exports.RuntimeFamily = {}));
/**
 * Lambda function runtime environment.
 *
 * If you need to use a runtime name that doesn't exist as a static member, you
 * can instantiate a `Runtime` object, e.g: `new Runtime('nodejs99.99')`.
 */
class Runtime {
    constructor(name, family, props = {}) {
        this.name = name;
        this.supportsInlineCode = !!props.supportsInlineCode;
        this.family = family;
        this.bundlingDockerImage = core_1.BundlingDockerImage.fromRegistry(`lambci/lambda:build-${name}`);
        Runtime.ALL.push(this);
    }
    toString() {
        return this.name;
    }
    runtimeEquals(other) {
        return other.name === this.name &&
            other.family === this.family &&
            other.supportsInlineCode === this.supportsInlineCode;
    }
}
exports.Runtime = Runtime;
/** A list of all known `Runtime`'s. */
Runtime.ALL = new Array();
/**
 * The NodeJS runtime (nodejs)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS = new Runtime('nodejs', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 4.3 runtime (nodejs4.3)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS_4_3 = new Runtime('nodejs4.3', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 6.10 runtime (nodejs6.10)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS_6_10 = new Runtime('nodejs6.10', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 8.10 runtime (nodejs8.10)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS_8_10 = new Runtime('nodejs8.10', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 10.x runtime (nodejs10.x)
 */
Runtime.NODEJS_10_X = new Runtime('nodejs10.x', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 12.x runtime (nodejs12.x)
 */
Runtime.NODEJS_12_X = new Runtime('nodejs12.x', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The Python 2.7 runtime (python2.7)
 */
Runtime.PYTHON_2_7 = new Runtime('python2.7', RuntimeFamily.PYTHON, { supportsInlineCode: true });
/**
 * The Python 3.6 runtime (python3.6)
 */
Runtime.PYTHON_3_6 = new Runtime('python3.6', RuntimeFamily.PYTHON, { supportsInlineCode: true });
/**
 * The Python 3.7 runtime (python3.7)
 */
Runtime.PYTHON_3_7 = new Runtime('python3.7', RuntimeFamily.PYTHON, { supportsInlineCode: true });
/**
 * The Python 3.8 runtime (python3.8)
 */
Runtime.PYTHON_3_8 = new Runtime('python3.8', RuntimeFamily.PYTHON);
/**
 * The Java 8 runtime (java8)
 */
Runtime.JAVA_8 = new Runtime('java8', RuntimeFamily.JAVA);
/**
 * The Java 11 runtime (java11)
 */
Runtime.JAVA_11 = new Runtime('java11', RuntimeFamily.JAVA);
/**
 * The .NET Core 1.0 runtime (dotnetcore1.0)
 *
 * @deprecated Use {@link DOTNET_CORE_2_1}
 */
Runtime.DOTNET_CORE_1 = new Runtime('dotnetcore1.0', RuntimeFamily.DOTNET_CORE);
/**
 * The .NET Core 2.0 runtime (dotnetcore2.0)
 *
 * @deprecated Use {@link DOTNET_CORE_2_1}
 */
Runtime.DOTNET_CORE_2 = new Runtime('dotnetcore2.0', RuntimeFamily.DOTNET_CORE);
/**
 * The .NET Core 2.1 runtime (dotnetcore2.1)
 */
Runtime.DOTNET_CORE_2_1 = new Runtime('dotnetcore2.1', RuntimeFamily.DOTNET_CORE);
/**
 * The .NET Core 3.1 runtime (dotnetcore3.1)
 */
Runtime.DOTNET_CORE_3_1 = new Runtime('dotnetcore3.1', RuntimeFamily.DOTNET_CORE);
/**
 * The Go 1.x runtime (go1.x)
 */
Runtime.GO_1_X = new Runtime('go1.x', RuntimeFamily.GO);
/**
 * The Ruby 2.5 runtime (ruby2.5)
 */
Runtime.RUBY_2_5 = new Runtime('ruby2.5', RuntimeFamily.RUBY);
/**
 * The Ruby 2.7 runtime (ruby2.7)
 */
Runtime.RUBY_2_7 = new Runtime('ruby2.7', RuntimeFamily.RUBY);
/**
 * The custom provided runtime (provided)
 */
Runtime.PROVIDED = new Runtime('provided', RuntimeFamily.OTHER);
//# sourceMappingURL=data:application/json;base64,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