# Copyright 2020 QuantumBlack Visual Analytics Limited
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND
# NONINFRINGEMENT. IN NO EVENT WILL THE LICENSOR OR OTHER CONTRIBUTORS
# BE LIABLE FOR ANY CLAIM, DAMAGES, OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF, OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#
# The QuantumBlack Visual Analytics Limited ("QuantumBlack") name and logo
# (either separately or in combination, "QuantumBlack Trademarks") are
# trademarks of QuantumBlack. The License does not grant you any right or
# license to the QuantumBlack Trademarks. You may not use the QuantumBlack
# Trademarks or any confusingly similar mark as a trademark for your product,
#     or use the QuantumBlack Trademarks in any other manner that might cause
# confusion in the marketplace, including but not limited to in advertising,
# on websites, or on software.
#
# See the License for the specific language governing permissions and
# limitations under the License.
"""``TextLocalDataSet`` loads and saves data to a local text file. The data is
accessed text data using the python open function.
"""
import copy
import os
from pathlib import Path
from typing import Any, Dict

from kedro.io.core import AbstractVersionedDataSet, Version, deprecation_warning


class TextLocalDataSet(AbstractVersionedDataSet):
    """``TextLocalDataSet`` loads and saves unstructured text files.

    Example:
    ::

        >>> from kedro.io import TextLocalDataSet
        >>>
        >>> string_to_write = "This will go in a file."
        >>>
        >>> data_set = TextLocalDataSet(filepath="docs/new_documentation.md")
        >>> data_set.save(string_to_write)
        >>> reloaded = data_set.load()
    """

    DEFAULT_LOAD_ARGS = {"mode": "r"}  # type: Dict[str, Any]
    DEFAULT_SAVE_ARGS = {"mode": "w"}  # type: Dict[str, Any]

    def __init__(
        self,
        filepath: str,
        load_args: Dict[str, Any] = None,
        save_args: Dict[str, Any] = None,
        version: Version = None,
    ) -> None:
        """Creates a new instance of ``TextFile``.

        Args:
            filepath: path to a text file.
            load_args: Load arguments should be specified in accordance with
                the built in open function. This can be found at
                https://docs.python.org/3/library/functions.html#open
            save_args: Save arguments should be specified in accordance with
                the built in open function. This can be found at
                https://docs.python.org/3/library/functions.html#open
            version: If specified, should be an instance of
                ``kedro.io.core.Version``. If its ``load`` attribute is
                None, the latest version will be loaded. If its ``save``
                attribute is None, save version will be autogenerated.
        """
        deprecation_warning(self.__class__.__name__)
        super().__init__(Path(os.path.expanduser(filepath)), version)

        # Handle default load and save arguments
        self._load_args = copy.deepcopy(self.DEFAULT_LOAD_ARGS)
        if load_args is not None:
            self._load_args.update(load_args)
        self._save_args = copy.deepcopy(self.DEFAULT_SAVE_ARGS)
        if save_args is not None:
            self._save_args.update(save_args)

    def _load(self) -> str:
        load_path = Path(self._get_load_path())
        with load_path.open(**self._load_args) as _file:
            return _file.read()

    def _save(self, data: str) -> None:
        save_path = Path(self._get_save_path())
        save_path.parent.mkdir(parents=True, exist_ok=True)
        with save_path.open(**self._save_args) as _file:
            _file.write(data)

    def _exists(self) -> bool:
        path = self._get_load_path()
        return Path(path).is_file()

    def _describe(self) -> Dict[str, Any]:
        return dict(
            filepath=self._filepath,
            load_args=self._load_args,
            save_args=self._save_args,
            version=self._version,
        )
