# Copyright 2020 QuantumBlack Visual Analytics Limited
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND
# NONINFRINGEMENT. IN NO EVENT WILL THE LICENSOR OR OTHER CONTRIBUTORS
# BE LIABLE FOR ANY CLAIM, DAMAGES, OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF, OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#
# The QuantumBlack Visual Analytics Limited ("QuantumBlack") name and logo
# (either separately or in combination, "QuantumBlack Trademarks") are
# trademarks of QuantumBlack. The License does not grant you any right or
# license to the QuantumBlack Trademarks. You may not use the QuantumBlack
# Trademarks or any confusingly similar mark as a trademark for your product,
#     or use the QuantumBlack Trademarks in any other manner that might cause
# confusion in the marketplace, including but not limited to in advertising,
# on websites, or on software.
#
# See the License for the specific language governing permissions and
# limitations under the License.

"""``JSONLocalDataSet`` encodes a given object to json and saves it to a local
file.
"""
import copy
import json
from pathlib import Path
from typing import Any, Dict

import pandas as pd

from kedro.io.core import AbstractVersionedDataSet, Version, deprecation_warning


class JSONLocalDataSet(AbstractVersionedDataSet):
    """``JSONLocalDataSet`` encodes data as json and saves it to a local file
    or reads in and decodes an existing json file. The encoding/decoding
    functionality is provided by Python's ``json`` library.

    Example:
    ::

        >>> from kedro.io import JSONLocalDataSet
        >>> my_dict = {
        >>>     'a_string': 'Hello, World!',
        >>>     'a_list': [1, 2, 3]
        >>> }
        >>> data_set = JSONLocalDataSet(filepath="test.json")
        >>> data_set.save(my_dict)
        >>> reloaded = data_set.load()
        >>> assert my_dict == reloaded

    """

    DEFAULT_LOAD_ARGS = {}  # type: Dict[str, Any]
    DEFAULT_SAVE_ARGS = {"indent": 4}  # type: Dict[str, Any]

    def __init__(
        self,
        filepath: str,
        load_args: Dict[str, Any] = None,
        save_args: Dict[str, Any] = None,
        version: Version = None,
    ) -> None:
        """Creates a new instance of ``JSONLocalDataSet`` pointing to a concrete
        filepath.

        Args:
            filepath: path to a local json file.
            load_args: Arguments passed on to ```json.load``.
                See https://docs.python.org/3/library/json.html for details.
                All defaults are preserved.
            save_args: Arguments passed on to ```json.dump``.
                See https://docs.python.org/3/library/json.html
                for details. All defaults are preserved.
            version: If specified, should be an instance of
                ``kedro.io.core.Version``. If its ``load`` attribute is
                None, the latest version will be loaded. If its ``save``
                attribute is None, save version will be autogenerated.

        """
        deprecation_warning(self.__class__.__name__)
        super().__init__(Path(filepath), version)

        # Handle default load and save arguments
        self._load_args = copy.deepcopy(self.DEFAULT_LOAD_ARGS)
        if load_args is not None:
            self._load_args.update(load_args)
        self._save_args = copy.deepcopy(self.DEFAULT_SAVE_ARGS)
        if save_args is not None:
            self._save_args.update(save_args)

    def _describe(self) -> Dict[str, Any]:
        return dict(
            filepath=self._filepath,
            load_args=self._load_args,
            save_args=self._save_args,
            version=self._version,
        )

    def _load(self) -> Any:
        load_path = Path(self._get_load_path())
        with load_path.open("r") as local_file:
            return json.load(local_file, **self._load_args)

    def _save(self, data: pd.DataFrame) -> None:
        save_path = Path(self._get_save_path())
        save_path.parent.mkdir(parents=True, exist_ok=True)
        with save_path.open("w") as local_file:
            json.dump(data, local_file, **self._save_args)

    def _exists(self) -> bool:
        path = self._get_load_path()
        return Path(path).is_file()
