import os
from typing import Dict, NamedTuple, Optional, TypedDict

import validators

from ..errors import CliError
from .config import Config


class Profile(NamedTuple):
    display_name: str
    region: str
    arn: str
    aws_saml_url: Optional[str] = None


class OrgParams(TypedDict, total=False):
    resource_env_var: Optional[str]
    ssh_user: str
    domain: str
    aws_saml_url: str
    aws_okta_params: Dict[str, str]
    saml2aws_params: Dict[str, str]
    profiles: Dict[str, Profile]


PARAMS: Dict[str, OrgParams] = {
    "launch-darkly": {
        "resource_env_var": "ENVIRONMENT",
        "ssh_user": "ubuntu",
        "domain": "launchdarkly.com",
        "aws_saml_url": (
            "https://launchdarkly.okta.com/home/amazon_aws/0oaj4aow7gPk26Fy6356/272"
        ),
        "aws_okta_params": {
            "mfa_provider": "OKTA",
            "mfa_factor_type": "push",
            "assume_role_ttl": "1h",
            "session_ttl": "15m",
        },
        "saml2aws_params": {"mfa": "PUSH", "aws_session_duration": "900"},
        "profiles": {
            "prod": Profile(
                display_name="Prod",
                region="us-east-1",
                arn="arn:aws:iam::554582317989:role/ops/SSHAdmin-production",
            ),
            "staging": Profile(
                display_name="Staging",
                region="us-east-1",
                arn="arn:aws:iam::554582317989:role/ops/SSHAdmin-staging",
            ),
            "catfood": Profile(
                display_name="Catfood",
                region="us-east-1",
                arn="arn:aws:iam::554582317989:role/ops/SSHAdmin-catamorphic",
            ),
            "prod-intuit": Profile(
                display_name="Prod: Intuit",
                region="us-west-2",
                arn="arn:aws:iam::527291094460:role/ops/SSHAdmin-production",
            ),
            "staging-intuit": Profile(
                display_name="Staging: Intuit",
                region="us-west-2",
                arn="arn:aws:iam::527291094460:role/ops/SSHAdmin-staging",
            ),
        },
    },
    "sym": {
        "resource_env_var": "ENVIRONMENT",
        "ssh_user": "ubuntu",
        "domain": "symops.io",
        "aws_saml_url": (
            "https://dev-291131.okta.com/home/amazon_aws/0oaqlmsn7GMVgAyBK4x6/272"
        ),
        "aws_okta_params": {
            "mfa_provider": "OKTA",
            "mfa_factor_type": "push",
            "assume_role_ttl": "1h",
            "session_ttl": "15m",
        },
        "saml2aws_params": {"mfa": "PUSH", "aws_session_duration": "900"},
        "profiles": {
            "test": Profile(
                display_name="Test",
                region="us-east-1",
                arn="arn:aws:iam::838419636750:role/SSMTestRole",
            ),
        },
    },
}


def get_org_params() -> OrgParams:
    return PARAMS[Config.get_org()]


def get_aws_saml_url(resource: str) -> str:
    org_params = get_org_params()
    profile = org_params["profiles"][resource]
    return profile.aws_saml_url or org_params["aws_saml_url"]


def get_aws_okta_params() -> Dict[str, str]:
    return get_org_params()["aws_okta_params"]


def get_saml2aws_params() -> Dict[str, str]:
    return get_org_params()["saml2aws_params"]


def get_profiles() -> Dict[str, Profile]:
    return get_org_params()["profiles"]


def get_domain() -> str:
    return get_org_params()["domain"]


def get_ssh_user() -> str:
    return get_org_params()["ssh_user"]


def get_resource_env_var() -> Optional[str]:
    try:
        env_var = get_org_params()["resource_env_var"]
        default = os.getenv(env_var) if env_var else None
    except KeyError:
        default = None
    return Config.instance().get("default_resource", default)


def get_profile(resource: str) -> Profile:
    return get_profiles()[resource]


def set_login_fields(org: str, email: str):
    if org not in PARAMS:
        raise CliError(f"Invalid org: {org}")

    config = Config.instance()
    config["org"] = org

    if not validators.email(email):
        raise CliError(f"Invalid email: {email}")

    if (domain := email.split("@")[-1]) != get_domain():
        raise CliError(f"Invalid domain: {domain}")

    config["email"] = email
