# encoding: utf-8

# This file is part of CycloneDX Python Lib
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
# SPDX-License-Identifier: Apache-2.0
# Copyright (c) OWASP Foundation. All Rights Reserved.

import re
import warnings
from datetime import datetime
from decimal import Decimal
from enum import Enum
from typing import List, Optional, Tuple, Union
from urllib.parse import ParseResult, urlparse

from . import OrganizationalContact, OrganizationalEntity, Tool, XsUri
from .impact_analysis import ImpactAnalysisAffectedStatus, ImpactAnalysisJustification, ImpactAnalysisResponse, \
    ImpactAnalysisState
from ..exception.model import MutuallyExclusivePropertiesException, NoPropertiesProvidedException

"""
This set of classes represents the data that is possible about known Vulnerabilities.

Prior to CycloneDX schema version 1.4, vulnerabilities were possible in XML versions ONLY of the standard through
a schema extension: https://cyclonedx.org/ext/vulnerability.

Since CycloneDX schema version 1.4, this has become part of the core schema.

.. note::
    See the CycloneDX Schema extension definition https://cyclonedx.org/docs/1.4/#type_vulnerabilitiesType
"""


class BomTargetVersionRange:
    """
    Class that represents either a version or version range and its affected status.

    `version` and `version_range` are mutually exclusive.

    .. note::
        See the CycloneDX schema: https://cyclonedx.org/docs/1.4/#type_vulnerabilityType
    """

    def __init__(self, version: Optional[str] = None, version_range: Optional[str] = None,
                 status: Optional[ImpactAnalysisAffectedStatus] = None) -> None:
        if not version and not version_range:
            raise NoPropertiesProvidedException(
                'One of version or version_range must be provided for BomTargetVersionRange - neither provided.'
            )
        if version and version_range:
            raise MutuallyExclusivePropertiesException(
                'Either version or version_range should be provided for BomTargetVersionRange - both provided.'
            )
        self.version = version
        self.range = version_range
        self.status = status

    @property
    def version(self) -> Optional[str]:
        """
        A single version of a component or service.
        """
        return self._version

    @version.setter
    def version(self, version: Optional[str]) -> None:
        self._version = version

    @property
    def range(self) -> Optional[str]:
        """
        A version range specified in Package URL Version Range syntax (vers) which is defined at
        https://github.com/package-url/purl-spec/VERSION-RANGE-SPEC.rst

        .. note::
            The VERSION-RANGE-SPEC from Package URL is not a formalised standard at the time of writing and this no
            validation of conformance with this draft standard is performed.
        """
        return self._range

    @range.setter
    def range(self, version_range: Optional[str]) -> None:
        self._range = version_range

    @property
    def status(self) -> Optional[ImpactAnalysisAffectedStatus]:
        """
        The vulnerability status for the version or range of versions.
        """""
        return self._status

    @status.setter
    def status(self, status: Optional[ImpactAnalysisAffectedStatus]) -> None:
        self._status = status


class BomTarget:
    """
    Class that represents referencing a Component or Service in a BOM.

    Aims to represent the sub-element `target` of the complex type `vulnerabilityType`.

    You can either create a `cyclonedx.model.bom.Bom` yourself programmatically, or generate a `cyclonedx.model.bom.Bom`
    from a `cyclonedx.parser.BaseParser` implementation.

    .. note::
        See the CycloneDX schema: https://cyclonedx.org/docs/1.4/#type_vulnerabilityType
    """

    def __init__(self, ref: str, versions: Optional[List[BomTargetVersionRange]] = None) -> None:
        self.ref = ref
        self.versions = versions

    @property
    def ref(self) -> str:
        """
        Reference to a component or service by the objects `bom-ref`.
        """
        return self._ref

    @ref.setter
    def ref(self, ref: str) -> None:
        self._ref = ref

    @property
    def versions(self) -> Optional[List[BomTargetVersionRange]]:
        """
        Zero or more individual versions or range of versions.
        """
        return self._versions

    @versions.setter
    def versions(self, versions: Optional[List[BomTargetVersionRange]]) -> None:
        self._versions = versions


class VulnerabilityAnalysis:
    """
    Class that models the `analysis` sub-element of the `vulnerabilityType` complex type.

    .. note::
        See the CycloneDX schema: https://cyclonedx.org/docs/1.4/#type_vulnerabilityType
    """

    def __init__(self, state: Optional[ImpactAnalysisState] = None,
                 justification: Optional[ImpactAnalysisJustification] = None,
                 responses: Optional[List[ImpactAnalysisResponse]] = None,
                 detail: Optional[str] = None) -> None:
        if not state and not justification and not responses and not detail:
            raise NoPropertiesProvidedException(
                'At least one of state, justification, responses or detail must be provided for VulnerabilityAnalysis '
                '- none supplied'
            )
        self.state = state
        self.justification = justification
        self.response = responses
        self.detail = detail

    @property
    def state(self) -> Optional[ImpactAnalysisState]:
        """
        The declared current state of an occurrence of a vulnerability, after automated or manual analysis.
        """
        return self._state

    @state.setter
    def state(self, state: Optional[ImpactAnalysisState]) -> None:
        self._state = state

    @property
    def justification(self) -> Optional[ImpactAnalysisJustification]:
        """
        The rationale of why the impact analysis state was asserted.
        """
        return self._justification

    @justification.setter
    def justification(self, justification: Optional[ImpactAnalysisJustification]) -> None:
        self._justification = justification

    @property
    def response(self) -> Optional[List[ImpactAnalysisResponse]]:
        """
        A list of responses to the vulnerability by the manufacturer, supplier, or project responsible for the
        affected component or service. More than one response is allowed. Responses are strongly encouraged for
        vulnerabilities where the analysis state is exploitable.
        """
        return self._response

    @response.setter
    def response(self, responses: Optional[List[ImpactAnalysisResponse]]) -> None:
        self._response = responses

    @property
    def detail(self) -> Optional[str]:
        """
        A detailed description of the impact including methods used during assessment. If a vulnerability is not
        exploitable, this field should include specific details on why the component or service is not impacted by this
        vulnerability.
        :return:
        """
        return self._detail

    @detail.setter
    def detail(self, detail: Optional[str]) -> None:
        self._detail = detail


class VulnerabilityAdvisory:
    """
    Class that models the `advisoryType` complex type.

    .. note::
        See the CycloneDX schema: https://cyclonedx.org/docs/1.4/#type_advisoryType
    """

    def __init__(self, url: XsUri, title: Optional[str] = None) -> None:
        self.title = title
        self.url = url

    @property
    def title(self) -> Optional[str]:
        """
        The title of this advisory.
        """
        return self._title

    @title.setter
    def title(self, title: Optional[str]) -> None:
        self._title = title

    @property
    def url(self) -> XsUri:
        """
        The url of this advisory.
        """
        return self._url

    @url.setter
    def url(self, url: XsUri) -> None:
        self._url = url


class VulnerabilitySource:
    """
    Class that models the `vulnerabilitySourceType` complex type.

    This type is used for multiple purposes in the CycloneDX schema.

    .. note::
        See the CycloneDX schema: https://cyclonedx.org/docs/1.4/#type_vulnerabilitySourceType
    """

    def __init__(self, name: Optional[str] = None, url: Optional[XsUri] = None) -> None:
        if not name and not url:
            raise NoPropertiesProvidedException(
                'Either name or url must be provided for a VulnerabilitySource - neither provided'
            )
        self.name = name
        self.url = url

    @property
    def name(self) -> Optional[str]:
        """
        Name of this Source.
        """
        return self._name

    @name.setter
    def name(self, name: Optional[str]) -> None:
        self._name = name

    @property
    def url(self) -> Optional[XsUri]:
        """
        The url of this Source.
        """
        return self._url

    @url.setter
    def url(self, url: XsUri) -> None:
        self._url = url


class VulnerabilityReference:
    """
    Class that models the nested `reference` within the `vulnerabilityType` complex type.

    Vulnerabilities may benefit from pointers to vulnerabilities that are the equivalent of the vulnerability specified.
    Often times, the same vulnerability may exist in multiple sources of vulnerability intelligence, but have different
    identifiers. These references provide a way to correlate vulnerabilities across multiple sources of vulnerability
    intelligence.

    .. note::
        See the CycloneDX schema: https://cyclonedx.org/docs/1.4/#type_vulnerabilityType
    """

    def __init__(self, id: Optional[str] = None, source: Optional[VulnerabilitySource] = None) -> None:
        if not id and not source:
            raise NoPropertiesProvidedException(
                'Either id or source must be provided for a VulnerabilityReference - neither provided'
            )
        self.id = id
        self.source = source

    @property
    def id(self) -> Optional[str]:
        """
        The identifier that uniquely identifies the vulnerability in the associated Source. For example: CVE-2021-39182.
        """
        return self._id

    @id.setter
    def id(self, id: Optional[str]) -> None:
        self._id = id

    @property
    def source(self) -> Optional[VulnerabilitySource]:
        """
        The source that published the vulnerability.
        """
        return self._source

    @source.setter
    def source(self, source: Optional[VulnerabilitySource]) -> None:
        self._source = source


class VulnerabilityScoreSource(Enum):
    """
    Enum object that defines the permissible source types for a Vulnerability's score.

    .. note::
        See the CycloneDX Schema definition: https://cyclonedx.org/docs/1.4/#type_scoreSourceType
    """
    CVSS_V2 = 'CVSSv2'
    CVSS_V3 = 'CVSSv3'
    CVSS_V3_1 = 'CVSSv31'
    OWASP = 'OWASP'  # Name change in 1.4
    OPEN_FAIR = 'Open FAIR'  # Only < 1.4
    OTHER = 'other'

    @staticmethod
    def get_from_vector(vector: str) -> 'VulnerabilityScoreSource':
        """
        Attempt to derive the correct SourceType from an attack vector.

        For example, often attack vector strings are prefixed with the scheme in question - such
        that __CVSS:3.0/AV:L/AC:L/PR:N/UI:R/S:C/C:L/I:N/A:N__ would be the vector
        __AV:L/AC:L/PR:N/UI:R/S:C/C:L/I:N/A:N__ under the __CVSS 3__ scheme.

        Returns:
            Always returns an instance of `VulnerabilityScoreSource`. `VulnerabilityScoreSource.OTHER` is
            returned if the scheme is not obvious or known to us.
        """
        if vector.startswith('CVSS:3.'):
            return VulnerabilityScoreSource.CVSS_V3
        elif vector.startswith('CVSS:2.'):
            return VulnerabilityScoreSource.CVSS_V2
        elif vector.startswith('OWASP'):
            return VulnerabilityScoreSource.OWASP
        else:
            return VulnerabilityScoreSource.OTHER

    def get_localised_vector(self, vector: str) -> str:
        """
        This method will remove any Source Scheme type from the supplied vector, returning just the vector.

        .. Note::
            Currently supports CVSS 3.x, CVSS 2.x and OWASP schemes.

        Returns:
            The vector without any scheme prefix as a `str`.
        """
        if self == VulnerabilityScoreSource.CVSS_V3 and vector.startswith('CVSS:3.'):
            return re.sub('^CVSS:3\\.\\d/?', '', vector)

        if self == VulnerabilityScoreSource.CVSS_V2 and vector.startswith('CVSS:2.'):
            return re.sub('^CVSS:2\\.\\d/?', '', vector)

        if self == VulnerabilityScoreSource.OWASP and vector.startswith('OWASP'):
            return re.sub('^OWASP/?', '', vector)

        return vector

    def get_value_pre_1_4(self) -> str:
        """
        Some of the enum values changed in 1.4 of the CycloneDX spec. This method allows us to
        backport some of the changes for pre-1.4.

        Returns:
            `str`
        """
        if self == VulnerabilityScoreSource.OWASP:
            return 'OWASP Risk'
        return self.value


class VulnerabilitySeverity(Enum):
    """
    Class that defines the permissible severities for a Vulnerability.

    .. note::
        See the CycloneDX schema: https://cyclonedx.org/docs/1.4/#type_severityType
    """
    NONE = 'none'
    INFO = 'info'  # Only >= 1.4
    LOW = 'low'
    MEDIUM = 'medium'
    HIGH = 'high'
    CRITICAL = 'critical'
    UNKNOWN = 'unknown'

    @staticmethod
    def get_from_cvss_scores(scores: Union[Tuple[float], float, None]) -> 'VulnerabilitySeverity':
        """
        Derives the Severity of a Vulnerability from it's declared CVSS scores.

        Args:
            scores: A `tuple` of CVSS scores. CVSS scoring system allows for up to three separate scores.

        Returns:
            Always returns an instance of `VulnerabilitySeverity`.
        """
        if type(scores) is float:
            scores = (scores,)

        if scores is None:
            return VulnerabilitySeverity.UNKNOWN

        max_cvss_score: float
        if isinstance(scores, tuple):
            max_cvss_score = max(scores)
        else:
            max_cvss_score = float(scores)

        if max_cvss_score >= 9.0:
            return VulnerabilitySeverity.CRITICAL
        elif max_cvss_score >= 7.0:
            return VulnerabilitySeverity.HIGH
        elif max_cvss_score >= 4.0:
            return VulnerabilitySeverity.MEDIUM
        elif max_cvss_score > 0.0:
            return VulnerabilitySeverity.LOW
        else:
            return VulnerabilitySeverity.NONE


class VulnerabilityRating:
    """
    Class that models the `ratingType` complex element CycloneDX core schema.

    This class previously modelled the `scoreType` complexe type in the schema extension used prior to schema version
    1.4 - see https://github.com/CycloneDX/specification/blob/master/schema/ext/vulnerability-1.0.xsd.

    .. note::
        See `ratingType` in https://cyclonedx.org/docs/1.4/#ratingType

    .. warning::
        As part of implementing support for CycloneDX schema version 1.4, the three score types defined in the schema
        externsion used prior to 1.4 have been deprecated. The deprecated `score_base` should loosely be equivalent to
        the new `score` in 1.4 schema. Both `score_impact` and `score_exploitability` are deprecated and removed as
        they are redundant if you have the vector (the vector allows you to calculate the scores).
    """

    def __init__(self, source: Optional[VulnerabilitySource] = None, score: Optional[Decimal] = None,
                 severity: Optional[VulnerabilitySeverity] = None,
                 method: Optional[VulnerabilityScoreSource] = None, vector: Optional[str] = None,
                 justification: Optional[str] = None,
                 # Deprecated parameters follow that are left to aid backwards compatability
                 score_base: Optional[float] = None) -> None:
        if not source and not score and not severity and not method and not vector and not justification:
            raise NoPropertiesProvidedException(
                'At least one property must be provided when creating a VulnerabilityRating - none supplied.'
            )

        self.source = source
        self.score = score
        self.severity = severity
        self.method = method
        self.vector = vector
        self.justification = justification

        if score_base:
            warnings.warn('`score_base` is deprecated - use `score`', DeprecationWarning)
            if score:
                warnings.warn('Both `score` and `score_base` supplied - the deprecated `score_base` will be discarded',
                              DeprecationWarning)
            else:
                self.score = Decimal(score_base)

        if vector and method:
            self.vector = method.get_localised_vector(vector=vector)

    @property
    def source(self) -> Optional[VulnerabilitySource]:
        """
        The source that published the vulnerability.
        """
        return self._source

    @source.setter
    def source(self, source: Optional[VulnerabilitySource]) -> None:
        self._source = source

    @property
    def score(self) -> Optional[Decimal]:
        """
        The numerical score of the rating.
        """
        return self._score

    @score.setter
    def score(self, score: Optional[Decimal]) -> None:
        self._score = score

    @property
    def severity(self) -> Optional[VulnerabilitySeverity]:
        """
        The textual representation of the severity that corresponds to the numerical score of the rating.
        """
        return self._severity

    @severity.setter
    def severity(self, severity: Optional[VulnerabilitySeverity]) -> None:
        self._severity = severity

    @property
    def method(self) -> Optional[VulnerabilityScoreSource]:
        """
        The risk scoring methodology/standard used.
        """
        return self._method

    @method.setter
    def method(self, score_source: Optional[VulnerabilityScoreSource]) -> None:
        self._method = score_source

    @property
    def vector(self) -> Optional[str]:
        """
        The textual representation of the metric values used to score the vulnerability - also known as the vector.
        """
        return self._vector

    @vector.setter
    def vector(self, vector: Optional[str]) -> None:
        self._vector = vector

    @property
    def justification(self) -> Optional[str]:
        """
        An optional reason for rating the vulnerability as it was.
        """
        return self._justification

    @justification.setter
    def justification(self, justification: Optional[str]) -> None:
        self._justification = justification


class VulnerabilityCredits:
    """
    Class that models the `credits` of `vulnerabilityType` complex type in the CycloneDX schema (version >= 1.4).

    This class also provides data support for schema versions < 1.4 where Vulnerabilites were possible through a schema
    extension (in XML only).

    .. note::
        See the CycloneDX schema: https://cyclonedx.org/docs/1.4/#type_vulnerabilityType
    """

    def __init__(self, organizations: Optional[List[OrganizationalEntity]] = None,
                 individuals: Optional[List[OrganizationalContact]] = None) -> None:
        if not organizations and not individuals:
            raise NoPropertiesProvidedException(
                'One of `organizations` or `individuals` must be populated - neither were'
            )
        self.organizations = organizations
        self.individuals = individuals

    @property
    def organizations(self) -> Optional[List[OrganizationalEntity]]:
        """
        The organizations credited with vulnerability discovery.

        Returns:
             List of `OrganizationalEntity` or `None`
        """
        return self._organizations

    @organizations.setter
    def organizations(self, organizations: List[OrganizationalEntity]) -> None:
        self._organizations = organizations

    @property
    def individuals(self) -> Optional[List[OrganizationalContact]]:
        """
        The individuals, not associated with organizations, that are credited with vulnerability discovery.

        Returns:
            List of `OrganizationalContact` or `None`
        """
        return self._individuals

    @individuals.setter
    def individuals(self, individuals: List[OrganizationalContact]) -> None:
        self._individuals = individuals


class Vulnerability:
    """
    Class that models the `vulnerabilityType` complex type in the CycloneDX schema (version >= 1.4).

    This class also provides data support for schema versions < 1.4 where Vulnerabilites were possible through a schema
    extension (in XML only).

    .. note::
        See the CycloneDX schema: https://cyclonedx.org/docs/1.4/#type_vulnerabilityType
    """

    def __init__(self, bom_ref: Optional[str] = None, id: Optional[str] = None,
                 source: Optional[VulnerabilitySource] = None,
                 references: Optional[List[VulnerabilityReference]] = None,
                 ratings: Optional[List[VulnerabilityRating]] = None, cwes: Optional[List[int]] = None,
                 description: Optional[str] = None, detail: Optional[str] = None, recommendation: Optional[str] = None,
                 advisories: Optional[List[VulnerabilityAdvisory]] = None, created: Optional[datetime] = None,
                 published: Optional[datetime] = None, updated: Optional[datetime] = None,
                 credits: Optional[VulnerabilityCredits] = None,
                 tools: Optional[List[Tool]] = None, analysis: Optional[VulnerabilityAnalysis] = None,
                 affects_targets: Optional[List[BomTarget]] = None,
                 # Deprecated Parameters kept for backwards compatibility
                 source_name: Optional[str] = None, source_url: Optional[str] = None,
                 recommendations: Optional[List[str]] = None) -> None:
        self.bom_ref = bom_ref
        self.id = id
        self.source = source
        self.references = references or []
        self.ratings = ratings or []
        self.cwes = cwes or []
        self.description = description
        self.detail = detail
        self.recommendation = recommendation
        self.advisories = advisories or []
        self.created = created
        self.published = published
        self.updated = updated
        self.credits = credits
        self.tools = tools or []
        self.analysis = analysis
        self.affects = affects_targets or []

        if source_name or source_url:
            warnings.warn('`source_name` and `source_url` are deprecated - use `source`', DeprecationWarning)
            if not source:
                self.source = VulnerabilitySource(name=source_name, url=XsUri(source_url) if source_url else None)

        if recommendations:
            warnings.warn('`recommendations` is deprecated - use `recommendation`', DeprecationWarning)
            if not recommendation:
                self.recommendation = recommendations.pop()

    @property
    def bom_ref(self) -> Optional[str]:
        """
        Get the unique reference for this Vulnerability in this BOM.

        Returns:
           `str` if set else `None`
        """
        return self._bom_ref

    @bom_ref.setter
    def bom_ref(self, bom_ref: str) -> None:
        self._bom_ref = bom_ref

    @property
    def id(self) -> Optional[str]:
        """
        The identifier that uniquely identifies the vulnerability. For example: CVE-2021-39182.
        """
        return self._id

    @id.setter
    def id(self, id: Optional[str]) -> None:
        self._id = id

    @property
    def source(self) -> Optional[VulnerabilitySource]:
        """
        The source that published the vulnerability.
        """
        return self._source

    @source.setter
    def source(self, source: VulnerabilitySource) -> None:
        self._source = source

    @property
    def references(self) -> List[VulnerabilityReference]:
        """
        Zero or more pointers to vulnerabilities that are the equivalent of the vulnerability specified. Often times,
        the same vulnerability may exist in multiple sources of vulnerability intelligence, but have different
        identifiers. References provides a way to correlate vulnerabilities across multiple sources of vulnerability
        intelligence.
        """
        return self._references

    @references.setter
    def references(self, references: List[VulnerabilityReference]) -> None:
        self._references = references

    def add_reference(self, reference: VulnerabilityReference) -> None:
        """
        Add an additional reference for this Vulnerability.

        Vulnerabilities may benefit from pointers to vulnerabilities that are the equivalent of the vulnerability
        specified. Often times, the same vulnerability may exist in multiple sources of vulnerability intelligence, but
        have different identifiers. These references provide a way to correlate vulnerabilities across multiple sources
        of vulnerability intelligence.

        Args:
            reference:
                `VulnerabilityReference` reference to add
        """
        self.references = self.references + [reference]

    @property
    def ratings(self) -> List[VulnerabilityRating]:
        """
        List of vulnerability ratings.
        """
        return self._ratings

    @ratings.setter
    def ratings(self, ratings: List[VulnerabilityRating]) -> None:
        self._ratings = ratings

    def add_rating(self, rating: VulnerabilityRating) -> None:
        """
        Add a vulnerability rating.

        Args:
            rating:
                `VulnerabilityRating`
        """
        self.ratings = self.ratings + [rating]

    @property
    def cwes(self) -> List[int]:
        """
        A list of CWE (Common Weakness Enumeration) identifiers.

        .. note::
            See https://cwe.mitre.org/
        """
        return self._cwes

    @cwes.setter
    def cwes(self, cwes: List[int]) -> None:
        self._cwes = cwes

    def add_cwe(self, cwe: int) -> None:
        """
        Add a CWE identifier.

        Args:
            cwe:
                `int` identifier for the CWE
        """
        self.cwes = self.cwes + [cwe]

    @property
    def description(self) -> Optional[str]:
        """
        A description of the vulnerability as provided by the source.
        """
        return self._description

    @description.setter
    def description(self, description: Optional[str]) -> None:
        self._description = description

    @property
    def detail(self) -> Optional[str]:
        """
        If available, an in-depth description of the vulnerability as provided by the source organization. Details
        often include examples, proof-of-concepts, and other information useful in understanding root cause.
        """
        return self._detail

    @detail.setter
    def detail(self, detail: Optional[str]) -> None:
        self._detail = detail

    @property
    def recommendation(self) -> Optional[str]:
        """
        Recommendations of how the vulnerability can be remediated or mitigated.
        """
        return self._recommendation

    @recommendation.setter
    def recommendation(self, recommendation: Optional[str]) -> None:
        self._recommendation = recommendation

    @property
    def advisories(self) -> List[VulnerabilityAdvisory]:
        """
        Advisories relating to the Vulnerability.
        """
        return self._advisories

    @advisories.setter
    def advisories(self, advisories: List[VulnerabilityAdvisory]) -> None:
        self._advisories = advisories

    def add_advisory(self, advisory: VulnerabilityAdvisory) -> None:
        """
        Add a advisory.

        Args:
            advisory:
                `VulnerabilityAdvisory`
        """
        self.advisories = self.advisories + [advisory]

    @property
    def created(self) -> Optional[datetime]:
        return self._created

    @created.setter
    def created(self, created: Optional[datetime]) -> None:
        self._created = created

    @property
    def published(self) -> Optional[datetime]:
        return self._published

    @published.setter
    def published(self, published: Optional[datetime]) -> None:
        self._published = published

    @property
    def updated(self) -> Optional[datetime]:
        return self._updated

    @updated.setter
    def updated(self, updated: Optional[datetime]) -> None:
        self._updated = updated

    @property
    def credits(self) -> Optional[VulnerabilityCredits]:
        """
        Individuals or organizations credited with the discovery of the vulnerability.
        """
        return self._credits

    @credits.setter
    def credits(self, credits: VulnerabilityCredits) -> None:
        self._credits = credits

    @property
    def tools(self) -> List[Tool]:
        """
        The tool(s) used to identify, confirm, or score the vulnerability.
        """
        return self._tools

    @tools.setter
    def tools(self, tools: List[Tool]) -> None:
        self._tools = tools

    def add_tool(self, tool: Tool) -> None:
        """
        Add a tool used to identify, confirm, or score the vulnerability.

        Args:
            tool:
                `Tool`
        """
        self.tools = self.tools + [tool]

    @property
    def analysis(self) -> Optional[VulnerabilityAnalysis]:
        """
        Analysis of the Vulnerability in your context.
        """
        return self._analysis

    @analysis.setter
    def analysis(self, analysis: Optional[VulnerabilityAnalysis]) -> None:
        self._analysis = analysis

    @property
    def affects(self) -> Optional[List[BomTarget]]:
        """
        The components or services that are affected by the vulnerability.
        """
        return self._affects

    @affects.setter
    def affects(self, affects_targets: Optional[List[BomTarget]]) -> None:
        self._affects = affects_targets

    # Methods pre-dating 1.4 that are kept for some backwards compatability - they will be removed in a future release!
    def get_source_name(self) -> Optional[str]:
        """
        Prior to Schema Version 1.4 when Vulnerabilities were supported by a schema extension, `source_name` and
        `source_url` where represented differently in the model.

        .. warning::
            Deprecated - this method will be removed in a future version.

            See `Vulnerability.source.get_name()`
        """
        warnings.warn(
            'The source of a Vulnerability is represnted differently in Schema Versions >= 1.4. '
            'Vulnerability.get_source_name() is deprecated and will be removed in a future release. '
            'Use Vulnerability.source.get_name()',
            DeprecationWarning
        )
        if self.source:
            return self.source.name
        return None

    def get_source_url(self) -> Optional[ParseResult]:
        """
        Prior to Schema Version 1.4 when Vulnerabilities were supported by a schema extension, `source_name` and
        `source_url` where represented differently in the model.

        .. warning::
            Deprecated - this method will be removed in a future version.

            See `Vulnerability.source.get_url()`
        """
        warnings.warn(
            'The source of a Vulnerability is represnted differently in Schema Versions >= 1.4. '
            'Vulnerability.get_source_name() is deprecated and will be removed in a future release. '
            'Use Vulnerability.source.get_url()',
            DeprecationWarning
        )
        if self.source and self.source.url:
            return urlparse(str(self.source.url))
        return None

    def get_recommendations(self) -> List[str]:
        """
        Prior to Schema Version 1.4 when Vulnerabilities were supported by a schema extension, multiple recommendations
        where permissible.

        .. warning::
            Deprecated - this method will be removed in a future version.

            See `Vulnerability.recommendation`
        """
        warnings.warn(
            'A Vulnerability has only a single recommendation from Schema Version >= 1.4. '
            'Vulnerability.get_recommendations() is deprecated and will be removed in a future release.',
            DeprecationWarning
        )
        return [self.recommendation] if self.recommendation else []
