"""Provisional JWST instrument and observational settings.
ZP can be found here : https://jwst-docs.stsci.edu/files/182256933/182256934/1/1669487685625/NRC_ZPs_0995pmap.txt

Sky Brightness needs to be derived from the ETC
"""
import lenstronomy.Util.util as util

__all__ = ['JWST']


NIRCAM_F200W_band_obs = {'exposure_time': 3600.,
                       'sky_brightness': 29.52, #this is derived using the ETC
                       'magnitude_zero_point': 28.00,
                        #'detector': 'NRCA1',
                       'num_exposures': 1,
                       'seeing': None,
                       'psf_type': 'PIXEL',
                       }


NIRCAM_F356W_band_obs = {'exposure_time': 3600.,
                        'sky_brightness': 28.39, #this is derived using the ETC
                        'magnitude_zero_point': 26.47,
                        #'detector': 'NRCALONG',
                        'num_exposures': 1,
                        'seeing': None,
                        'psf_type': 'PIXEL'  # note kernel_point_source (the PSF map) must be provided separately
                        }

"""
:keyword exposure_time: exposure time per image (in seconds)
:keyword sky_brightness: sky brightness (in magnitude per square arcseconds in units of electrons)
:keyword magnitude_zero_point: magnitude in which 1 count (e-) per second per arcsecond square is registered
:keyword num_exposures: number of exposures that are combined (depends on coadd_years)
:keyword seeing: Full-Width-at-Half-Maximum (FWHM) of PSF
:keyword psf_type: string, type of PSF ('GAUSSIAN' and 'PIXEL' supported)
"""


class JWST(object):
    """
    class contains JWST instrument and observation configurations
    """

    def __init__(self, band='F200W', psf_type='PIXEL', coadd_years=None):
        """

        :param band: string, 'F200W' or 'F356W' supported. Determines obs dictionary.
        :param psf_type: string, type of PSF ('GAUSSIAN', 'PIXEL' supported).
        :param coadd_years: int, number of years corresponding to num_exposures in obs dict. Currently supported: None.
        """

        if band == 'F200W':
            self.obs = NIRCAM_F200W_band_obs
            self.arm = 'short'
        elif band == 'F356W':
            self.obs = NIRCAM_F356W_band_obs
            self.arm = 'long'

        else:
            raise ValueError("band %s not supported!"% band)

        if psf_type == 'GAUSSIAN':
            self.obs['psf_type'] = 'GAUSSIAN'
        elif psf_type != 'PIXEL':
            raise ValueError("psf_type %s not supported!" % psf_type)

        if coadd_years is not None:
            raise ValueError(" %s coadd_years not supported! "
                             "You may manually adjust num_exposures in obs dict if required." % coadd_years)

        # NIRCAM camera settings
        if self.arm == 'short':
            self.camera = {'read_noise': 15.77,
                           'pixel_scale': 0.031,
                           'ccd_gain': 2.05,
                           }
        elif self.arm == 'long':
            self.camera = {'read_noise': 13.25,
                           'pixel_scale': 0.063,
                           'ccd_gain': 1.82,
                           }
        """
        :keyword read_noise: std of noise generated by read-out (in units of electrons)
        :keyword pixel_scale: scale (in arcseconds) of pixels
        :keyword ccd_gain: electrons/ADU (analog-to-digital unit).
        """

    def kwargs_single_band(self):
        """

        :return: merged kwargs from camera and obs dicts
        """
        kwargs = util.merge_dicts(self.camera, self.obs)
        return kwargs
