"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretTargetAttachment = exports.AttachmentTargetType = exports.Secret = void 0;
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const policy_1 = require("./policy");
const rotation_schedule_1 = require("./rotation-schedule");
const secretsmanager = require("./secretsmanager.generated");
/**
 * The common behavior of Secrets. Users should not use this class directly, and instead use ``Secret``.
 */
class SecretBase extends core_1.Resource {
    grantRead(grantee, versionStages) {
        // @see https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: ['secretsmanager:GetSecretValue', 'secretsmanager:DescribeSecret'],
            resourceArns: [this.arnForPolicies],
            scope: this,
        });
        if (versionStages != null && result.principalStatement) {
            result.principalStatement.addCondition('ForAnyValue:StringEquals', {
                'secretsmanager:VersionStage': versionStages,
            });
        }
        if (this.encryptionKey) {
            // @see https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantDecrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        return result;
    }
    grantWrite(grantee) {
        // See https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: ['secretsmanager:PutSecretValue', 'secretsmanager:UpdateSecret'],
            resourceArns: [this.arnForPolicies],
            scope: this,
        });
        if (this.encryptionKey) {
            // See https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantEncrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        return result;
    }
    get secretValue() {
        return this.secretValueFromJson('');
    }
    secretValueFromJson(jsonField) {
        return core_1.SecretValue.secretsManager(this.secretArn, { jsonField });
    }
    addRotationSchedule(id, options) {
        return new rotation_schedule_1.RotationSchedule(this, id, {
            secret: this,
            ...options,
        });
    }
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.ResourcePolicy(this, 'Policy', { secret: this });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    validate() {
        var _a;
        const errors = super.validate();
        errors.push(...((_a = this.policy) === null || _a === void 0 ? void 0 : _a.document.validateForResourcePolicy()) || []);
        return errors;
    }
    denyAccountRootDelete() {
        this.addToResourcePolicy(new iam.PolicyStatement({
            actions: ['secretsmanager:DeleteSecret'],
            effect: iam.Effect.DENY,
            resources: ['*'],
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
    /**
     * Provides an identifier for this secret for use in IAM policies. Typically, this is just the secret ARN.
     * However, secrets imported by name require a different format.
     */
    get arnForPolicies() { return this.secretArn; }
}
/**
 * Creates a new secret in AWS SecretsManager.
 */
class Secret extends SecretBase {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id, {
            physicalName: props.secretName,
        });
        this.autoCreatePolicy = true;
        if (props.generateSecretString &&
            (props.generateSecretString.secretStringTemplate || props.generateSecretString.generateStringKey) &&
            !(props.generateSecretString.secretStringTemplate && props.generateSecretString.generateStringKey)) {
            throw new Error('`secretStringTemplate` and `generateStringKey` must be specified together.');
        }
        const resource = new secretsmanager.CfnSecret(this, 'Resource', {
            description: props.description,
            kmsKeyId: props.encryptionKey && props.encryptionKey.keyArn,
            generateSecretString: props.generateSecretString || {},
            name: this.physicalName,
        });
        if (props.removalPolicy) {
            resource.applyRemovalPolicy(props.removalPolicy);
        }
        this.secretArn = this.getResourceArnAttribute(resource.ref, {
            service: 'secretsmanager',
            resource: 'secret',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.encryptionKey = props.encryptionKey;
        this.secretName = this.physicalName;
        // @see https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html#asm-authz
        const principal = new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, new iam.AccountPrincipal(core_1.Stack.of(this).account));
        (_a = this.encryptionKey) === null || _a === void 0 ? void 0 : _a.grantEncryptDecrypt(principal);
        (_b = this.encryptionKey) === null || _b === void 0 ? void 0 : _b.grant(principal, 'kms:CreateGrant', 'kms:DescribeKey');
    }
    static fromSecretArn(scope, id, secretArn) {
        return Secret.fromSecretAttributes(scope, id, { secretArn });
    }
    /**
     * Imports a secret by secret name; the ARN of the Secret will be set to the secret name.
     * A secret with this name must exist in the same account & region.
     */
    static fromSecretName(scope, id, secretName) {
        return new class extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = undefined;
                this.secretArn = secretName;
                this.secretName = secretName;
                this.autoCreatePolicy = false;
            }
            // Overrides the secretArn for grant* methods, where the secretArn must be in ARN format.
            // Also adds a wildcard to the resource name to support the SecretsManager-provided suffix.
            get arnForPolicies() {
                return core_1.Stack.of(this).formatArn({
                    service: 'secretsmanager',
                    resource: 'secret',
                    resourceName: this.secretName + '*',
                    sep: ':',
                });
            }
        }(scope, id);
    }
    /**
     * Import an existing secret into the Stack.
     *
     * @param scope the scope of the import.
     * @param id    the ID of the imported Secret in the construct tree.
     * @param attrs the attributes of the imported secret.
     */
    static fromSecretAttributes(scope, id, attrs) {
        class Import extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = attrs.encryptionKey;
                this.secretArn = attrs.secretArn;
                this.secretName = parseSecretName(scope, attrs.secretArn);
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a target attachment to the secret.
     *
     * @returns an AttachedSecret
     *
     * @deprecated use `attach()` instead
     */
    addTargetAttachment(id, options) {
        return new SecretTargetAttachment(this, id, {
            secret: this,
            ...options,
        });
    }
    /**
     * Attach a target to this secret
     *
     * @param target The target to attach
     * @returns An attached secret
     */
    attach(target) {
        const id = 'Attachment';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('Secret is already attached to a target.');
        }
        return new SecretTargetAttachment(this, id, {
            secret: this,
            target,
        });
    }
}
exports.Secret = Secret;
/**
 * The type of service or database that's being associated with the secret.
 */
var AttachmentTargetType;
(function (AttachmentTargetType) {
    /**
     * A database instance
     *
     * @deprecated use RDS_DB_INSTANCE instead
     */
    AttachmentTargetType["INSTANCE"] = "AWS::RDS::DBInstance";
    /**
     * A database cluster
     *
     * @deprecated use RDS_DB_CLUSTER instead
     */
    AttachmentTargetType["CLUSTER"] = "AWS::RDS::DBCluster";
    /**
     * AWS::RDS::DBInstance
     */
    AttachmentTargetType["RDS_DB_INSTANCE"] = "AWS::RDS::DBInstance";
    /**
     * AWS::RDS::DBCluster
     */
    AttachmentTargetType["RDS_DB_CLUSTER"] = "AWS::RDS::DBCluster";
    /**
     * AWS::RDS::DBProxy
     */
    AttachmentTargetType["RDS_DB_PROXY"] = "AWS::RDS::DBProxy";
    /**
     * AWS::Redshift::Cluster
     */
    AttachmentTargetType["REDSHIFT_CLUSTER"] = "AWS::Redshift::Cluster";
    /**
     * AWS::DocDB::DBInstance
     */
    AttachmentTargetType["DOCDB_DB_INSTANCE"] = "AWS::DocDB::DBInstance";
    /**
     * AWS::DocDB::DBCluster
     */
    AttachmentTargetType["DOCDB_DB_CLUSTER"] = "AWS::DocDB::DBCluster";
})(AttachmentTargetType = exports.AttachmentTargetType || (exports.AttachmentTargetType = {}));
/**
 * An attached secret.
 */
class SecretTargetAttachment extends SecretBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.autoCreatePolicy = true;
        const attachment = new secretsmanager.CfnSecretTargetAttachment(this, 'Resource', {
            secretId: props.secret.secretArn,
            targetId: props.target.asSecretAttachmentTarget().targetId,
            targetType: props.target.asSecretAttachmentTarget().targetType,
        });
        this.encryptionKey = props.secret.encryptionKey;
        this.secretName = props.secret.secretName;
        // This allows to reference the secret after attachment (dependency).
        this.secretArn = attachment.ref;
        this.secretTargetAttachmentSecretArn = attachment.ref;
    }
    static fromSecretTargetAttachmentSecretArn(scope, id, secretTargetAttachmentSecretArn) {
        class Import extends SecretBase {
            constructor() {
                super(...arguments);
                this.secretArn = secretTargetAttachmentSecretArn;
                this.secretTargetAttachmentSecretArn = secretTargetAttachmentSecretArn;
                this.secretName = parseSecretName(scope, secretTargetAttachmentSecretArn);
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
}
exports.SecretTargetAttachment = SecretTargetAttachment;
/** Parses the secret name from the ARN. */
function parseSecretName(construct, secretArn) {
    const resourceName = core_1.Stack.of(construct).parseArn(secretArn).resourceName;
    if (resourceName) {
        // Secret resource names are in the format `${secretName}-${SecretsManager suffix}`
        const secretNameFromArn = resourceName.substr(0, resourceName.lastIndexOf('-'));
        if (secretNameFromArn) {
            return secretNameFromArn;
        }
    }
    throw new Error('invalid ARN format; no secret name provided');
}
//# sourceMappingURL=data:application/json;base64,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