"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretRotation = exports.SecretRotationApplication = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const lambda = require("@aws-cdk/aws-lambda");
const serverless = require("@aws-cdk/aws-sam");
const core_1 = require("@aws-cdk/core");
/**
 * A secret rotation serverless application.
 */
class SecretRotationApplication {
    constructor(applicationId, semanticVersion, options) {
        this.applicationId = `arn:aws:serverlessrepo:us-east-1:297356227824:applications/${applicationId}`;
        this.semanticVersion = semanticVersion;
        this.isMultiUser = options && options.isMultiUser;
    }
}
exports.SecretRotationApplication = SecretRotationApplication;
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MariaDB using the single user rotation scheme
 */
SecretRotationApplication.MARIADB_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSMariaDBRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MariaDB using the multi user rotation scheme
 */
SecretRotationApplication.MARIADB_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSMariaDBRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MySQL using the single user rotation scheme
 */
SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSMySQLRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MySQL using the multi user rotation scheme
 */
SecretRotationApplication.MYSQL_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSMySQLRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS Oracle using the single user rotation scheme
 */
SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSOracleRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS Oracle using the multi user rotation scheme
 */
SecretRotationApplication.ORACLE_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSOracleRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS PostgreSQL using the single user rotation scheme
 */
SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSPostgreSQLRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS PostgreSQL using the multi user rotation scheme
 */
SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSPostgreSQLRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS SQL Server using the single user rotation scheme
 */
SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSSQLServerRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS SQL Server using the multi user rotation scheme
 */
SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSSQLServerRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for Amazon Redshift using the single user rotation scheme
 */
SecretRotationApplication.REDSHIFT_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRedshiftRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for Amazon Redshift using the multi user rotation scheme
 */
SecretRotationApplication.REDSHIFT_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRedshiftRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for MongoDB using the single user rotation scheme
 */
SecretRotationApplication.MONGODB_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerMongoDBRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for MongoDB using the multi user rotation scheme
 */
SecretRotationApplication.MONGODB_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerMongoDBRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Secret rotation for a service or database
 */
class SecretRotation extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.target.connections.defaultPort) {
            throw new Error('The `target` connections must have a default port range.');
        }
        if (props.application.isMultiUser && !props.masterSecret) {
            throw new Error('The `masterSecret` must be specified for application using the multi user scheme.');
        }
        // Max length of 64 chars, get the last 64 chars
        const uniqueId = this.node.uniqueId;
        const rotationFunctionName = uniqueId.substring(Math.max(uniqueId.length - 64, 0), uniqueId.length);
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.vpc,
        });
        props.target.connections.allowDefaultPortFrom(securityGroup);
        const parameters = {
            endpoint: `https://secretsmanager.${core_1.Stack.of(this).region}.${core_1.Stack.of(this).urlSuffix}`,
            functionName: rotationFunctionName,
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds.join(','),
            vpcSecurityGroupIds: securityGroup.securityGroupId,
        };
        if (props.excludeCharacters) {
            parameters.excludeCharacters = props.excludeCharacters;
        }
        if (props.secret.encryptionKey) {
            parameters.kmsKeyArn = props.secret.encryptionKey.keyArn;
        }
        if (props.masterSecret) {
            parameters.masterSecretArn = props.masterSecret.secretArn;
            if (props.masterSecret.encryptionKey) {
                parameters.masterSecretKmsKeyArn = props.masterSecret.encryptionKey.keyArn;
            }
        }
        const application = new serverless.CfnApplication(this, 'Resource', {
            location: props.application,
            parameters,
        });
        // This creates a CF a dependency between the rotation schedule and the
        // serverless application. This is needed because it's the application
        // that creates the Lambda permission to invoke the function.
        // See https://docs.aws.amazon.com/secretsmanager/latest/userguide/integrating_cloudformation.html
        const rotationLambda = lambda.Function.fromFunctionArn(this, 'RotationLambda', core_1.Token.asString(application.getAtt('Outputs.RotationLambdaARN')));
        props.secret.addRotationSchedule('RotationSchedule', {
            rotationLambda,
            automaticallyAfter: props.automaticallyAfter,
        });
        // Prevent secrets deletions when rotation is in place
        props.secret.denyAccountRootDelete();
        if (props.masterSecret) {
            props.masterSecret.denyAccountRootDelete();
        }
    }
}
exports.SecretRotation = SecretRotation;
//# sourceMappingURL=data:application/json;base64,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