import { Duration } from 'aws-cdk-lib/core';
import { Construct } from 'constructs';
import { AssertionsProvider } from './providers';
/**
 * Options for creating a WaiterStateMachine
 */
export interface WaiterStateMachineOptions {
    /**
     * The total time that the state machine will wait
     * for a successful response
     *
     * @default Duration.minutes(30)
     */
    readonly totalTimeout?: Duration;
    /**
     * The interval (number of seconds) to wait between attempts.
     *
     * @default Duration.seconds(5)
     */
    readonly interval?: Duration;
    /**
     * Backoff between attempts.
     *
     * This is the multiplier by which the retry interval increases
     * after each retry attempt.
     *
     * By default there is no backoff. Each retry will wait the amount of time
     * specified by `interval`.
     *
     * @default 1 (no backoff)
     */
    readonly backoffRate?: number;
}
/**
 * Props for creating a WaiterStateMachine
 */
export interface WaiterStateMachineProps extends WaiterStateMachineOptions {
}
/**
 * A very simple StateMachine construct highly customized to the provider framework.
 * This is so that this package does not need to depend on aws-stepfunctions module.
 *
 * The state machine continuously calls the isCompleteHandler, until it succeeds or times out.
 * The handler is called `maxAttempts` times with an `interval` duration and a `backoffRate` rate.
 *
 * For example with:
 * - maxAttempts = 360 (30 minutes)
 * - interval = 5
 * - backoffRate = 1 (no backoff)
 *
 * it will make the API Call every 5 seconds and fail after 360 failures.
 *
 * If the backoff rate is changed to 2 (for example), it will
 * - make the first call
 * - wait 5 seconds
 * - make the second call
 * - wait 15 seconds
 * - etc.
 */
export declare class WaiterStateMachine extends Construct {
    /**
     * The ARN of the statemachine
     */
    readonly stateMachineArn: string;
    /**
     * The IAM Role ARN of the role used by the state machine
     */
    readonly roleArn: string;
    /**
     * The AssertionsProvide that handles async requests
     */
    readonly isCompleteProvider: AssertionsProvider;
    constructor(scope: Construct, id: string, props?: WaiterStateMachineProps);
}
